/**
 * PropertyPro - Automated Recurring Payment Generation Service
 * Automatically generates monthly rent charges and payment schedules for leases
 */

import {
  ILease,
  IPayment,
  PaymentType,
  PaymentStatus,
  PaymentFrequency,
  IPaymentSchedule,
  ILateFeeConfig,
} from "@/types";
import { Payment, Lease } from "@/models";
import { paymentService } from "./payment.service";
import { paymentStatusService } from "./payment-status.service";
import mongoose from "mongoose";

export interface PaymentGenerationConfig {
  autoCreatePayments: boolean;
  autoGenerateInvoices: boolean;
  autoEmailInvoices: boolean;
  enableProration: boolean;
  gracePeriodDays: number;
  lateFeeAmount: number;
  lateFeeType: "fixed" | "percentage";
  maxLateFee?: number;
}

export interface ProrationCalculation {
  totalDays: number;
  daysInMonth: number;
  dailyRate: number;
  proratedAmount: number;
  fullMonthAmount: number;
  startDate: Date;
  endDate: Date;
}

export interface PaymentScheduleResult {
  payments: IPayment[];
  totalAmount: number;
  scheduleGenerated: boolean;
  prorationApplied: boolean;
  firstPaymentProrated: boolean;
  lastPaymentProrated: boolean;
  errors: string[];
}

export class RecurringPaymentGeneratorService {
  private defaultConfig: PaymentGenerationConfig = {
    autoCreatePayments: true,
    autoGenerateInvoices: true,
    autoEmailInvoices: false,
    enableProration: true,
    gracePeriodDays: 5,
    lateFeeAmount: 50,
    lateFeeType: "fixed",
    maxLateFee: 200,
  };

  /**
   * Calculate prorated rent for partial months
   */
  calculateProration(
    monthlyRent: number,
    startDate: Date,
    endDate: Date
  ): ProrationCalculation {
    const start = new Date(startDate);
    const end = new Date(endDate);

    // Get the last day of the month for the start date
    const lastDayOfMonth = new Date(
      start.getFullYear(),
      start.getMonth() + 1,
      0
    ).getDate();
    const daysInMonth = lastDayOfMonth;

    // Calculate days in the period
    const timeDiff = end.getTime() - start.getTime();
    const totalDays = Math.ceil(timeDiff / (1000 * 60 * 60 * 24)) + 1; // +1 to include both start and end dates

    const dailyRate = monthlyRent / daysInMonth;
    const proratedAmount = Math.round(dailyRate * totalDays * 100) / 100;

    return {
      totalDays,
      daysInMonth,
      dailyRate: Math.round(dailyRate * 100) / 100,
      proratedAmount,
      fullMonthAmount: monthlyRent,
      startDate: start,
      endDate: end,
    };
  }

  /**
   * Generate payment schedule for entire lease term
   */
  async generatePaymentSchedule(
    leaseId: string,
    config: Partial<PaymentGenerationConfig> = {}
  ): Promise<PaymentScheduleResult> {
    const finalConfig = { ...this.defaultConfig, ...config };
    const result: PaymentScheduleResult = {
      payments: [],
      totalAmount: 0,
      scheduleGenerated: false,
      prorationApplied: false,
      firstPaymentProrated: false,
      lastPaymentProrated: false,
      errors: [],
    };

    try {
      const lease = await Lease.findById(leaseId)
        .populate("tenantId")
        .populate("propertyId");

      if (!lease) {
        result.errors.push(`Lease not found: ${leaseId}`);
        return result;
      }

      const monthlyRent = lease.terms?.rentAmount || 0;
      if (monthlyRent <= 0) {
        result.errors.push("Invalid monthly rent amount");
        return result;
      }

      const leaseStart = new Date(lease.startDate);
      const leaseEnd = new Date(lease.endDate);
      const payments: IPayment[] = [];

      // Generate late fee configuration
      const lateFeeConfig: ILateFeeConfig = {
        enabled: true,
        gracePeriodDays: finalConfig.gracePeriodDays,
        feeType: finalConfig.lateFeeType,
        feeAmount: finalConfig.lateFeeAmount,
        maxFeeAmount: finalConfig.maxLateFee,
        compoundDaily: false,
      };

      // Generate payments month by month
      let currentDate = new Date(leaseStart);
      let paymentNumber = 1;

      while (currentDate <= leaseEnd) {
        const isFirstMonth = paymentNumber === 1;
        const monthStart = new Date(
          currentDate.getFullYear(),
          currentDate.getMonth(),
          leaseStart.getDate()
        );
        const monthEnd = new Date(
          currentDate.getFullYear(),
          currentDate.getMonth() + 1,
          leaseStart.getDate() - 1
        );

        // Adjust for lease boundaries
        const paymentStart = isFirstMonth ? leaseStart : monthStart;
        const paymentEnd = monthEnd > leaseEnd ? leaseEnd : monthEnd;

        let paymentAmount = monthlyRent;
        let isProrated = false;

        // Calculate proration if enabled
        if (finalConfig.enableProration) {
          const isPartialMonth =
            paymentStart.getDate() !== leaseStart.getDate() ||
            paymentEnd < monthEnd;

          if (isPartialMonth) {
            const proration = this.calculateProration(
              monthlyRent,
              paymentStart,
              paymentEnd
            );
            paymentAmount = proration.proratedAmount;
            isProrated = true;
            result.prorationApplied = true;

            if (isFirstMonth) {
              result.firstPaymentProrated = true;
            }
            if (paymentEnd >= leaseEnd) {
              result.lastPaymentProrated = true;
            }
          }
        }

        // Create payment schedule
        const schedule: IPaymentSchedule = {
          frequency: PaymentFrequency.MONTHLY,
          startDate: paymentStart,
          endDate: paymentEnd,
          dayOfMonth: leaseStart.getDate(),
          isActive: true,
          nextDueDate: monthStart,
        };

        // Create payment data
        const paymentData = {
          tenantId: lease.tenantId._id.toString(),
          propertyId: lease.propertyId._id.toString(),
          leaseId: leaseId,
          amount: paymentAmount,
          type: PaymentType.RENT,
          dueDate: monthStart,
          description: `Monthly rent payment #${paymentNumber}${
            isProrated ? " (prorated)" : ""
          }`,
          notes: isProrated
            ? `Prorated for ${paymentStart.toDateString()} to ${paymentEnd.toDateString()}`
            : "",
          schedule,
          lateFeeConfig,
        };

        if (finalConfig.autoCreatePayments) {
          const payment = await paymentService.createPayment(paymentData);
          payments.push(payment);
        }

        result.totalAmount += paymentAmount;
        paymentNumber++;

        // Move to next month
        currentDate = new Date(
          currentDate.getFullYear(),
          currentDate.getMonth() + 1,
          1
        );
      }

      result.payments = payments;
      result.scheduleGenerated = true;

      // Update payment statuses based on current date
      if (payments.length > 0) {
        const paymentIds = payments.map((p) => p._id.toString());
        await paymentStatusService.batchUpdatePaymentStatuses(paymentIds);
      }
    } catch (error) {
      result.errors.push(
        error instanceof Error ? error.message : "Unknown error occurred"
      );
    }

    return result;
  }

  /**
   * Generate security deposit payment
   */
  async generateSecurityDepositPayment(
    leaseId: string,
    config: Partial<PaymentGenerationConfig> = {}
  ): Promise<IPayment | null> {
    try {
      const lease = await Lease.findById(leaseId)
        .populate("tenantId")
        .populate("propertyId");

      if (!lease) {
        throw new Error(`Lease not found: ${leaseId}`);
      }

      const securityDeposit = lease.terms?.securityDeposit || 0;
      if (securityDeposit <= 0) {
        return null; // No security deposit required
      }

      const paymentData = {
        tenantId: lease.tenantId._id.toString(),
        propertyId: lease.propertyId._id.toString(),
        leaseId: leaseId,
        amount: securityDeposit,
        type: PaymentType.SECURITY_DEPOSIT,
        dueDate: new Date(lease.startDate),
        description: "Security deposit payment",
        notes: "Required before lease commencement",
      };

      const payment = await paymentService.createPayment(paymentData);

      // Update payment status
      await paymentStatusService.updatePaymentStatus(payment._id.toString());

      return payment;
    } catch (error) {
      console.error("Error generating security deposit payment:", error);
      return null;
    }
  }

  /**
   * Generate additional payments (pet deposit, utilities, etc.)
   */
  async generateAdditionalPayments(
    leaseId: string,
    additionalCharges: Array<{
      type: PaymentType;
      amount: number;
      description: string;
      dueDate?: Date;
      recurring?: boolean;
    }>,
    config: Partial<PaymentGenerationConfig> = {}
  ): Promise<IPayment[]> {
    const payments: IPayment[] = [];

    try {
      const lease = await Lease.findById(leaseId)
        .populate("tenantId")
        .populate("propertyId");

      if (!lease) {
        throw new Error(`Lease not found: ${leaseId}`);
      }

      for (const charge of additionalCharges) {
        const paymentData = {
          tenantId: lease.tenantId._id.toString(),
          propertyId: lease.propertyId._id.toString(),
          leaseId: leaseId,
          amount: charge.amount,
          type: charge.type,
          dueDate: charge.dueDate || new Date(lease.startDate),
          description: charge.description,
          notes: charge.recurring ? "Recurring charge" : "One-time charge",
          isRecurring: charge.recurring || false,
        };

        const payment = await paymentService.createPayment(paymentData);
        payments.push(payment);
      }

      // Update payment statuses
      if (payments.length > 0) {
        const paymentIds = payments.map((p) => p._id.toString());
        await paymentStatusService.batchUpdatePaymentStatuses(paymentIds);
      }
    } catch (error) {
      console.error("Error generating additional payments:", error);
    }

    return payments;
  }

  /**
   * Setup complete payment system for a new lease
   */
  async setupLeasePaymentSystem(
    leaseId: string,
    config: Partial<PaymentGenerationConfig> = {}
  ): Promise<{
    rentPayments: IPayment[];
    securityDeposit: IPayment | null;
    additionalPayments: IPayment[];
    totalSetupAmount: number;
    success: boolean;
    errors: string[];
  }> {
    const result = {
      rentPayments: [] as IPayment[],
      securityDeposit: null as IPayment | null,
      additionalPayments: [] as IPayment[],
      totalSetupAmount: 0,
      success: false,
      errors: [] as string[],
    };

    const session = await mongoose.startSession();

    try {
      await session.withTransaction(async () => {
        // Generate rent payment schedule
        const scheduleResult = await this.generatePaymentSchedule(
          leaseId,
          config
        );
        result.rentPayments = scheduleResult.payments;
        result.errors.push(...scheduleResult.errors);

        // Generate security deposit payment
        result.securityDeposit = await this.generateSecurityDepositPayment(
          leaseId,
          config
        );

        // Calculate total setup amount (first month + security deposit)
        const firstMonthRent =
          result.rentPayments.length > 0 ? result.rentPayments[0].amount : 0;
        const securityAmount = result.securityDeposit?.amount || 0;
        result.totalSetupAmount = firstMonthRent + securityAmount;

        result.success = result.errors.length === 0;
      });
    } catch (error) {
      result.errors.push(
        error instanceof Error ? error.message : "Transaction failed"
      );
    } finally {
      await session.endSession();
    }

    return result;
  }
}

export const recurringPaymentGeneratorService =
  new RecurringPaymentGeneratorService();
