/**
 * PropertyPro - Invoice Print Utilities
 * Shared helpers to render and print invoice HTML on the client
 */

import {
  InvoiceAddress,
  InvoiceCompanyInfo,
  InvoiceLineItemInfo,
  InvoicePartyInfo,
  InvoicePropertyInfo,
  NormalizeInvoiceOptions,
  NormalizedInvoice,
  normalizeInvoiceForPrint,
} from "@/lib/invoice/invoice-shared";
import { deriveCompanyInitials } from "@/lib/invoice/logo-utils";
import { localizationService } from "@/lib/services/localization.service";

export type PrintableInvoiceAddress = InvoiceAddress;
export type PrintableInvoiceLineItem = InvoiceLineItemInfo;
export type PrintableInvoiceParty = InvoicePartyInfo;
export type PrintableInvoiceProperty = InvoicePropertyInfo;
export type PrintableInvoice = NormalizedInvoice;

function isApproximatelyZero(value: number): boolean {
  return Math.abs(value) < 1e-6;
}

function formatCurrency(amount: number = 0): string {
  const safeAmount = Number.isFinite(amount) ? amount : 0;
  const normalizedAmount = isApproximatelyZero(safeAmount) ? 0 : safeAmount;
  try {
    return new Intl.NumberFormat("en-US", {
      style: "currency",
      currency: "USD",
    }).format(normalizedAmount);
  } catch {
    const activeCurrencyCode = localizationService.getCurrentCurrency();
    const currency = localizationService.getCurrency(activeCurrencyCode);
    const decimals = currency?.decimals ?? 2;
    const formattedAmount = normalizedAmount.toFixed(decimals);

    if (currency?.symbolPosition === "after") {
      return `${formattedAmount} ${currency.symbol}`;
    }

    return `${currency?.symbol ?? "$"}${formattedAmount}`;
  }
}

function formatAddress(address?: PrintableInvoiceAddress | string): string {
  if (!address) return "N/A";
  if (typeof address === "string") return address;
  const { street = "", city = "", state = "", zipCode = "" } = address;
  if (!street && !city && !state && !zipCode) return "N/A";
  return `${street}${street ? ", " : ""}${city}${
    city ? ", " : ""
  }${state} ${zipCode}`.trim();
}

function ensurePrintableInvoice(
  invoice: PrintableInvoice | unknown,
  options?: NormalizeInvoiceOptions
): NormalizedInvoice {
  if (
    invoice &&
    typeof invoice === "object" &&
    (invoice as NormalizedInvoice).statusMeta &&
    (invoice as NormalizedInvoice).totals
  ) {
    if (options?.companyInfo) {
      return normalizeInvoiceForPrint(invoice, options);
    }

    return invoice as NormalizedInvoice;
  }

  return normalizeInvoiceForPrint(invoice, options);
}

// Generates an invoice HTML body (no <html> wrapper) using the existing lease invoice design
export function generateInvoiceHTML(
  invoice: PrintableInvoice | unknown,
  companyInfo?: Partial<InvoiceCompanyInfo>
): string {
  const normalized = ensurePrintableInvoice(invoice, { companyInfo });

  const issue = normalized.issueDate;
  const due = normalized.dueDate;
  const tenantName = normalized.tenant
    ? `${normalized.tenant.firstName || ""} ${
        normalized.tenant.lastName || ""
      }`.trim() ||
      normalized.tenant.name ||
      "Tenant"
    : "Tenant";

  const property =
    normalized.property ||
    normalized.leaseId?.propertyId ||
    ({ name: "Property" } as InvoicePropertyInfo);

  const {
    totals: {
      subtotal,
      taxAmount,
      discountAmount,
      shippingAmount,
      adjustmentsAmount,
      total: totalAmount,
      amountPaid,
      balanceDue,
    },
  } = normalized;

  const company = normalized.companyInfo;
  const companyInitials = deriveCompanyInitials(company.name);
  const rawLogo = typeof company.logo === "string" ? company.logo.trim() : "";
  const hasLogo = rawLogo.length > 0;
  const safeLogoUrl = rawLogo.replace(/"/g, "%22");
  const safeCompanyAlt = (company.name || "Company").replace(/"/g, "&quot;");
  const logoContainerStyle = hasLogo
    ? "width:40px; height:40px; border-radius:6px; display:flex; align-items:center; justify-content:center; margin-bottom:16px; overflow:hidden; background-color:#ffffff; border:1px solid #e5e7eb;"
    : "width:40px; height:40px; background-color:#10b981; border-radius:6px; display:flex; align-items:center; justify-content:center; margin-bottom:16px; overflow:hidden;";
  const logoMarkup = hasLogo
    ? `<img src="${safeLogoUrl}" alt="${safeCompanyAlt}" style="width:100%; height:100%; object-fit:contain;" crossorigin="anonymous" />`
    : `<span style="color:white; font-weight:bold; font-size:16px; font-family: Arial, sans-serif;">${companyInitials}</span>`;

  const statusBadge = `<div style=\"background-color:${
    normalized.statusMeta.badgeBackground
  }; color:${normalized.statusMeta.badgeColor}; padding:6px 16px; border-radius:4px; font-size:14px; font-weight:600; display:inline-block; margin-bottom:8px;\">${
    normalized.statusMeta.label
  }</div>`;

  const lineItems: InvoiceLineItemInfo[] = normalized.lineItems;

  const lineItemsRows =
    lineItems.length > 0
      ? lineItems
          .map((item, idx) => {
            const qty = item.quantity ?? 1;
            const unit = item.unitPrice ?? 0;
            const totalLine = item.total ?? item.amount ?? qty * unit;
            return `
        <tr>
          <td style=\"padding:12px 16px; font-size:13px; color:#111827; border-bottom:1px solid #e5e7eb; border-right:1px solid #e5e7eb;\">${
            idx + 1
          }</td>
          <td style=\"padding:12px 16px; border-bottom:1px solid #e5e7eb; border-right:1px solid #e5e7eb;\">
            <div style=\"font-size:13px; font-weight:600; color:#111827; margin-bottom:2px;\">${
              item.description
            }</div>
            <div style=\"font-size:12px; color:#6b7280;\">${(
              item.type || ""
            ).replace("_", " ")}</div>
          </td>
          <td style=\"padding:12px 16px; text-align:right; font-size:13px; color:#111827; border-bottom:1px solid #e5e7eb; border-right:1px solid #e5e7eb;\">${qty}</td>
          <td style=\"padding:12px 16px; text-align:right; font-size:13px; color:#111827; border-bottom:1px solid #e5e7eb; border-right:1px solid #e5e7eb;\">${formatCurrency(
              Number(unit)
            )}</td>
          <td style=\"padding:12px 16px; text-align:right; font-size:13px; color:#111827; border-bottom:1px solid #e5e7eb;\">${formatCurrency(
              Number(totalLine)
            )}</td>
        </tr>`;
          })
          .join("\n")
      : `<tr>
          <td colspan=\"5\" style=\"padding:16px; text-align:center; font-size:13px; color:#6b7280; border-bottom:1px solid #e5e7eb;\">No line items available</td>
        </tr>`;

  return `
      <!-- Header -->
      <div style=\"display:flex; justify-content:space-between; align-items:flex-start; margin-bottom:40px;\">
        <div>
          <div style=\"${logoContainerStyle}\">
            ${logoMarkup}
          </div>
          <h1 style=\"font-size:20px; font-weight:bold; color:#111827; margin:0 0 4px 0; line-height:1.2;\">${
            company.name
          }</h1>
          <div style=\"color:#6b7280; font-size:13px; line-height:1.4; margin-bottom:16px;\">
            <p style=\"margin:1px 0;\">${company.address}</p>
            <p style=\"margin:1px 0;\">${company.phone}</p>
            <p style=\"margin:1px 0;\">${company.email}</p>
            ${
              company.website
                ? `<p style="margin:1px 0;">${company.website}</p>`
                : ""
            }
          </div>
          <div><span style=\"color:#6b7280; font-size:13px;\">${
            normalized.statusMeta.label
          }</span></div>
        </div>
        <div style=\"text-align:right;\">
          ${statusBadge}
          <p style=\"margin:0 0 12px 0; font-size:13px; color:#6b7280; font-weight:500;\">${
            normalized.invoiceNumber
          }</p>
          <div style=\"font-size:13px; color:#6b7280; text-align:right;\">
            <p style=\"margin:2px 0; font-weight:500;\">Issue Date: ${issue.toLocaleDateString(
              "en-US",
              { year: "numeric", month: "long", day: "numeric" }
            )}</p>
            <p style=\"margin:2px 0; font-weight:500;\">Due Date: ${due.toLocaleDateString(
              "en-US",
              { year: "numeric", month: "long", day: "numeric" }
            )}</p>
          </div>
        </div>
      </div>

      <!-- Invoice From/To -->
      <div style=\"display:grid; grid-template-columns:1fr 1fr; gap:48px; margin-bottom:32px;\">
        <div>
          <h3 style=\"font-weight:600; color:#111827; margin-bottom:8px; font-size:14px; margin:0 0 8px 0;\">Invoice from</h3>
          <div style=\"color:#374151; font-size:13px; line-height:1.4;\">
            <p style=\"font-weight:600; margin:1px 0; color:#111827;\">${
              company.name
            }</p>
            <p style=\"margin:1px 0;\">${company.address}</p>
            <p style=\"margin:1px 0;\">${company.phone}</p>
          </div>
        </div>
        <div>
          <h3 style=\"font-weight:600; color:#111827; margin-bottom:8px; font-size:14px; margin:0 0 8px 0;\">Invoice to</h3>
          <div style=\"color:#374151; font-size:13px; line-height:1.4;\">
            <p style=\"font-weight:600; margin:1px 0; color:#111827;\">${
              tenantName || "N/A"
            }</p>
            <p style=\"margin:1px 0;\">${formatAddress(property?.address)}</p>
            ${
              normalized.tenantId?.email
                ? `<p style="margin:1px 0;">${normalized.tenantId.email}</p>`
                : ""
            }
          </div>
        </div>
      </div>

      <!-- Items -->
      <div style=\"margin-bottom:32px;\">
        <table style=\"width:100%; border-collapse:collapse; border:1px solid #e5e7eb;\">
          <thead>
            <tr>
              <th style=\"padding:12px 16px; background:#f9fafb; font-weight:600; color:#111827; font-size:13px; text-align:left; border-right:1px solid #e5e7eb;\">#</th>
              <th style=\"padding:12px 16px; background:#f9fafb; font-weight:600; color:#111827; font-size:13px; text-align:left; border-right:1px solid #e5e7eb;\">Description</th>
              <th style=\"padding:12px 16px; background:#f9fafb; font-weight:600; color:#111827; font-size:13px; text-align:right; border-right:1px solid #e5e7eb;\">Qty</th>
              <th style=\"padding:12px 16px; background:#f9fafb; font-weight:600; color:#111827; font-size:13px; text-align:right; border-right:1px solid #e5e7eb;\">Unit price</th>
              <th style=\"padding:12px 16px; background:#f9fafb; font-weight:600; color:#111827; font-size:13px; text-align:right;\">Total</th>
            </tr>
          </thead>
          <tbody>
            ${lineItemsRows}
          </tbody>
        </table>
      </div>

      <!-- Totals -->
      <div style=\"display:flex; justify-content:flex-end; margin-bottom:48px;\">
        <div style=\"width:280px;\">
          <div style=\"display:flex; flex-direction:column; gap:6px;\">
            <div style=\"display:flex; justify-content:space-between; font-size:13px; padding-bottom:4px;\">
              <span style=\"color:#6b7280;\">Subtotal</span>
              <span style=\"color:#111827;\">${formatCurrency(subtotal)}</span>
            </div>
            <div style=\"display:flex; justify-content:space-between; font-size:13px; padding-bottom:4px;\">
              <span style=\"color:#6b7280;\">Shipping</span>
              <span style=\"color:#111827;\">${formatCurrency(
                shippingAmount
              )}</span>
            </div>
            <div style=\"display:flex; justify-content:space-between; font-size:13px; padding-bottom:4px;\">
              <span style=\"color:#6b7280;\">Discount</span>
              <span style=\"color:#111827;\">${formatCurrency(
                discountAmount === 0 ? 0 : -Math.abs(discountAmount)
              )}</span>
            </div>
            ${
              adjustmentsAmount !== 0
                ? `<div style="display:flex; justify-content:space-between; font-size:13px; padding-bottom:4px;">
                    <span style="color:#6b7280;">Adjustments</span>
                    <span style="color:#111827;">${formatCurrency(
                      adjustmentsAmount
                    )}</span>
                  </div>`
                : ""
            }
            <div style=\"display:flex; justify-content:space-between; font-size:13px; padding-bottom:8px;\">
              <span style=\"color:#6b7280;\">Taxes</span>
              <span style=\"color:#111827;\">${formatCurrency(taxAmount)}</span>
            </div>
            <div style=\"border-top:1px solid #e5e7eb; padding-top:12px; margin-top:4px;\">
              <div style=\"display:flex; justify-content:space-between;\">
                <span style=\"font-size:16px; font-weight:700; color:#111827;\">Total</span>
                <span style=\"font-size:16px; font-weight:700; color:#111827;\">${formatCurrency(
                  totalAmount
                )}</span>
              </div>
            </div>
            <div style=\"display:flex; justify-content:space-between; font-size:13px; padding-top:8px;\">
              <span style=\"color:#6b7280;\">Amount Paid</span>
              <span style=\"color:#111827;\">${formatCurrency(amountPaid)}</span>
            </div>
            <div style=\"display:flex; justify-content:space-between; font-size:13px; padding-top:4px;\">
              <span style=\"color:#6b7280;\">Balance Due</span>
              <span style=\"color:#111827;\">${formatCurrency(balanceDue)}</span>
            </div>
          </div>
        </div>
      </div>

      <!-- Notes -->
      <div style=\"border-top:1px solid #e5e7eb; padding-top:24px; display:flex; justify-content:space-between; align-items:flex-start;\">
        <div>
          <h3 style=\"font-weight:600; color:#111827; margin-bottom:6px; font-size:14px; margin:0 0 6px 0;\">NOTES</h3>
          <p style=\"font-size:13px; color:#6b7280; max-width:320px; line-height:1.4; margin:0;\">${
            normalized.notes
          }</p>
        </div>
        <div style=\"text-align:right;\">
          <h3 style=\"font-weight:600; color:#111827; margin-bottom:6px; font-size:14px; margin:0 0 6px 0;\">Have a question?</h3>
          <p style=\"font-size:13px; color:#6b7280; margin:0;\">${
            company.email
          }</p>
        </div>
      </div>`;
}

export function buildPrintableDocument(
  invoice: PrintableInvoice | unknown
): string {
  const normalized = ensurePrintableInvoice(invoice);
  const body = generateInvoiceHTML(normalized);
  return `<!DOCTYPE html>
  <html lang="en">
    <head>
      <meta charset="UTF-8" />
      <meta name="viewport" content="width=device-width, initial-scale=1.0" />
      <title>Invoice ${normalized.invoiceNumber}</title>
      <style>
        * { box-sizing: border-box; }
        body { font-family: Arial, sans-serif; color: #111827; margin: 24px; }
        @page { size: A4; margin: 18mm; }
        @media print {
          .no-print { display: none !important; }
          body { -webkit-print-color-adjust: exact; color-adjust: exact; print-color-adjust: exact; }
        }
      </style>
    </head>
    <body>
      ${body}
      <script>
        window.onload = function() {
          window.print();
          window.onafterprint = function() { window.close(); };
        };
      <\/script>
    </body>
  </html>`;
}

export function printInvoice(invoice: PrintableInvoice | unknown): void {
  const w = window.open("", "_blank", "width=900,height=700");
  if (!w) return;
  const html = buildPrintableDocument(invoice);
  w.document.open();
  w.document.write(html);
  w.document.close();
}

// Download the same HTML design as PDF using html2canvas + jsPDF
export async function downloadInvoiceAsPDF(
  invoice: PrintableInvoice | unknown
) {
  const normalized = ensurePrintableInvoice(invoice);
  const [{ default: html2canvas }, { default: jsPDF }] = await Promise.all([
    import("html2canvas"),
    import("jspdf"),
  ]);

  const temp = document.createElement("div");
  temp.style.position = "absolute";
  temp.style.left = "-10000px";
  temp.style.top = "0";
  temp.style.width = "794px"; // A4 width @96DPI
  temp.style.minHeight = "1123px"; // A4 height @96DPI
  temp.style.backgroundColor = "#fff";
  temp.style.padding = "32px";
  temp.innerHTML = generateInvoiceHTML(normalized);
  document.body.appendChild(temp);

  const canvas = await html2canvas(temp, {
    scale: 2,
    useCORS: true,
    allowTaint: true,
    backgroundColor: "#ffffff",
    scrollX: 0,
    scrollY: 0,
    windowWidth: 794,
    windowHeight: 1123,
  });

  const pdf = new jsPDF({
    orientation: "portrait",
    unit: "mm",
    format: "a4",
    compress: true,
  });
  const imgData = canvas.toDataURL("image/png", 1.0);
  const pdfWidth = pdf.internal.pageSize.getWidth();
  const pdfHeight = pdf.internal.pageSize.getHeight();
  pdf.addImage(imgData, "PNG", 0, 0, pdfWidth, pdfHeight, undefined, "FAST");
  const fileName = `invoice-${(
    normalized.invoiceNumber ||
    normalized._id ||
    Date.now()
  ).toString()}.pdf`;
  pdf.save(fileName);

  document.body.removeChild(temp);
}
