/**
 * PropertyPro - Document Generation Service
 * Automated document generation from templates
 */

import {
  IDocumentTemplate,
  ITemplateVariable,
  ITemplateSection,
} from "@/models/DocumentTemplate";
import DocumentTemplate from "@/models/DocumentTemplate";
import Document from "@/models/Document";
import { IDocument } from "@/models/Document";
import mongoose from "mongoose";

// Document generation data interface
export interface IDocumentGenerationData {
  templateId: string;
  variables: Record<string, any>;
  outputName: string;
  outputDescription?: string;
  tenantId?: string;
  propertyId?: string;
  leaseId?: string;
  generatedBy: string;
}

// Document generation result interface
export interface IDocumentGenerationResult {
  success: boolean;
  documentId?: string;
  fileUrl?: string;
  errors?: string[];
  warnings?: string[];
}

export class DocumentGenerationService {
  // Generate document from template
  async generateDocument(
    data: IDocumentGenerationData
  ): Promise<IDocumentGenerationResult> {
    try {
      // Fetch template
      const template = await DocumentTemplate.findById(data.templateId);
      if (!template) {
        return {
          success: false,
          errors: ["Template not found"],
        };
      }

      // Validate template is usable
      if (!template.isUsable) {
        return {
          success: false,
          errors: ["Template is not active or has been deleted"],
        };
      }

      // Validate required variables
      const validationResult = this.validateVariables(
        template.variables,
        data.variables
      );
      if (!validationResult.isValid) {
        return {
          success: false,
          errors: validationResult.errors,
        };
      }

      // Generate content
      const generatedContent = await this.processTemplate(
        template,
        data.variables
      );

      // Convert to desired output format
      const outputResult = await this.convertToOutputFormat(
        generatedContent,
        template.outputFormat,
        template.styles
      );

      if (!outputResult.success) {
        return {
          success: false,
          errors: outputResult.errors,
        };
      }

      // Create document record
      const documentData: Partial<IDocument> = {
        name: data.outputName,
        description: data.outputDescription,
        type: template.type,
        category: template.category,
        fileUrl: outputResult.fileUrl!,
        fileSize: outputResult.fileSize!,
        mimeType: this.getMimeType(template.outputFormat),
        uploadedBy: new mongoose.Types.ObjectId(data.generatedBy),
        tenantId: data.tenantId
          ? new mongoose.Types.ObjectId(data.tenantId)
          : undefined,
        propertyId: data.propertyId
          ? new mongoose.Types.ObjectId(data.propertyId)
          : undefined,
        leaseId: data.leaseId
          ? new mongoose.Types.ObjectId(data.leaseId)
          : undefined,
        status: "active",
        autoGenerated: true,
        templateId: new mongoose.Types.ObjectId(data.templateId),
        r2ObjectKey: outputResult.r2ObjectKey,
        metadata: {
          author: template.name,
          subject: data.outputName,
          language: "en",
        },
      };

      const document = new Document(documentData);
      await document.save();

      // Update template usage
      await template.incrementUsage();

      return {
        success: true,
        documentId: document._id.toString(),
        fileUrl: outputResult.fileUrl,
        warnings: validationResult.warnings,
      };
    } catch (error) {
      console.error("Document generation failed:", error);
      return {
        success: false,
        errors: ["Document generation failed: " + (error as Error).message],
      };
    }
  }

  // Validate template variables
  private validateVariables(
    templateVariables: ITemplateVariable[],
    providedVariables: Record<string, any>
  ): { isValid: boolean; errors: string[]; warnings: string[] } {
    const errors: string[] = [];
    const warnings: string[] = [];

    for (const templateVar of templateVariables) {
      const value = providedVariables[templateVar.name];

      // Check required variables
      if (
        templateVar.required &&
        (value === undefined || value === null || value === "")
      ) {
        errors.push(`Required variable '${templateVar.label}' is missing`);
        continue;
      }

      // Skip validation if value is not provided and not required
      if (value === undefined || value === null) {
        continue;
      }

      // Type validation
      const typeValidation = this.validateVariableType(templateVar, value);
      if (!typeValidation.isValid) {
        errors.push(`Variable '${templateVar.label}': ${typeValidation.error}`);
      }

      // Custom validation
      if (templateVar.validation) {
        const customValidation = this.validateCustomRules(templateVar, value);
        if (!customValidation.isValid) {
          errors.push(
            `Variable '${templateVar.label}': ${customValidation.error}`
          );
        }
      }
    }

    return {
      isValid: errors.length === 0,
      errors,
      warnings,
    };
  }

  // Validate variable type
  private validateVariableType(
    templateVar: ITemplateVariable,
    value: any
  ): { isValid: boolean; error?: string } {
    switch (templateVar.type) {
      case "text":
        if (typeof value !== "string") {
          return { isValid: false, error: "Must be a text value" };
        }
        break;
      case "number":
        if (typeof value !== "number" || isNaN(value)) {
          return { isValid: false, error: "Must be a valid number" };
        }
        break;
      case "date":
        if (!(value instanceof Date) && !Date.parse(value)) {
          return { isValid: false, error: "Must be a valid date" };
        }
        break;
      case "boolean":
        if (typeof value !== "boolean") {
          return { isValid: false, error: "Must be true or false" };
        }
        break;
      case "select":
        if (!templateVar.options?.includes(value)) {
          return {
            isValid: false,
            error: `Must be one of: ${templateVar.options?.join(", ")}`,
          };
        }
        break;
      case "multiselect":
        if (
          !Array.isArray(value) ||
          !value.every((v) => templateVar.options?.includes(v))
        ) {
          return {
            isValid: false,
            error: `Must be an array of: ${templateVar.options?.join(", ")}`,
          };
        }
        break;
    }
    return { isValid: true };
  }

  // Validate custom rules
  private validateCustomRules(
    templateVar: ITemplateVariable,
    value: any
  ): { isValid: boolean; error?: string } {
    const validation = templateVar.validation!;

    if (validation.min !== undefined && value < validation.min) {
      return {
        isValid: false,
        error: validation.message || `Must be at least ${validation.min}`,
      };
    }

    if (validation.max !== undefined && value > validation.max) {
      return {
        isValid: false,
        error: validation.message || `Must be at most ${validation.max}`,
      };
    }

    if (validation.pattern && typeof value === "string") {
      const regex = new RegExp(validation.pattern);
      if (!regex.test(value)) {
        return {
          isValid: false,
          error: validation.message || "Invalid format",
        };
      }
    }

    return { isValid: true };
  }

  // Process template content with variables
  private async processTemplate(
    template: IDocumentTemplate,
    variables: Record<string, any>
  ): Promise<string> {
    let content = template.content;

    // Process sections
    if (template.sections && template.sections.length > 0) {
      const processedSections = await this.processSections(
        template.sections,
        variables
      );
      content = processedSections.join("\n\n");
    }

    // Replace variables in content
    content = this.replaceVariables(content, variables);

    return content;
  }

  // Process template sections
  private async processSections(
    sections: ITemplateSection[],
    variables: Record<string, any>
  ): Promise<string[]> {
    const processedSections: string[] = [];

    // Sort sections by order
    const sortedSections = sections.sort((a, b) => a.order - b.order);

    for (const section of sortedSections) {
      // Check conditional logic
      if (section.conditional) {
        const conditionMet = this.evaluateCondition(
          section.conditional,
          variables
        );
        if (!conditionMet) {
          continue;
        }
      }

      let sectionContent = section.content;

      // Handle repeatable sections
      if (section.repeatable && section.repeatSource) {
        const repeatData = variables[section.repeatSource];
        if (Array.isArray(repeatData)) {
          const repeatedContent: string[] = [];
          for (const item of repeatData) {
            const itemContent = this.replaceVariables(sectionContent, {
              ...variables,
              ...item,
            });
            repeatedContent.push(itemContent);
          }
          sectionContent = repeatedContent.join("\n");
        }
      } else {
        sectionContent = this.replaceVariables(sectionContent, variables);
      }

      processedSections.push(sectionContent);
    }

    return processedSections;
  }

  // Evaluate conditional logic
  private evaluateCondition(
    condition: { variable: string; operator: string; value: any },
    variables: Record<string, any>
  ): boolean {
    const variableValue = variables[condition.variable];

    switch (condition.operator) {
      case "equals":
        return variableValue === condition.value;
      case "not_equals":
        return variableValue !== condition.value;
      case "contains":
        return String(variableValue).includes(String(condition.value));
      case "greater_than":
        return Number(variableValue) > Number(condition.value);
      case "less_than":
        return Number(variableValue) < Number(condition.value);
      default:
        return false;
    }
  }

  // Replace variables in content
  private replaceVariables(
    content: string,
    variables: Record<string, any>
  ): string {
    return content.replace(/\{\{(\w+)\}\}/g, (match, variableName) => {
      const value = variables[variableName];
      if (value === undefined || value === null) {
        return match; // Keep placeholder if variable not found
      }

      // Format dates
      if (value instanceof Date) {
        return value.toLocaleDateString();
      }

      // Format arrays
      if (Array.isArray(value)) {
        return value.join(", ");
      }

      return String(value);
    });
  }

  // Convert content to output format
  private async convertToOutputFormat(
    content: string,
    format: string,
    styles: any
  ): Promise<{
    success: boolean;
    fileUrl?: string;
    fileSize?: number;
    r2ObjectKey?: string;
    errors?: string[];
  }> {
    try {
      // This is a simplified implementation
      // In a real application, you would use libraries like:
      // - puppeteer for PDF generation
      // - docx for Word document generation
      // - html-pdf for PDF from HTML

      const mockFileUrl = `https://example.com/generated-document.${format}`;
      const mockFileSize = content.length * 2; // Rough estimate
      const mockR2ObjectKey = `generated_${Date.now()}`;

      return {
        success: true,
        fileUrl: mockFileUrl,
        fileSize: mockFileSize,
        r2ObjectKey: mockR2ObjectKey,
      };
    } catch (error) {
      return {
        success: false,
        errors: ["Failed to convert document to " + format],
      };
    }
  }

  // Get MIME type for output format
  private getMimeType(format: string): string {
    switch (format) {
      case "pdf":
        return "application/pdf";
      case "docx":
        return "application/vnd.openxmlformats-officedocument.wordprocessingml.document";
      case "html":
        return "text/html";
      default:
        return "application/octet-stream";
    }
  }
}

// Create singleton instance
export const documentGenerationService = new DocumentGenerationService();
