/**
 * PropertyPro - Sidebar Counts Hook
 * Custom hook for fetching and managing sidebar navigation counts
 */

import { useState, useEffect, useCallback } from "react";
import { useSession } from "next-auth/react";

interface SidebarCounts {
  applications: number;
  expiringLeases: number;
  emergencyMaintenance: number;
  overduePayments: number;
}

interface UseSidebarCountsReturn {
  counts: SidebarCounts;
  loading: boolean;
  error: string | null;
  refetch: () => Promise<void>;
}

const defaultCounts: SidebarCounts = {
  applications: 0,
  expiringLeases: 0,
  emergencyMaintenance: 0,
  overduePayments: 0,
};

export function useSidebarCounts(
  options: {
    refreshInterval?: number; // in milliseconds
    enabled?: boolean;
  } = {}
): UseSidebarCountsReturn {
  const { refreshInterval = 30000, enabled = true } = options; // Default 30 seconds
  const { data: session } = useSession();

  const [counts, setCounts] = useState<SidebarCounts>(defaultCounts);
  const [loading, setLoading] = useState(false);
  const [error, setError] = useState<string | null>(null);

  const fetchCounts = useCallback(async () => {
    if (!session?.user || !enabled) {
      return;
    }

    try {
      setLoading(true);
      setError(null);

      const response = await fetch("/api/sidebar/counts", {
        method: "GET",
        headers: {
          "Content-Type": "application/json",
        },
      });

      if (!response.ok) {
        throw new Error(
          `Failed to fetch sidebar counts: ${response.statusText}`
        );
      }

      const data = await response.json();

      if (data.success && data.data) {
        setCounts(data.data);
      } else {
        throw new Error(data.message || "Failed to fetch sidebar counts");
      }
    } catch (err) {
      console.error("Error fetching sidebar counts:", err);
      setError(err instanceof Error ? err.message : "Unknown error occurred");
      // Keep previous counts on error
    } finally {
      setLoading(false);
    }
  }, [session?.user, enabled]);

  // Initial fetch
  useEffect(() => {
    fetchCounts();
  }, [fetchCounts]);

  // Set up refresh interval
  useEffect(() => {
    if (!enabled || !refreshInterval) {
      return;
    }

    const interval = setInterval(fetchCounts, refreshInterval);
    return () => clearInterval(interval);
  }, [fetchCounts, refreshInterval, enabled]);

  // Listen for custom events to trigger refresh
  useEffect(() => {
    const handleRefresh = () => {
      fetchCounts();
    };

    // Listen for various events that might affect counts
    const events = [
      "sidebar-counts-refresh",
      "application-submitted",
      "lease-created",
      "lease-updated",
      "maintenance-created",
      "maintenance-updated",
      "payment-created",
      "payment-updated",
    ];

    events.forEach((event) => {
      window.addEventListener(event, handleRefresh);
    });

    return () => {
      events.forEach((event) => {
        window.removeEventListener(event, handleRefresh);
      });
    };
  }, [fetchCounts]);

  return {
    counts,
    loading,
    error,
    refetch: fetchCounts,
  };
}

// Re-export utility functions for convenience
export {
  refreshSidebarCounts,
  refreshApplicationCounts,
  refreshLeaseCounts,
  refreshMaintenanceCounts,
  refreshPaymentCounts,
  debouncedRefreshSidebarCounts,
  useSidebarRefreshTriggers,
} from "@/lib/sidebar-utils";
