/**
 * PropertyPro - User Activity Log Component
 * Display user activity and audit trail
 */

"use client";

import React, { useState, useEffect } from "react";
import { format } from "date-fns";
import {
  Card,
  CardContent,
  CardDescription,
  CardHeader,
  CardTitle,
} from "@/components/ui/card";
import { Badge } from "@/components/ui/badge";
import { Button } from "@/components/ui/button";
import { Avatar, AvatarFallback, AvatarImage } from "@/components/ui/avatar";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select";
import {
  Activity,
  User,
  Shield,
  LogIn,
  LogOut,
  Edit,
  Plus,
  Trash2,
  RefreshCw,
  Filter,
  Calendar,
  Clock,
} from "lucide-react";
import { UserRole } from "@/types";

interface ActivityEntry {
  id: string;
  userId: string;
  userName: string;
  userAvatar?: string;
  action: ActivityAction;
  target?: string;
  targetId?: string;
  details?: Record<string, any>;
  ipAddress?: string;
  userAgent?: string;
  timestamp: Date;
}

type ActivityAction =
  | "user_created"
  | "user_updated"
  | "user_deleted"
  | "user_activated"
  | "user_deactivated"
  | "role_changed"
  | "login"
  | "logout"
  | "password_changed"
  | "profile_updated"
  | "avatar_uploaded"
  | "bulk_operation";

interface UserActivityLogProps {
  userId?: string;
  limit?: number;
  showFilters?: boolean;
  className?: string;
}

const activityConfig: Record<
  ActivityAction,
  {
    label: string;
    icon: React.ReactNode;
    color: string;
    description: string;
  }
> = {
  user_created: {
    label: "User Created",
    icon: <Plus className="h-4 w-4" />,
    color: "bg-green-100 text-green-800 border-green-200",
    description: "New user account created",
  },
  user_updated: {
    label: "User Updated",
    icon: <Edit className="h-4 w-4" />,
    color: "bg-blue-100 text-blue-800 border-blue-200",
    description: "User information updated",
  },
  user_deleted: {
    label: "User Deleted",
    icon: <Trash2 className="h-4 w-4" />,
    color: "bg-red-100 text-red-800 border-red-200",
    description: "User account deleted",
  },
  user_activated: {
    label: "User Activated",
    icon: <User className="h-4 w-4" />,
    color: "bg-green-100 text-green-800 border-green-200",
    description: "User account activated",
  },
  user_deactivated: {
    label: "User Deactivated",
    icon: <User className="h-4 w-4" />,
    color: "bg-orange-100 text-orange-800 border-orange-200",
    description: "User account deactivated",
  },
  role_changed: {
    label: "Role Changed",
    icon: <Shield className="h-4 w-4" />,
    color: "bg-purple-100 text-purple-800 border-purple-200",
    description: "User role modified",
  },
  login: {
    label: "Login",
    icon: <LogIn className="h-4 w-4" />,
    color: "bg-blue-100 text-blue-800 border-blue-200",
    description: "User logged in",
  },
  logout: {
    label: "Logout",
    icon: <LogOut className="h-4 w-4" />,
    color: "bg-gray-100 text-gray-800 border-gray-200",
    description: "User logged out",
  },
  password_changed: {
    label: "Password Changed",
    icon: <Shield className="h-4 w-4" />,
    color: "bg-yellow-100 text-yellow-800 border-yellow-200",
    description: "Password updated",
  },
  profile_updated: {
    label: "Profile Updated",
    icon: <Edit className="h-4 w-4" />,
    color: "bg-blue-100 text-blue-800 border-blue-200",
    description: "Profile information updated",
  },
  avatar_uploaded: {
    label: "Avatar Uploaded",
    icon: <User className="h-4 w-4" />,
    color: "bg-green-100 text-green-800 border-green-200",
    description: "Profile picture updated",
  },
  bulk_operation: {
    label: "Bulk Operation",
    icon: <Activity className="h-4 w-4" />,
    color: "bg-purple-100 text-purple-800 border-purple-200",
    description: "Bulk operation performed",
  },
};

export function UserActivityLog({
  userId,
  limit = 50,
  showFilters = true,
  className,
}: UserActivityLogProps) {
  const [activities, setActivities] = useState<ActivityEntry[]>([]);
  const [isLoading, setIsLoading] = useState(true);
  const [selectedAction, setSelectedAction] = useState<ActivityAction | "all">(
    "all"
  );
  const [dateRange, setDateRange] = useState<string>("7d");

  // Mock data for demonstration
  const mockActivities: ActivityEntry[] = [
    {
      id: "1",
      userId: "user1",
      userName: "John Doe",
      action: "login",
      timestamp: new Date(Date.now() - 1000 * 60 * 30), // 30 minutes ago
      ipAddress: "192.168.1.100",
    },
    {
      id: "2",
      userId: "admin1",
      userName: "Admin User",
      action: "user_created",
      target: "Jane Smith",
      targetId: "user2",
      timestamp: new Date(Date.now() - 1000 * 60 * 60 * 2), // 2 hours ago
      details: { role: UserRole.TENANT },
    },
    {
      id: "3",
      userId: "user1",
      userName: "John Doe",
      action: "profile_updated",
      timestamp: new Date(Date.now() - 1000 * 60 * 60 * 4), // 4 hours ago
      details: { fields: ["phone", "avatar"] },
    },
    {
      id: "4",
      userId: "admin1",
      userName: "Admin User",
      action: "role_changed",
      target: "Mike Johnson",
      targetId: "user3",
      timestamp: new Date(Date.now() - 1000 * 60 * 60 * 24), // 1 day ago
      details: { oldRole: UserRole.TENANT, newRole: UserRole.MANAGER },
    },
    {
      id: "5",
      userId: "admin1",
      userName: "Admin User",
      action: "bulk_operation",
      timestamp: new Date(Date.now() - 1000 * 60 * 60 * 24 * 2), // 2 days ago
      details: { operation: "activate", userCount: 5 },
    },
  ];

  useEffect(() => {
    // Simulate API call
    const fetchActivities = async () => {
      setIsLoading(true);
      // In a real app, this would be an API call
      await new Promise((resolve) => setTimeout(resolve, 1000));

      let filteredActivities = mockActivities;

      if (userId) {
        filteredActivities = filteredActivities.filter(
          (activity) =>
            activity.userId === userId || activity.targetId === userId
        );
      }

      if (selectedAction !== "all") {
        filteredActivities = filteredActivities.filter(
          (activity) => activity.action === selectedAction
        );
      }

      setActivities(filteredActivities.slice(0, limit));
      setIsLoading(false);
    };

    fetchActivities();
  }, [userId, selectedAction, dateRange, limit]);

  const formatActivityDetails = (activity: ActivityEntry) => {
    const config = activityConfig[activity.action];
    let details = config.description;

    if (activity.target) {
      details += ` for ${activity.target}`;
    }

    if (activity.details) {
      switch (activity.action) {
        case "role_changed":
          details += ` from ${activity.details.oldRole?.replace(
            "_",
            " "
          )} to ${activity.details.newRole?.replace("_", " ")}`;
          break;
        case "bulk_operation":
          details += ` (${activity.details.operation} ${activity.details.userCount} users)`;
          break;
        case "profile_updated":
          details += ` (${activity.details.fields?.join(", ")})`;
          break;
      }
    }

    return details;
  };

  return (
    <Card className={className}>
      <CardHeader>
        <CardTitle className="flex items-center gap-2">
          <Activity className="h-5 w-5" />
          Activity Log
        </CardTitle>
        <CardDescription>
          {userId ? "User activity history" : "Recent system activity"}
        </CardDescription>
      </CardHeader>
      <CardContent>
        {showFilters && (
          <div className="flex flex-col sm:flex-row gap-4 mb-6">
            <Select
              value={selectedAction}
              onValueChange={(value) =>
                setSelectedAction(value as ActivityAction | "all")
              }
            >
              <SelectTrigger className="w-full sm:w-48">
                <SelectValue placeholder="Filter by action" />
              </SelectTrigger>
              <SelectContent>
                <SelectItem value="all">All Actions</SelectItem>
                {Object.entries(activityConfig).map(([action, config]) => (
                  <SelectItem key={action} value={action}>
                    <div className="flex items-center gap-2">
                      {config.icon}
                      {config.label}
                    </div>
                  </SelectItem>
                ))}
              </SelectContent>
            </Select>

            <Select value={dateRange} onValueChange={setDateRange}>
              <SelectTrigger className="w-full sm:w-32">
                <SelectValue />
              </SelectTrigger>
              <SelectContent>
                <SelectItem value="1d">Today</SelectItem>
                <SelectItem value="7d">7 Days</SelectItem>
                <SelectItem value="30d">30 Days</SelectItem>
                <SelectItem value="90d">90 Days</SelectItem>
              </SelectContent>
            </Select>
          </div>
        )}

        <div className="space-y-4">
          {isLoading ? (
            Array.from({ length: 5 }).map((_, index) => (
              <div
                key={index}
                className="flex items-start space-x-3 p-3 border rounded-lg"
              >
                <div className="h-8 w-8 bg-muted rounded-full animate-pulse" />
                <div className="flex-1 space-y-2">
                  <div className="h-4 w-3/4 bg-muted rounded animate-pulse" />
                  <div className="h-3 w-1/2 bg-muted rounded animate-pulse" />
                </div>
                <div className="h-6 w-16 bg-muted rounded animate-pulse" />
              </div>
            ))
          ) : activities.length === 0 ? (
            <div className="text-center py-8">
              <Activity className="h-8 w-8 text-muted-foreground mx-auto mb-2" />
              <p className="text-muted-foreground">No activity found</p>
            </div>
          ) : (
            activities.map((activity) => {
              const config = activityConfig[activity.action];
              return (
                <div
                  key={activity.id}
                  className="flex items-start space-x-3 p-3 border rounded-lg hover:bg-muted/50 transition-colors"
                >
                  <Avatar className="h-8 w-8">
                    <AvatarImage src={activity.userAvatar} />
                    <AvatarFallback>
                      {activity.userName
                        .split(" ")
                        .map((n) => n[0])
                        .join("")}
                    </AvatarFallback>
                  </Avatar>

                  <div className="flex-1 min-w-0">
                    <div className="flex items-center gap-2 mb-1">
                      <span className="font-medium">{activity.userName}</span>
                      <Badge className={`text-xs ${config.color}`}>
                        <span className="flex items-center gap-1">
                          {config.icon}
                          {config.label}
                        </span>
                      </Badge>
                    </div>

                    <p className="text-sm text-muted-foreground mb-1">
                      {formatActivityDetails(activity)}
                    </p>

                    <div className="flex items-center gap-4 text-xs text-muted-foreground">
                      <span className="flex items-center gap-1">
                        <Clock className="h-3 w-3" />
                        {format(activity.timestamp, "MMM d, yyyy 'at' h:mm a")}
                      </span>
                      {activity.ipAddress && (
                        <span>IP: {activity.ipAddress}</span>
                      )}
                    </div>
                  </div>
                </div>
              );
            })
          )}
        </div>
      </CardContent>
    </Card>
  );
}
