/**
 * PropertyPro - Bulk User Operations Component
 * Handle bulk operations for user management
 */

"use client";

import React, { useState } from "react";
import { toast } from "sonner";
import {
  Dialog,
  DialogContent,
  DialogDescription,
  DialogFooter,
  DialogHeader,
  DialogTitle,
} from "@/components/ui/dialog";
import { Button } from "@/components/ui/button";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select";
import { Label } from "@/components/ui/label";
import { Textarea } from "@/components/ui/textarea";
import { Badge } from "@/components/ui/badge";
import { Alert, AlertDescription } from "@/components/ui/alert";
import {
  Users,
  UserCheck,
  UserX,
  Shield,
  Mail,
  AlertTriangle,
  Loader2,
} from "lucide-react";
import { UserRole, IUser } from "@/types";
import { RoleBadge, getRoleLabel } from "@/components/ui/role-badge";

interface BulkOperationsProps {
  selectedUsers: IUser[];
  isOpen: boolean;
  onClose: () => void;
  onSuccess: () => void;
}

type BulkAction =
  | "activate"
  | "deactivate"
  | "change_role"
  | "send_email"
  | "delete";

interface BulkActionConfig {
  id: BulkAction;
  label: string;
  description: string;
  icon: React.ReactNode;
  variant: "default" | "destructive" | "secondary";
  requiresConfirmation: boolean;
}

const bulkActions: BulkActionConfig[] = [
  {
    id: "activate",
    label: "Activate Users",
    description: "Enable selected users to access the system",
    icon: <UserCheck className="h-4 w-4" />,
    variant: "default",
    requiresConfirmation: false,
  },
  {
    id: "deactivate",
    label: "Deactivate Users",
    description: "Disable selected users from accessing the system",
    icon: <UserX className="h-4 w-4" />,
    variant: "secondary",
    requiresConfirmation: true,
  },
  {
    id: "change_role",
    label: "Change Role",
    description: "Update the role for selected users",
    icon: <Shield className="h-4 w-4" />,
    variant: "default",
    requiresConfirmation: true,
  },
  {
    id: "send_email",
    label: "Send Email",
    description: "Send a notification email to selected users",
    icon: <Mail className="h-4 w-4" />,
    variant: "default",
    requiresConfirmation: false,
  },
  {
    id: "delete",
    label: "Delete Users",
    description: "Permanently remove selected users (cannot be undone)",
    icon: <AlertTriangle className="h-4 w-4" />,
    variant: "destructive",
    requiresConfirmation: true,
  },
];

export function BulkOperations({
  selectedUsers,
  isOpen,
  onClose,
  onSuccess,
}: BulkOperationsProps) {
  const [selectedAction, setSelectedAction] = useState<BulkAction | "">("");
  const [newRole, setNewRole] = useState<UserRole | "">("");
  const [emailSubject, setEmailSubject] = useState("");
  const [emailMessage, setEmailMessage] = useState("");
  const [isLoading, setIsLoading] = useState(false);
  const [showConfirmation, setShowConfirmation] = useState(false);

  const selectedActionConfig = bulkActions.find(
    (action) => action.id === selectedAction
  );

  const handleActionSelect = (action: BulkAction) => {
    setSelectedAction(action);
    const actionConfig = bulkActions.find((a) => a.id === action);

    if (actionConfig?.requiresConfirmation) {
      setShowConfirmation(true);
    }
  };

  const executeBulkAction = async () => {
    if (!selectedAction || selectedUsers.length === 0) return;

    try {
      setIsLoading(true);
      const userIds = selectedUsers.map((user) => user._id.toString());

      let endpoint = "/api/users";
      let method = "PUT";
      let body: any = { userIds };

      switch (selectedAction) {
        case "activate":
          body.updates = { isActive: true };
          break;

        case "deactivate":
          body.updates = { isActive: false };
          break;

        case "change_role":
          if (!newRole) {
            toast.error("Please select a new role");
            return;
          }
          body.updates = { role: newRole };
          break;

        case "send_email":
          if (!emailSubject || !emailMessage) {
            toast.error("Please provide email subject and message");
            return;
          }
          endpoint = "/api/users/bulk-email";
          body = {
            userIds,
            subject: emailSubject,
            message: emailMessage,
          };
          method = "POST";
          break;

        case "delete":
          endpoint = `/api/users?ids=${userIds.join(",")}`;
          method = "DELETE";
          body = undefined;
          break;

        default:
          toast.error("Invalid action selected");
          return;
      }

      const response = await fetch(endpoint, {
        method,
        headers:
          method !== "DELETE"
            ? {
                "Content-Type": "application/json",
              }
            : {},
        body: body ? JSON.stringify(body) : undefined,
      });

      const result = await response.json();

      if (!response.ok) {
        throw new Error(result.message || "Failed to execute bulk action");
      }

      toast.success(
        `Successfully ${selectedActionConfig?.label.toLowerCase()} ${
          selectedUsers.length
        } user(s)`
      );
      onSuccess();
      handleClose();
    } catch (error) {
      console.error("Bulk operation error:", error);
      toast.error(
        error instanceof Error ? error.message : "Failed to execute bulk action"
      );
    } finally {
      setIsLoading(false);
    }
  };

  const handleClose = () => {
    setSelectedAction("");
    setNewRole("");
    setEmailSubject("");
    setEmailMessage("");
    setShowConfirmation(false);
    onClose();
  };

  const renderActionForm = () => {
    switch (selectedAction) {
      case "change_role":
        return (
          <div className="space-y-4">
            <div>
              <Label htmlFor="new-role">New Role</Label>
              <Select
                value={newRole}
                onValueChange={(value) => setNewRole(value as UserRole)}
              >
                <SelectTrigger>
                  <SelectValue placeholder="Select new role" />
                </SelectTrigger>
                <SelectContent>
                  {Object.values(UserRole).map((role) => (
                    <SelectItem key={role} value={role}>
                      <div className="flex items-center gap-2">
                        <RoleBadge role={role} size="sm" />
                        <span>{getRoleLabel(role)}</span>
                      </div>
                    </SelectItem>
                  ))}
                </SelectContent>
              </Select>
            </div>
          </div>
        );

      case "send_email":
        return (
          <div className="space-y-4">
            <div>
              <Label htmlFor="email-subject">Subject</Label>
              <input
                id="email-subject"
                type="text"
                value={emailSubject}
                onChange={(e) => setEmailSubject(e.target.value)}
                placeholder="Email subject"
                className="w-full px-3 py-2 border border-input rounded-md"
              />
            </div>
            <div>
              <Label htmlFor="email-message">Message</Label>
              <Textarea
                id="email-message"
                value={emailMessage}
                onChange={(e) => setEmailMessage(e.target.value)}
                placeholder="Email message content"
                rows={4}
              />
            </div>
          </div>
        );

      default:
        return null;
    }
  };

  return (
    <Dialog open={isOpen} onOpenChange={handleClose}>
      <DialogContent className="sm:max-w-md">
        <DialogHeader>
          <DialogTitle className="flex items-center gap-2">
            <Users className="h-5 w-5" />
            Bulk Operations
          </DialogTitle>
          <DialogDescription>
            Perform actions on {selectedUsers.length} selected user(s)
          </DialogDescription>
        </DialogHeader>

        <div className="space-y-4">
          {/* Selected Users Summary */}
          <div className="p-3 bg-muted rounded-lg">
            <p className="text-sm font-medium mb-2">Selected Users:</p>
            <div className="flex flex-wrap gap-1">
              {selectedUsers.slice(0, 3).map((user) => (
                <Badge
                  key={user._id.toString()}
                  variant="outline"
                  className="text-xs"
                >
                  {user.firstName} {user.lastName}
                </Badge>
              ))}
              {selectedUsers.length > 3 && (
                <Badge variant="outline" className="text-xs">
                  +{selectedUsers.length - 3} more
                </Badge>
              )}
            </div>
          </div>

          {/* Action Selection */}
          {!showConfirmation && (
            <div>
              <Label>Select Action</Label>
              <div className="grid gap-2 mt-2">
                {bulkActions.map((action) => (
                  <Button
                    key={action.id}
                    variant={
                      selectedAction === action.id ? action.variant : "outline"
                    }
                    className="justify-start h-auto p-3"
                    onClick={() => handleActionSelect(action.id)}
                  >
                    <div className="flex items-start gap-3">
                      {action.icon}
                      <div className="text-left">
                        <div className="font-medium">{action.label}</div>
                        <div className="text-xs text-muted-foreground">
                          {action.description}
                        </div>
                      </div>
                    </div>
                  </Button>
                ))}
              </div>
            </div>
          )}

          {/* Action Form */}
          {selectedAction && !showConfirmation && renderActionForm()}

          {/* Confirmation */}
          {showConfirmation && selectedActionConfig && (
            <Alert>
              <AlertTriangle className="h-4 w-4" />
              <AlertDescription>
                Are you sure you want to{" "}
                {selectedActionConfig.label.toLowerCase()}{" "}
                {selectedUsers.length} user(s)?
                {selectedActionConfig.id === "delete" &&
                  " This action cannot be undone."}
              </AlertDescription>
            </Alert>
          )}
        </div>

        <DialogFooter>
          <Button variant="outline" onClick={handleClose} disabled={isLoading}>
            Cancel
          </Button>
          {selectedAction && (
            <Button
              variant={selectedActionConfig?.variant}
              onClick={
                showConfirmation
                  ? executeBulkAction
                  : () => setShowConfirmation(true)
              }
              disabled={isLoading}
            >
              {isLoading && <Loader2 className="mr-2 h-4 w-4 animate-spin" />}
              {showConfirmation ? "Confirm" : "Continue"}
            </Button>
          )}
        </DialogFooter>
      </DialogContent>
    </Dialog>
  );
}
