/**
 * PropertyPro - Image Upload Component
 * Reusable component for uploading images with drag & drop support
 */

"use client";

import React, { useState, useCallback, useRef } from "react";
import { Button } from "@/components/ui/button";
import { Progress } from "@/components/ui/progress";
import {
  Upload,
  Image as ImageIcon,
  Loader2,
  CheckCircle,
  X,
  Eye,
} from "lucide-react";
import { toast } from "sonner";
import Image from "next/image";

export interface UploadedImage {
  url: string;
  publicId: string;
  width?: number;
  height?: number;
  format?: string;
  bytes?: number;
}

interface ImageUploadProps {
  onImagesUploaded: (images: UploadedImage[]) => void;
  onImagesRemoved?: (images: UploadedImage[]) => void;
  existingImages?: UploadedImage[];
  maxFiles?: number;
  folder?: string;
  quality?: string;
  maxWidth?: number;
  maxHeight?: number;
  disabled?: boolean;
  className?: string;
}

export function ImageUpload({
  onImagesUploaded,
  onImagesRemoved,
  existingImages = [],
  maxFiles = 10,
  folder = "PropertyPro/properties",
  quality = "auto",
  maxWidth,
  maxHeight,
  disabled = false,
  className = "",
}: ImageUploadProps) {
  const [images, setImages] = useState<UploadedImage[]>(existingImages);
  const [uploading, setUploading] = useState(false);
  const [uploadProgress, setUploadProgress] = useState(0);
  const [dragActive, setDragActive] = useState(false);
  const fileInputRef = useRef<HTMLInputElement>(null);

  const handleFiles = useCallback(
    async (files: FileList | null) => {
      if (!files || files.length === 0 || disabled) return;

      const fileArray = Array.from(files);
      const remainingSlots = maxFiles - images.length;

      if (fileArray.length > remainingSlots) {
        toast.error(`You can only upload ${remainingSlots} more image(s)`);
        return;
      }

      // Validate files
      const validFiles = fileArray.filter((file) => {
        if (!file.type.startsWith("image/")) {
          toast.error(`${file.name} is not an image file`);
          return false;
        }
        if (file.size > 10 * 1024 * 1024) {
          toast.error(`${file.name} is too large (max 10MB)`);
          return false;
        }
        return true;
      });

      if (validFiles.length === 0) return;

      setUploading(true);
      setUploadProgress(0);

      try {
        const formData = new FormData();
        validFiles.forEach((file) => formData.append("files", file));
        formData.append("folder", folder);
        formData.append("quality", quality);
        if (maxWidth) formData.append("maxWidth", maxWidth.toString());
        if (maxHeight) formData.append("maxHeight", maxHeight.toString());

        const response = await fetch("/api/upload/images", {
          method: "POST",
          body: formData,
          credentials: "include",
        });

        const result = await response.json();

        if (!response.ok) {
          throw new Error(result.error || "Upload failed");
        }

        const newImages = result.images as UploadedImage[];
        const updatedImages = [...images, ...newImages];

        setImages(updatedImages);
        onImagesUploaded(newImages);

        toast.success(`Successfully uploaded ${newImages.length} image(s)`);
      } catch (error) {
        console.error("Upload error:", error);
        toast.error(error instanceof Error ? error.message : "Upload failed");
      } finally {
        setUploading(false);
        setUploadProgress(0);
        if (fileInputRef.current) {
          fileInputRef.current.value = "";
        }
      }
    },
    [
      images,
      maxFiles,
      folder,
      quality,
      maxWidth,
      maxHeight,
      disabled,
      onImagesUploaded,
    ]
  );

  const removeImage = useCallback(
    (index: number) => {
      const newImages = images.filter((_, i) => i !== index);
      const removedImage = images[index];

      setImages(newImages);
      if (onImagesRemoved) {
        onImagesRemoved([removedImage]);
      }
    },
    [images, onImagesRemoved]
  );

  const handleDrag = useCallback((e: React.DragEvent) => {
    e.preventDefault();
    e.stopPropagation();
    if (e.type === "dragenter" || e.type === "dragover") {
      setDragActive(true);
    } else if (e.type === "dragleave") {
      setDragActive(false);
    }
  }, []);

  const handleDrop = useCallback(
    (e: React.DragEvent) => {
      e.preventDefault();
      e.stopPropagation();
      setDragActive(false);

      if (disabled) return;

      const files = e.dataTransfer.files;
      handleFiles(files);
    },
    [handleFiles, disabled]
  );

  const openFileDialog = () => {
    if (!disabled) {
      fileInputRef.current?.click();
    }
  };

  return (
    <div className={`space-y-6 ${className}`}>
      {/* Upload Area */}
      <div className="relative">
        <div
          className={`border-2 border-dashed rounded-xl p-8 text-center transition-all duration-200 cursor-pointer ${
            dragActive
              ? "border-blue-400 bg-blue-50/50 scale-[1.02]"
              : disabled
              ? "border-gray-200 bg-gray-50 cursor-not-allowed"
              : "border-gray-300 hover:border-blue-300 hover:bg-blue-50/30"
          }`}
          onDragEnter={handleDrag}
          onDragLeave={handleDrag}
          onDragOver={handleDrag}
          onDrop={handleDrop}
          onClick={openFileDialog}
        >
          {uploading ? (
            <div className="space-y-4">
              <div className="relative">
                <div className="w-16 h-16 mx-auto bg-blue-100 rounded-full flex items-center justify-center">
                  <Loader2 className="h-8 w-8 text-blue-600 animate-spin" />
                </div>
              </div>
              <div className="space-y-2">
                <p className="text-lg font-semibold text-gray-900">
                  Uploading images...
                </p>
                <p className="text-sm text-gray-600">
                  Please wait while we process your files
                </p>
              </div>
              <div className="w-full max-w-sm mx-auto">
                <Progress value={uploadProgress} className="h-2" />
                <p className="text-xs text-gray-500 mt-1">
                  {uploadProgress}% complete
                </p>
              </div>
            </div>
          ) : (
            <div className="space-y-4">
              <div className="relative">
                <div className="w-16 h-16 mx-auto bg-gradient-to-br from-blue-100 to-blue-200 rounded-full flex items-center justify-center">
                  <Upload className="h-8 w-8 text-blue-600" />
                </div>
                {dragActive && (
                  <div className="absolute inset-0 w-16 h-16 mx-auto bg-blue-200 rounded-full animate-ping opacity-75"></div>
                )}
              </div>
              <div className="space-y-2">
                <p className="text-lg font-semibold text-gray-900">
                  {disabled
                    ? "Upload disabled"
                    : dragActive
                    ? "Drop your images here!"
                    : "Upload property images"}
                </p>
                <p className="text-sm text-gray-600">
                  {disabled
                    ? "Image upload is currently disabled"
                    : "Drag and drop your images here, or click to browse files"}
                </p>
              </div>
              <div className="flex items-center justify-center space-x-4 text-xs text-gray-500">
                <span className="flex items-center">
                  <CheckCircle className="h-3 w-3 mr-1 text-green-500" />
                  PNG, JPG, GIF
                </span>
                <span className="flex items-center">
                  <CheckCircle className="h-3 w-3 mr-1 text-green-500" />
                  Up to 10MB each
                </span>
                <span className="flex items-center">
                  <CheckCircle className="h-3 w-3 mr-1 text-green-500" />
                  {images.length}/{maxFiles} uploaded
                </span>
              </div>
              {!disabled && (
                <Button
                  type="button"
                  variant="outline"
                  className="bg-white hover:bg-blue-50 border-blue-200 text-blue-700 hover:border-blue-300"
                >
                  <ImageIcon className="h-4 w-4 mr-2" />
                  Choose Files
                </Button>
              )}
            </div>
          )}
        </div>

        <input
          ref={fileInputRef}
          type="file"
          multiple
          accept="image/*"
          onChange={(e) => handleFiles(e.target.files)}
          className="hidden"
          disabled={disabled}
        />
      </div>

      {/* Image Preview Grid */}
      {images.length > 0 && (
        <div className="mt-6 space-y-4">
          <div className="flex items-center justify-between">
            <h4 className="text-sm font-semibold text-gray-900 flex items-center">
              <ImageIcon className="h-4 w-4 mr-2 text-blue-600" />
              Uploaded Images ({images.length})
            </h4>
            {!disabled && images.length > 0 && (
              <Button
                size="sm"
                variant="outline"
                onClick={() => setImages([])}
                className="text-red-600 hover:text-red-700 hover:bg-red-50 border-red-200"
              >
                <X className="h-3 w-3 mr-1" />
                Clear All
              </Button>
            )}
          </div>

          <div className="grid grid-cols-2 md:grid-cols-3 lg:grid-cols-4 gap-4">
            {images.map((image, index) => (
              <div key={image.publicId || index} className="relative group">
                <div className="aspect-square relative overflow-hidden rounded-xl border-2 border-gray-200 bg-gray-50 hover:border-blue-300 transition-colors">
                  <Image
                    src={image.url}
                    alt={`Upload ${index + 1}`}
                    fill
                    className="object-cover transition-transform duration-200 group-hover:scale-105"
                    sizes="(max-width: 768px) 50vw, (max-width: 1024px) 33vw, 25vw"
                  />

                  {/* Success indicator */}
                  <div className="absolute top-2 left-2">
                    <div className="w-6 h-6 bg-green-500 rounded-full flex items-center justify-center shadow-lg">
                      <CheckCircle className="h-4 w-4 text-white" />
                    </div>
                  </div>

                  {/* Overlay with actions */}
                  <div className="absolute inset-0 bg-black/60 opacity-0 group-hover:opacity-100 transition-all duration-200 flex items-center justify-center space-x-2">
                    <Button
                      size="sm"
                      variant="secondary"
                      className="bg-white/90 hover:bg-white text-gray-900 shadow-lg"
                      onClick={(e) => {
                        e.stopPropagation();
                        window.open(image.url, "_blank");
                      }}
                    >
                      <Eye className="h-4 w-4" />
                    </Button>
                    {!disabled && (
                      <Button
                        size="sm"
                        variant="destructive"
                        className="bg-red-500/90 hover:bg-red-600 text-white shadow-lg"
                        onClick={(e) => {
                          e.stopPropagation();
                          removeImage(index);
                        }}
                      >
                        <X className="h-4 w-4" />
                      </Button>
                    )}
                  </div>
                </div>

                {/* Image info */}
                <div className="mt-2 space-y-1">
                  <div className="flex items-center justify-between text-xs">
                    <span className="text-gray-600 font-medium">
                      Image {index + 1}
                    </span>
                    {image.bytes && (
                      <span className="text-gray-500 bg-gray-100 px-2 py-0.5 rounded">
                        {(image.bytes / 1024 / 1024).toFixed(1)} MB
                      </span>
                    )}
                  </div>
                  {image.width && image.height && (
                    <p className="text-xs text-gray-500">
                      {image.width} × {image.height} px
                    </p>
                  )}
                </div>
              </div>
            ))}
          </div>

          {/* Upload Summary */}
          <div className="p-3 bg-green-50 border border-green-200 rounded-lg">
            <div className="flex items-center text-sm text-green-800">
              <CheckCircle className="h-4 w-4 mr-2 text-green-600" />
              <span className="font-medium">
                {images.length} image{images.length > 1 ? "s" : ""} ready to
                upload
              </span>
            </div>
            <p className="text-xs text-green-600 mt-1">
              Review your images above. You can remove individual images or
              clear all to start over.
            </p>
          </div>
        </div>
      )}
    </div>
  );
}
