"use client";

import { useState } from "react";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import {
  Card,
  CardContent,
  CardDescription,
  CardHeader,
  CardTitle,
} from "@/components/ui/card";
import {
  Dialog,
  DialogContent,
  DialogDescription,
  DialogFooter,
  DialogHeader,
  DialogTitle,
  DialogTrigger,
} from "@/components/ui/dialog";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select";
import { Textarea } from "@/components/ui/textarea";
import { Label } from "@/components/ui/label";
import { toast } from "sonner";
import {
  CheckCircle,
  XCircle,
  Clock,
  UserCheck,
  UserX,
  Settings,
  History,
  AlertTriangle,
} from "lucide-react";

interface TenantStatusManagerProps {
  tenant: {
    _id: string;
    firstName: string;
    lastName: string;
    tenantStatus?: string;
    displayStatus?: string;
    statusColor?: string;
    backgroundCheckStatus?: string;
    lastStatusUpdate?: string;
    statusHistory?: Array<{
      status: string;
      changedBy: {
        firstName: string;
        lastName: string;
      };
      changedAt: string;
      reason?: string;
      notes?: string;
    }>;
  };
  onStatusChange?: (newStatus: string) => void;
  userRole: string;
}

export default function TenantStatusManager({
  tenant,
  onStatusChange,
  userRole,
}: TenantStatusManagerProps) {
  const [isDialogOpen, setIsDialogOpen] = useState(false);
  const [selectedStatus, setSelectedStatus] = useState("");
  const [reason, setReason] = useState("");
  const [notes, setNotes] = useState("");
  const [isLoading, setIsLoading] = useState(false);
  const [showHistory, setShowHistory] = useState(false);

  const canManageStatus = ["admin", "manager"].includes(userRole);

  const statusOptions = [
    {
      value: "under_review",
      label: "Start Review",
      icon: Clock,
      color: "outline",
    },
    {
      value: "approved",
      label: "Approve Application",
      icon: CheckCircle,
      color: "default",
    },
    {
      value: "active",
      label: "Activate Tenant",
      icon: UserCheck,
      color: "default",
    },
    { value: "inactive", label: "Deactivate", icon: UserX, color: "secondary" },
    {
      value: "moved_out",
      label: "Mark Moved Out",
      icon: UserX,
      color: "secondary",
    },
    {
      value: "terminated",
      label: "Terminate",
      icon: XCircle,
      color: "destructive",
    },
  ];

  const getAvailableTransitions = (currentStatus: string) => {
    const transitions = {
      application_submitted: ["under_review", "terminated"],
      under_review: ["approved", "terminated"],
      approved: ["active", "terminated"],
      active: ["inactive", "moved_out", "terminated"],
      inactive: ["active", "moved_out", "terminated"],
      moved_out: ["terminated"],
      terminated: [],
    };
    return transitions[currentStatus] || [];
  };

  const handleStatusChange = async () => {
    if (!selectedStatus) {
      toast.error("Please select a status");
      return;
    }

    setIsLoading(true);
    try {
      const response = await fetch(`/api/tenants/${tenant._id}/status`, {
        method: "POST",
        headers: {
          "Content-Type": "application/json",
        },
        body: JSON.stringify({
          newStatus: selectedStatus,
          reason: reason.trim() || "Status updated by admin",
          notes: notes.trim(),
        }),
      });

      if (!response.ok) {
        const errorData = await response.json();
        throw new Error(errorData.message || "Failed to update status");
      }

      const data = await response.json();
      toast.success(`Tenant status updated to ${selectedStatus}`, {
        description: reason,
      });

      onStatusChange?.(selectedStatus);
      setIsDialogOpen(false);
      setSelectedStatus("");
      setReason("");
      setNotes("");
    } catch (error) {
      toast.error("Failed to update status", {
        description:
          error instanceof Error
            ? error.message
            : "An unexpected error occurred",
      });
    } finally {
      setIsLoading(false);
    }
  };

  const getStatusIcon = (status: string) => {
    const iconMap = {
      application_submitted: Clock,
      under_review: Clock,
      approved: CheckCircle,
      active: UserCheck,
      inactive: UserX,
      moved_out: UserX,
      terminated: XCircle,
    };
    return iconMap[status] || Clock;
  };

  const getStatusColor = (status: string) => {
    const colorMap = {
      application_submitted: "secondary",
      under_review: "outline",
      approved: "default",
      active: "default",
      inactive: "secondary",
      moved_out: "secondary",
      terminated: "destructive",
    };
    return colorMap[status] || "outline";
  };

  const currentStatus = tenant.tenantStatus || "application_submitted";
  const availableTransitions = getAvailableTransitions(currentStatus);
  const StatusIcon = getStatusIcon(currentStatus);

  return (
    <Card>
      <CardHeader>
        <div className="flex items-center justify-between">
          <div>
            <CardTitle className="flex items-center gap-2">
              <Settings className="h-5 w-5" />
              Tenant Status Management
            </CardTitle>
            <CardDescription>
              Current status and available actions for {tenant.firstName}{" "}
              {tenant.lastName}
            </CardDescription>
          </div>
          <Button
            variant="outline"
            size="sm"
            onClick={() => setShowHistory(!showHistory)}
          >
            <History className="h-4 w-4 mr-2" />
            {showHistory ? "Hide" : "Show"} History
          </Button>
        </div>
      </CardHeader>
      <CardContent className="space-y-4">
        {/* Current Status */}
        <div className="flex items-center justify-between p-4 border rounded-lg">
          <div className="flex items-center gap-3">
            <StatusIcon className="h-5 w-5" />
            <div>
              <p className="font-medium">Current Status</p>
              <Badge
                variant={getStatusColor(currentStatus) as any}
                className="mt-1"
              >
                {tenant.displayStatus ||
                  currentStatus.replace("_", " ").toUpperCase()}
              </Badge>
            </div>
          </div>
          {tenant.lastStatusUpdate && (
            <div className="text-right text-sm text-muted-foreground">
              <p>Last Updated</p>
              <p>{new Date(tenant.lastStatusUpdate).toLocaleDateString()}</p>
            </div>
          )}
        </div>

        {/* Status Actions */}
        {canManageStatus && availableTransitions.length > 0 && (
          <div className="space-y-2">
            <Label className="text-sm font-medium">Available Actions</Label>
            <div className="flex flex-wrap gap-2">
              {availableTransitions.map((status) => {
                const option = statusOptions.find(
                  (opt) => opt.value === status
                );
                if (!option) return null;

                const OptionIcon = option.icon;
                return (
                  <Dialog
                    key={status}
                    open={isDialogOpen && selectedStatus === status}
                    onOpenChange={(open) => {
                      if (!open) {
                        setIsDialogOpen(false);
                        setSelectedStatus("");
                        setReason("");
                        setNotes("");
                      }
                    }}
                  >
                    <DialogTrigger asChild>
                      <Button
                        variant="outline"
                        size="sm"
                        onClick={() => {
                          setSelectedStatus(status);
                          setIsDialogOpen(true);
                        }}
                        className="flex items-center gap-2"
                      >
                        <OptionIcon className="h-4 w-4" />
                        {option.label}
                      </Button>
                    </DialogTrigger>
                    <DialogContent>
                      <DialogHeader>
                        <DialogTitle>
                          {option.label} - {tenant.firstName} {tenant.lastName}
                        </DialogTitle>
                        <DialogDescription>
                          This action will change the tenant status to "
                          {option.label.toLowerCase()}". Please provide a reason
                          for this change.
                        </DialogDescription>
                      </DialogHeader>
                      <div className="space-y-4">
                        <div>
                          <Label htmlFor="reason">Reason (Required)</Label>
                          <Textarea
                            id="reason"
                            placeholder="Enter the reason for this status change..."
                            value={reason}
                            onChange={(e) => setReason(e.target.value)}
                            className="mt-1"
                          />
                        </div>
                        <div>
                          <Label htmlFor="notes">
                            Additional Notes (Optional)
                          </Label>
                          <Textarea
                            id="notes"
                            placeholder="Any additional notes or comments..."
                            value={notes}
                            onChange={(e) => setNotes(e.target.value)}
                            className="mt-1"
                          />
                        </div>
                      </div>
                      <DialogFooter>
                        <Button
                          variant="outline"
                          onClick={() => setIsDialogOpen(false)}
                          disabled={isLoading}
                        >
                          Cancel
                        </Button>
                        <Button
                          onClick={handleStatusChange}
                          disabled={isLoading || !reason.trim()}
                          variant={
                            option.color === "destructive"
                              ? "destructive"
                              : "default"
                          }
                        >
                          {isLoading
                            ? "Updating..."
                            : `Confirm ${option.label}`}
                        </Button>
                      </DialogFooter>
                    </DialogContent>
                  </Dialog>
                );
              })}
            </div>
          </div>
        )}

        {/* Status History */}
        {showHistory &&
          tenant.statusHistory &&
          tenant.statusHistory.length > 0 && (
            <div className="space-y-2">
              <Label className="text-sm font-medium">Status History</Label>
              <div className="space-y-2 max-h-60 overflow-y-auto">
                {tenant.statusHistory.map((entry, index) => (
                  <div key={index} className="p-3 border rounded-lg text-sm">
                    <div className="flex items-center justify-between">
                      <Badge
                        variant={getStatusColor(entry.status) as any}
                        className="text-xs"
                      >
                        {entry.status.replace("_", " ").toUpperCase()}
                      </Badge>
                      <span className="text-muted-foreground">
                        {new Date(entry.changedAt).toLocaleDateString()}
                      </span>
                    </div>
                    <p className="mt-1">
                      Changed by: {entry.changedBy.firstName}{" "}
                      {entry.changedBy.lastName}
                    </p>
                    {entry.reason && (
                      <p className="mt-1 text-muted-foreground">
                        Reason: {entry.reason}
                      </p>
                    )}
                    {entry.notes && (
                      <p className="mt-1 text-muted-foreground">
                        Notes: {entry.notes}
                      </p>
                    )}
                  </div>
                ))}
              </div>
            </div>
          )}

        {/* No Actions Available */}
        {canManageStatus && availableTransitions.length === 0 && (
          <div className="flex items-center gap-2 p-3 bg-muted rounded-lg">
            <AlertTriangle className="h-4 w-4 text-muted-foreground" />
            <p className="text-sm text-muted-foreground">
              No status changes available for the current state.
            </p>
          </div>
        )}

        {/* Permission Message */}
        {!canManageStatus && (
          <div className="flex items-center gap-2 p-3 bg-muted rounded-lg">
            <AlertTriangle className="h-4 w-4 text-muted-foreground" />
            <p className="text-sm text-muted-foreground">
              You don't have permission to manage tenant status.
            </p>
          </div>
        )}
      </CardContent>
    </Card>
  );
}
