"use client";

import { useState, useEffect } from "react";
import { toast } from "sonner";
import {
  Card,
  CardContent,
  CardDescription,
  CardHeader,
  CardTitle,
} from "@/components/ui/card";
import {
  Table,
  TableBody,
  TableCell,
  TableHead,
  TableHeader,
  TableRow,
} from "@/components/ui/table";
import { Badge } from "@/components/ui/badge";
import { Button } from "@/components/ui/button";
import { Skeleton } from "@/components/ui/skeleton";
import {
  DropdownMenu,
  DropdownMenuContent,
  DropdownMenuItem,
  DropdownMenuTrigger,
} from "@/components/ui/dropdown-menu";
import {
  FileText,
  MoreHorizontal,
  Eye,
  Download,
  Printer,
  Mail,
  Calendar,
  CheckCircle,
  Clock,
  AlertTriangle,
  XCircle,
  Send,
  Trash2,
} from "lucide-react";
import { InvoiceStatus } from "@/types";
import {
  printInvoice,
  downloadInvoiceAsPDF,
  type PrintableInvoice,
} from "@/lib/invoice-print";
import { normalizeInvoiceForPrint } from "@/lib/invoice/invoice-shared";

interface Invoice {
  _id: string;
  invoiceNumber: string;
  issueDate: string;
  dueDate: string;
  status: InvoiceStatus;
  totalAmount: number;
  amountPaid: number;
  balanceRemaining: number;
  propertyId?: {
    name: string;
    address: any;
  };
  leaseId?: {
    propertyId?: {
      name: string;
      address: any;
    };
  };
  lineItems: Array<{
    description: string;
    amount: number;
    type: string;
  }>;
  daysOverdue?: number;
  createdAt: string;
  updatedAt: string;
}

interface TenantInvoiceHistoryProps {
  tenantId: string;
  className?: string;
}

export default function TenantInvoiceHistory({
  tenantId,
  className,
}: TenantInvoiceHistoryProps) {
  const [invoices, setInvoices] = useState<Invoice[]>([]);
  const [loading, setLoading] = useState(true);
  const [pagination, setPagination] = useState({
    page: 1,
    limit: 10,
    total: 0,
    pages: 0,
  });

  useEffect(() => {
    fetchInvoices();
  }, [tenantId, pagination.page]);

  const fetchInvoices = async () => {
    try {
      setLoading(true);
      const params = new URLSearchParams({
        tenantId,
        page: pagination.page.toString(),
        limit: pagination.limit.toString(),
        sortBy: "dueDate",
        sortOrder: "desc",
      });

      const response = await fetch(`/api/invoices?${params}`);
      const data = await response.json();

      if (data.success) {
        setInvoices(data.data.invoices || []);
        setPagination(data.data.pagination || pagination);
      } else {
        toast.error("Failed to fetch invoice history", {
          description:
            typeof data.error === "string"
              ? data.error
              : "Please try again",
        });
      }
    } catch (error) {
      toast.error("Failed to fetch invoice history", {
        description:
          error instanceof Error ? error.message : "Please try again",
      });
    } finally {
      setLoading(false);
    }
  };

  const formatCurrency = (amount: number) => {
    return new Intl.NumberFormat("en-US", {
      style: "currency",
      currency: "USD",
    }).format(amount);
  };

  const formatDate = (dateString: string) => {
    return new Date(dateString).toLocaleDateString("en-US", {
      year: "numeric",
      month: "short",
      day: "numeric",
    });
  };

  const formatAddress = (address: any): string => {
    if (typeof address === "string") return address;
    if (address && typeof address === "object") {
      const { street, city, state, zipCode } = address;
      return `${street}, ${city}, ${state} ${zipCode}`;
    }
    return "N/A";
  };

  const getStatusBadge = (status: string, daysOverdue?: number) => {
    const statusConfig = {
      scheduled: {
        variant: "secondary" as const,
        label: "Scheduled",
        icon: Calendar,
        className: "bg-blue-100 text-blue-800 border-blue-200",
      },
      issued: {
        variant: "outline" as const,
        label: "Issued",
        icon: Send,
        className: "bg-gray-100 text-gray-800 border-gray-200",
      },
      paid: {
        variant: "default" as const,
        label: "Paid",
        icon: CheckCircle,
        className: "bg-green-100 text-green-800 border-green-200",
      },
      partial: {
        variant: "secondary" as const,
        label: "Partial",
        icon: Clock,
        className: "bg-orange-100 text-orange-800 border-orange-200",
      },
      overdue: {
        variant: "destructive" as const,
        label: `Overdue${daysOverdue ? ` (${daysOverdue}d)` : ""}`,
        icon: AlertTriangle,
        className: "bg-red-100 text-red-800 border-red-200",
      },
      cancelled: {
        variant: "outline" as const,
        label: "Cancelled",
        icon: Trash2,
        className: "bg-gray-100 text-gray-600 border-gray-200",
      },
    };

    const config = statusConfig[status as keyof typeof statusConfig] || {
      variant: "outline" as const,
      label: status,
      icon: FileText,
      className: "bg-gray-100 text-gray-800 border-gray-200",
    };

    const Icon = config.icon;

    return (
      <Badge variant={config.variant} className={config.className}>
        <Icon className="h-3 w-3 mr-1" />
        {config.label}
      </Badge>
    );
  };

  const handleViewInvoice = (invoiceId: string) => {
    window.open(`/dashboard/leases/invoices/${invoiceId}`, "_blank");
  };

  const handleDownloadInvoice = async (invoice: Invoice) => {
    try {
      // Fetch company info from display settings
      const { getCompanyInfo } = await import("@/lib/utils/company-info");
      const companyInfo = await getCompanyInfo();

      const printable = normalizeInvoiceForPrint(invoice, {
        companyInfo: companyInfo || undefined,
      }) as PrintableInvoice;
      await downloadInvoiceAsPDF(printable);
      toast.success("Invoice downloaded successfully");
    } catch (error) {
      toast.error("Failed to download invoice", {
        description:
          error instanceof Error ? error.message : "Please try again",
      });
    }
  };

  const handleEmailInvoice = async (invoiceId: string) => {
    try {
      const response = await fetch(`/api/invoices/email`, {
        method: "POST",
        headers: {
          "Content-Type": "application/json",
        },
        body: JSON.stringify({ invoiceId }),
      });

      if (response.ok) {
        toast.success("Invoice emailed successfully");
      } else {
        toast.error("Failed to email invoice");
      }
    } catch (error) {
      toast.error("Failed to email invoice", {
        description:
          error instanceof Error ? error.message : "Please try again",
      });
    }
  };

  const handlePrintInvoice = async (invoice: Invoice) => {
    try {
      // Fetch company info from display settings
      const { getCompanyInfo } = await import("@/lib/utils/company-info");
      const companyInfo = await getCompanyInfo();

      const printable = normalizeInvoiceForPrint(invoice, {
        companyInfo: companyInfo || undefined,
      }) as PrintableInvoice;
      printInvoice(printable);
    } catch (error) {
      toast.error("Failed to print invoice", {
        description:
          error instanceof Error ? error.message : "Please try again",
      });
    }
  };

  if (loading) {
    return (
      <Card className={className}>
        <CardHeader>
          <CardTitle className="flex items-center gap-2">
            <FileText className="h-5 w-5" />
            Invoice History
          </CardTitle>
          <CardDescription>
            Loading invoice history for this tenant...
          </CardDescription>
        </CardHeader>
        <CardContent>
          <div className="space-y-4">
            {[...Array(5)].map((_, i) => (
              <div key={i} className="flex items-center space-x-4">
                <Skeleton className="h-12 w-12 rounded" />
                <div className="space-y-2 flex-1">
                  <Skeleton className="h-4 w-[250px]" />
                  <Skeleton className="h-4 w-[200px]" />
                </div>
                <Skeleton className="h-4 w-[100px]" />
                <Skeleton className="h-4 w-[80px]" />
              </div>
            ))}
          </div>
        </CardContent>
      </Card>
    );
  }

  return (
    <Card className={className}>
      <CardHeader>
        <CardTitle className="flex items-center gap-2">
          <FileText className="h-5 w-5" />
          Invoice History
        </CardTitle>
        <CardDescription>
          {invoices.length > 0
            ? `${pagination.total} invoice${
                pagination.total !== 1 ? "s" : ""
              } found`
            : "No invoices found for this tenant"}
        </CardDescription>
      </CardHeader>
      <CardContent>
        {invoices.length === 0 ? (
          <div className="text-center py-8">
            <FileText className="h-12 w-12 text-muted-foreground mx-auto mb-4" />
            <h3 className="text-lg font-medium text-gray-900 mb-2">
              No Invoices Found
            </h3>
            <p className="text-muted-foreground">
              This tenant doesn't have any invoices yet.
            </p>
          </div>
        ) : (
          <>
            <div className="border rounded-lg">
              <Table>
                <TableHeader>
                  <TableRow>
                    <TableHead>Invoice #</TableHead>
                    <TableHead>Property</TableHead>
                    <TableHead>Issue Date</TableHead>
                    <TableHead>Due Date</TableHead>
                    <TableHead>Amount</TableHead>
                    <TableHead>Paid</TableHead>
                    <TableHead>Balance</TableHead>
                    <TableHead>Status</TableHead>
                    <TableHead className="w-12"></TableHead>
                  </TableRow>
                </TableHeader>
                <TableBody>
                  {invoices.map((invoice) => (
                    <TableRow key={invoice._id}>
                      <TableCell>
                        <div className="font-mono text-sm">
                          {invoice.invoiceNumber}
                        </div>
                      </TableCell>
                      <TableCell>
                        <div>
                          <div className="font-medium">
                            {invoice.leaseId?.propertyId?.name ||
                              invoice.propertyId?.name ||
                              "N/A"}
                          </div>
                          <div className="text-sm text-muted-foreground">
                            {formatAddress(
                              invoice.leaseId?.propertyId?.address ||
                                invoice.propertyId?.address
                            )}
                          </div>
                        </div>
                      </TableCell>
                      <TableCell>{formatDate(invoice.issueDate)}</TableCell>
                      <TableCell>
                        <div
                          className={`${
                            new Date(invoice.dueDate) < new Date() &&
                            invoice.status !== "paid"
                              ? "text-red-600 font-medium"
                              : ""
                          }`}
                        >
                          {formatDate(invoice.dueDate)}
                        </div>
                      </TableCell>
                      <TableCell className="font-medium">
                        {formatCurrency(invoice.totalAmount)}
                      </TableCell>
                      <TableCell className="text-green-600">
                        {formatCurrency(invoice.amountPaid)}
                      </TableCell>
                      <TableCell
                        className={`font-medium ${
                          invoice.balanceRemaining > 0
                            ? "text-red-600"
                            : "text-green-600"
                        }`}
                      >
                        {formatCurrency(invoice.balanceRemaining)}
                      </TableCell>
                      <TableCell>
                        {getStatusBadge(invoice.status, invoice.daysOverdue)}
                      </TableCell>
                      <TableCell>
                        <DropdownMenu>
                          <DropdownMenuTrigger asChild>
                            <Button variant="ghost" className="h-8 w-8 p-0">
                              <MoreHorizontal className="h-4 w-4" />
                            </Button>
                          </DropdownMenuTrigger>
                          <DropdownMenuContent align="end">
                            <DropdownMenuItem
                              onClick={() => handleViewInvoice(invoice._id)}
                            >
                              <Eye className="mr-2 h-4 w-4" />
                              View Details
                            </DropdownMenuItem>
                            <DropdownMenuItem
                              onClick={() => handleDownloadInvoice(invoice)}
                            >
                              <Download className="mr-2 h-4 w-4" />
                              Download PDF
                            </DropdownMenuItem>
                            <DropdownMenuItem
                              onClick={() => handlePrintInvoice(invoice)}
                            >
                              <Printer className="mr-2 h-4 w-4" />
                              Print Invoice
                            </DropdownMenuItem>
                            <DropdownMenuItem
                              onClick={() => handleEmailInvoice(invoice._id)}
                            >
                              <Mail className="mr-2 h-4 w-4" />
                              Email Invoice
                            </DropdownMenuItem>
                          </DropdownMenuContent>
                        </DropdownMenu>
                      </TableCell>
                    </TableRow>
                  ))}
                </TableBody>
              </Table>
            </div>

            {/* Pagination */}
            {pagination.pages > 1 && (
              <div className="flex items-center justify-between mt-4">
                <div className="text-sm text-muted-foreground">
                  Showing {(pagination.page - 1) * pagination.limit + 1} to{" "}
                  {Math.min(
                    pagination.page * pagination.limit,
                    pagination.total
                  )}{" "}
                  of {pagination.total} invoices
                </div>
                <div className="flex gap-2">
                  <Button
                    variant="outline"
                    size="sm"
                    onClick={() =>
                      setPagination((prev) => ({
                        ...prev,
                        page: prev.page - 1,
                      }))
                    }
                    disabled={pagination.page <= 1}
                  >
                    Previous
                  </Button>
                  <Button
                    variant="outline"
                    size="sm"
                    onClick={() =>
                      setPagination((prev) => ({
                        ...prev,
                        page: prev.page + 1,
                      }))
                    }
                    disabled={pagination.page >= pagination.pages}
                  >
                    Next
                  </Button>
                </div>
              </div>
            )}
          </>
        )}
      </CardContent>
    </Card>
  );
}
