"use client";

import { useState } from "react";
import { Button } from "@/components/ui/button";
import {
  Dialog,
  DialogContent,
  DialogDescription,
  DialogFooter,
  DialogHeader,
  DialogTitle,
} from "@/components/ui/dialog";
import { Textarea } from "@/components/ui/textarea";
import { Label } from "@/components/ui/label";
import { Checkbox } from "@/components/ui/checkbox";
import { toast } from "sonner";
import { AlertTriangle, Loader2, Trash2 } from "lucide-react";

interface TenantDeleteDialogProps {
  isOpen: boolean;
  onClose: () => void;
  tenant: {
    _id: string;
    firstName: string;
    lastName: string;
    tenantStatus?: string;
  };
  onDelete: (tenantId: string) => void;
  userRole: string;
}

export default function TenantDeleteDialog({
  isOpen,
  onClose,
  tenant,
  onDelete,
  userRole,
}: TenantDeleteDialogProps) {
  const [reason, setReason] = useState("");
  const [confirmDelete, setConfirmDelete] = useState(false);
  const [isLoading, setIsLoading] = useState(false);

  const canDelete = ["admin", "manager"].includes(userRole);

  const handleClose = () => {
    setReason("");
    setConfirmDelete(false);
    onClose();
  };

  const handleDelete = async () => {
    if (!reason.trim()) {
      toast.error("Please provide a reason for deletion");
      return;
    }

    if (!confirmDelete) {
      toast.error("Please confirm the deletion");
      return;
    }

    setIsLoading(true);
    try {
      const statusResponse = await fetch(`/api/tenants/${tenant._id}/status`, {
        method: "POST",
        headers: {
          "Content-Type": "application/json",
        },
        body: JSON.stringify({
          newStatus: "terminated",
          reason: `Tenant deleted: ${reason.trim()}`,
          notes: "Tenant record deleted by administrator",
        }),
      });

      if (!statusResponse.ok) {
        toast.warning(
          "Tenant status could not be updated before deletion. Continuing with removal."
        );
      }

      const deleteResponse = await fetch(`/api/tenants/${tenant._id}`, {
        method: "DELETE",
        headers: {
          "Content-Type": "application/json",
        },
        body: JSON.stringify({
          reason: reason.trim(),
        }),
      });

      if (!deleteResponse.ok) {
        const errorData = await deleteResponse.json();
        throw new Error(errorData.message || "Failed to delete tenant");
      }

      toast.success("Tenant deleted successfully", {
        description: `${tenant.firstName} ${tenant.lastName} has been removed from the system`,
      });

      onDelete(tenant._id);
      handleClose();
    } catch (error) {
      toast.error("Failed to delete tenant", {
        description:
          error instanceof Error ? error.message : "Please try again",
      });
    } finally {
      setIsLoading(false);
    }
  };

  if (!canDelete) {
    return null;
  }

  return (
    <Dialog open={isOpen} onOpenChange={handleClose}>
      <DialogContent className="sm:max-w-[500px]">
        <DialogHeader>
          <DialogTitle className="flex items-center gap-2 text-destructive">
            <Trash2 className="h-5 w-5" />
            Delete Tenant
          </DialogTitle>
          <DialogDescription>
            Are you sure you want to delete {tenant.firstName} {tenant.lastName}
            ?
          </DialogDescription>
        </DialogHeader>

        <div className="space-y-6">
          {/* Warning */}
          <div className="flex items-start gap-2 p-4 border border-destructive/20 bg-destructive/5 rounded-lg">
            <AlertTriangle className="h-5 w-5 text-destructive mt-0.5" />
            <div className="text-sm">
              <p className="font-medium text-destructive">
                Warning: This action cannot be undone
              </p>
              <div className="mt-2 space-y-1 text-muted-foreground">
                <p>• The tenant will be marked as terminated</p>
                <p>• All tenant data will be soft-deleted</p>
                <p>
                  • Associated leases and payments will remain for record
                  keeping
                </p>
                <p>• The tenant will no longer appear in active lists</p>
              </div>
            </div>
          </div>

          {/* Tenant Info */}
          <div className="p-3 border rounded-lg bg-muted/50">
            <div className="space-y-1">
              <p className="font-medium">
                {tenant.firstName} {tenant.lastName}
              </p>
              <p className="text-sm text-muted-foreground">
                Current Status:{" "}
                {tenant.tenantStatus?.replace("_", " ").toUpperCase() ||
                  "Unknown"}
              </p>
            </div>
          </div>

          {/* Reason */}
          <div className="space-y-2">
            <Label htmlFor="reason">Reason for Deletion *</Label>
            <Textarea
              id="reason"
              placeholder="Enter the reason for deleting this tenant (required for audit trail)"
              value={reason}
              onChange={(e) => setReason(e.target.value)}
              rows={3}
            />
          </div>

          {/* Confirmation */}
          <div className="flex items-start gap-2 p-3 border rounded-lg">
            <Checkbox
              id="confirm"
              checked={confirmDelete}
              onCheckedChange={(checked) =>
                setConfirmDelete(checked as boolean)
              }
            />
            <div className="space-y-1">
              <Label
                htmlFor="confirm"
                className="text-sm font-medium cursor-pointer"
              >
                I understand this action cannot be undone
              </Label>
              <p className="text-xs text-muted-foreground">
                This will permanently remove the tenant from active management
                while preserving historical records.
              </p>
            </div>
          </div>
        </div>

        <DialogFooter>
          <Button variant="outline" onClick={handleClose} disabled={isLoading}>
            Cancel
          </Button>
          <Button
            variant="destructive"
            onClick={handleDelete}
            disabled={isLoading || !reason.trim() || !confirmDelete}
          >
            {isLoading && <Loader2 className="mr-2 h-4 w-4 animate-spin" />}
            Delete Tenant
          </Button>
        </DialogFooter>
      </DialogContent>
    </Dialog>
  );
}
