/**
 * PropertyPro - Lease Table Component
 * Comprehensive table for displaying and managing tenant leases
 */

"use client";

import React, { useState } from "react";
import {
  Card,
  CardContent,
  CardDescription,
  CardHeader,
  CardTitle,
} from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import {
  Table,
  TableBody,
  TableCell,
  TableHead,
  TableHeader,
  TableRow,
} from "@/components/ui/table";
import {
  Dialog,
  DialogContent,
  DialogDescription,
  DialogHeader,
  DialogTitle,
  DialogTrigger,
} from "@/components/ui/dialog";
import {
  DropdownMenu,
  DropdownMenuContent,
  DropdownMenuItem,
  DropdownMenuTrigger,
} from "@/components/ui/dropdown-menu";
import {
  Pagination,
  PaginationContent,
  PaginationItem,
  PaginationLink,
  PaginationNext,
  PaginationPrevious,
} from "@/components/ui/pagination";
import {
  Building2,
  Calendar,
  DollarSign,
  Download,
  Eye,
  MoreHorizontal,
  MessageSquare,
  FileText,
  AlertTriangle,
  CheckCircle,
  Clock,
  MapPin,
  Home,
} from "lucide-react";

interface Lease {
  _id: string;
  propertyId: {
    _id: string;
    name: string;
    address:
      | string
      | {
          street?: string;
          city?: string;
          state?: string;
          zipCode?: string;
          country?: string;
        };
    type: string;
  };
  startDate: string;
  endDate: string;
  status: string;
  terms: {
    rentAmount: number;
    securityDeposit?: number;
    lateFee?: number;
  };
  daysUntilExpiration: number;
  daysUntilStart: number;
  isActive: boolean;
  isUpcoming: boolean;
  isExpired: boolean;
  documents?: string[];
  signedDate?: string;
  renewalOptions?: {
    available: boolean;
    terms?: string;
  };
}

interface LeaseTableProps {
  leases: Lease[];
  onLeaseAction?: (action: string, lease: Lease) => void;
  className?: string;
}

const ITEMS_PER_PAGE = 10;

export default function LeaseTable({
  leases,
  onLeaseAction,
  className,
}: LeaseTableProps) {
  const [currentPage, setCurrentPage] = useState(1);
  const [selectedLease, setSelectedLease] = useState<Lease | null>(null);
  const [showDetailsDialog, setShowDetailsDialog] = useState(false);

  const totalPages = Math.ceil(leases.length / ITEMS_PER_PAGE);
  const startIndex = (currentPage - 1) * ITEMS_PER_PAGE;
  const endIndex = startIndex + ITEMS_PER_PAGE;
  const currentLeases = leases.slice(startIndex, endIndex);

  const formatCurrency = (amount: number) => {
    return new Intl.NumberFormat("en-US", {
      style: "currency",
      currency: "USD",
    }).format(amount);
  };

  const formatDate = (date: string) => {
    return new Date(date).toLocaleDateString("en-US", {
      year: "numeric",
      month: "short",
      day: "numeric",
    });
  };

  const formatAddress = (address: Lease["propertyId"]["address"]) => {
    if (!address) {
      return "Address not available";
    }

    if (typeof address === "string") {
      return address;
    }

    const { street, city, state, zipCode, country } = address;
    const cityState = [city, state].filter(Boolean).join(", ");
    return (
      [street, cityState, zipCode, country]
        .filter((segment) => segment && segment.toString().trim().length > 0)
        .join(", ") || "Address not available"
    );
  };

  const getStatusBadge = (lease: Lease) => {
    if (lease.isActive) {
      return (
        <Badge variant="default" className="bg-green-500">
          <CheckCircle className="w-3 h-3 mr-1" />
          Active
        </Badge>
      );
    } else if (lease.isUpcoming) {
      return (
        <Badge variant="outline" className="border-blue-500 text-blue-500">
          <Clock className="w-3 h-3 mr-1" />
          Upcoming
        </Badge>
      );
    } else if (lease.isExpired) {
      return (
        <Badge variant="secondary" className="bg-gray-500">
          <AlertTriangle className="w-3 h-3 mr-1" />
          Expired
        </Badge>
      );
    }
    return <Badge variant="outline">{lease.status}</Badge>;
  };

  const getDaysDisplay = (lease: Lease) => {
    if (lease.isActive) {
      if (lease.daysUntilExpiration > 0) {
        return (
          <span className="text-sm text-muted-foreground">
            {lease.daysUntilExpiration} days remaining
          </span>
        );
      } else if (lease.daysUntilExpiration === 0) {
        return (
          <span className="text-sm text-red-600 font-medium">
            Expires today
          </span>
        );
      } else {
        return (
          <span className="text-sm text-red-600">
            Expired {Math.abs(lease.daysUntilExpiration)} days ago
          </span>
        );
      }
    } else if (lease.isUpcoming) {
      return (
        <span className="text-sm text-blue-600">
          Starts in {lease.daysUntilStart} days
        </span>
      );
    } else if (lease.isExpired) {
      return (
        <span className="text-sm text-gray-600">
          Expired {Math.abs(lease.daysUntilExpiration)} days ago
        </span>
      );
    }
    return null;
  };

  const handleAction = (action: string, lease: Lease) => {
    if (action === "view-details") {
      setSelectedLease(lease);
      setShowDetailsDialog(true);
    } else if (onLeaseAction) {
      onLeaseAction(action, lease);
    }
  };

  const handlePageChange = (page: number) => {
    setCurrentPage(page);
  };

  return (
    <div className={`space-y-4 ${className}`}>
      <Card>
        <CardHeader>
          <CardTitle className="flex items-center gap-2">
            <Building2 className="h-5 w-5" />
            My Leases ({leases.length})
          </CardTitle>
          <CardDescription>
            Manage all your lease agreements and property details
          </CardDescription>
        </CardHeader>
        <CardContent>
          {leases.length === 0 ? (
            <div className="text-center py-8">
              <Building2 className="h-12 w-12 text-muted-foreground mx-auto mb-4" />
              <h3 className="text-lg font-semibold mb-2">No Leases Found</h3>
              <p className="text-muted-foreground">
                You don't have any leases at this time.
              </p>
            </div>
          ) : (
            <>
              <div className="rounded-md border">
                <Table>
                  <TableHeader>
                    <TableRow>
                      <TableHead>Property</TableHead>
                      <TableHead>Lease Period</TableHead>
                      <TableHead>Monthly Rent</TableHead>
                      <TableHead>Status</TableHead>
                      <TableHead>Days Until</TableHead>
                      <TableHead className="text-right">Actions</TableHead>
                    </TableRow>
                  </TableHeader>
                  <TableBody>
                    {currentLeases.map((lease) => (
                      <TableRow key={lease._id}>
                        <TableCell>
                          <div className="flex flex-col">
                            <div className="font-medium">
                              {lease.propertyId.name}
                            </div>
                            <div className="text-sm text-muted-foreground flex items-center gap-1">
                              <MapPin className="h-3 w-3" />
                              {formatAddress(lease.propertyId.address)}
                            </div>
                            <div className="text-xs text-muted-foreground capitalize">
                              {lease.propertyId.type}
                            </div>
                          </div>
                        </TableCell>
                        <TableCell>
                          <div className="flex flex-col">
                            <div className="text-sm font-medium">
                              {formatDate(lease.startDate)} -{" "}
                              {formatDate(lease.endDate)}
                            </div>
                            <div className="text-xs text-muted-foreground">
                              {Math.ceil(
                                (new Date(lease.endDate).getTime() -
                                  new Date(lease.startDate).getTime()) /
                                  (1000 * 60 * 60 * 24 * 30)
                              )}{" "}
                              months
                            </div>
                          </div>
                        </TableCell>
                        <TableCell>
                          <div className="flex flex-col">
                            <div className="font-medium">
                              {formatCurrency(lease.terms.rentAmount)}
                            </div>
                            {lease.terms.securityDeposit && (
                              <div className="text-xs text-muted-foreground">
                                Deposit:{" "}
                                {formatCurrency(lease.terms.securityDeposit)}
                              </div>
                            )}
                          </div>
                        </TableCell>
                        <TableCell>{getStatusBadge(lease)}</TableCell>
                        <TableCell>{getDaysDisplay(lease)}</TableCell>
                        <TableCell className="text-right">
                          <DropdownMenu>
                            <DropdownMenuTrigger asChild>
                              <Button variant="ghost" className="h-8 w-8 p-0">
                                <span className="sr-only">Open menu</span>
                                <MoreHorizontal className="h-4 w-4" />
                              </Button>
                            </DropdownMenuTrigger>
                            <DropdownMenuContent align="end">
                              <DropdownMenuItem
                                onClick={() =>
                                  handleAction("view-details", lease)
                                }
                              >
                                <Eye className="mr-2 h-4 w-4" />
                                View Details
                              </DropdownMenuItem>
                              <DropdownMenuItem
                                onClick={() =>
                                  handleAction("download-agreement", lease)
                                }
                              >
                                <Download className="mr-2 h-4 w-4" />
                                Download Agreement
                              </DropdownMenuItem>
                              <DropdownMenuItem
                                onClick={() =>
                                  handleAction("view-invoices", lease)
                                }
                              >
                                <FileText className="mr-2 h-4 w-4" />
                                View Invoices
                              </DropdownMenuItem>
                              <DropdownMenuItem
                                onClick={() =>
                                  handleAction("contact-manager", lease)
                                }
                              >
                                <MessageSquare className="mr-2 h-4 w-4" />
                                Contact Manager
                              </DropdownMenuItem>
                              {lease.renewalOptions?.available && (
                                <DropdownMenuItem
                                  onClick={() =>
                                    handleAction("request-renewal", lease)
                                  }
                                >
                                  <Calendar className="mr-2 h-4 w-4" />
                                  Request Renewal
                                </DropdownMenuItem>
                              )}
                            </DropdownMenuContent>
                          </DropdownMenu>
                        </TableCell>
                      </TableRow>
                    ))}
                  </TableBody>
                </Table>
              </div>

              {/* Pagination */}
              {totalPages > 1 && (
                <div className="flex items-center justify-center space-x-2 pt-4">
                  <Pagination>
                    <PaginationContent>
                      <PaginationItem>
                        <PaginationPrevious
                          onClick={() =>
                            handlePageChange(Math.max(1, currentPage - 1))
                          }
                          className={
                            currentPage === 1
                              ? "pointer-events-none opacity-50"
                              : "cursor-pointer"
                          }
                        />
                      </PaginationItem>

                      {[...Array(totalPages)].map((_, i) => (
                        <PaginationItem key={i + 1}>
                          <PaginationLink
                            onClick={() => handlePageChange(i + 1)}
                            isActive={currentPage === i + 1}
                            className="cursor-pointer"
                          >
                            {i + 1}
                          </PaginationLink>
                        </PaginationItem>
                      ))}

                      <PaginationItem>
                        <PaginationNext
                          onClick={() =>
                            handlePageChange(
                              Math.min(totalPages, currentPage + 1)
                            )
                          }
                          className={
                            currentPage === totalPages
                              ? "pointer-events-none opacity-50"
                              : "cursor-pointer"
                          }
                        />
                      </PaginationItem>
                    </PaginationContent>
                  </Pagination>
                </div>
              )}
            </>
          )}
        </CardContent>
      </Card>

      {/* Lease Details Dialog */}
      <Dialog open={showDetailsDialog} onOpenChange={setShowDetailsDialog}>
        <DialogContent className="max-w-4xl max-h-[80vh] overflow-y-auto">
          <DialogHeader>
            <DialogTitle className="flex items-center gap-2">
              <Building2 className="h-5 w-5" />
              Lease Details - {selectedLease?.propertyId.name}
            </DialogTitle>
            <DialogDescription>
              Complete lease information and terms
            </DialogDescription>
          </DialogHeader>

          {selectedLease && (
            <div className="space-y-6">
              {/* Property Information */}
              <Card>
                <CardHeader>
                  <CardTitle className="text-lg">
                    Property Information
                  </CardTitle>
                </CardHeader>
                <CardContent className="grid grid-cols-1 md:grid-cols-2 gap-4">
                  <div>
                    <label className="text-sm font-medium text-muted-foreground">
                      Property Name
                    </label>
                    <p className="text-lg font-semibold">
                      {selectedLease.propertyId.name}
                    </p>
                  </div>
                  <div>
                    <label className="text-sm font-medium text-muted-foreground">
                      Property Type
                    </label>
                    <p className="capitalize">
                      {selectedLease.propertyId.type}
                    </p>
                  </div>
                  <div className="md:col-span-2">
                    <label className="text-sm font-medium text-muted-foreground">
                      Address
                    </label>
                    <p className="flex items-center gap-1">
                      <MapPin className="h-4 w-4" />
                      {formatAddress(selectedLease.propertyId.address)}
                    </p>
                  </div>
                </CardContent>
              </Card>

              {/* Lease Terms */}
              <Card>
                <CardHeader>
                  <CardTitle className="text-lg">Lease Terms</CardTitle>
                </CardHeader>
                <CardContent className="grid grid-cols-1 md:grid-cols-2 gap-4">
                  <div>
                    <label className="text-sm font-medium text-muted-foreground">
                      Lease Period
                    </label>
                    <p className="font-semibold">
                      {formatDate(selectedLease.startDate)} -{" "}
                      {formatDate(selectedLease.endDate)}
                    </p>
                  </div>
                  <div>
                    <label className="text-sm font-medium text-muted-foreground">
                      Status
                    </label>
                    <div className="mt-1">{getStatusBadge(selectedLease)}</div>
                  </div>
                  <div>
                    <label className="text-sm font-medium text-muted-foreground">
                      Monthly Rent
                    </label>
                    <p className="text-lg font-semibold">
                      {formatCurrency(selectedLease.terms.rentAmount)}
                    </p>
                  </div>
                  {selectedLease.terms.securityDeposit && (
                    <div>
                      <label className="text-sm font-medium text-muted-foreground">
                        Security Deposit
                      </label>
                      <p className="font-semibold">
                        {formatCurrency(selectedLease.terms.securityDeposit)}
                      </p>
                    </div>
                  )}
                  {selectedLease.terms.lateFee && (
                    <div>
                      <label className="text-sm font-medium text-muted-foreground">
                        Late Fee
                      </label>
                      <p className="font-semibold">
                        {formatCurrency(selectedLease.terms.lateFee)}
                      </p>
                    </div>
                  )}
                  <div>
                    <label className="text-sm font-medium text-muted-foreground">
                      Signed Date
                    </label>
                    <p>
                      {selectedLease.signedDate
                        ? formatDate(selectedLease.signedDate)
                        : "Not signed"}
                    </p>
                  </div>
                </CardContent>
              </Card>

              {/* Renewal Options */}
              {selectedLease.renewalOptions?.available && (
                <Card>
                  <CardHeader>
                    <CardTitle className="text-lg">Renewal Options</CardTitle>
                  </CardHeader>
                  <CardContent>
                    <div className="flex items-center gap-2 mb-2">
                      <CheckCircle className="h-4 w-4 text-green-500" />
                      <span className="font-medium">Renewal Available</span>
                    </div>
                    {selectedLease.renewalOptions.terms && (
                      <p className="text-sm text-muted-foreground">
                        {selectedLease.renewalOptions.terms}
                      </p>
                    )}
                    <Button
                      className="mt-4"
                      onClick={() =>
                        handleAction("request-renewal", selectedLease)
                      }
                    >
                      Request Renewal
                    </Button>
                  </CardContent>
                </Card>
              )}

              {/* Actions */}
              <div className="flex flex-wrap gap-2 pt-4 border-t">
                <Button
                  onClick={() =>
                    handleAction("download-agreement", selectedLease)
                  }
                >
                  <Download className="mr-2 h-4 w-4" />
                  Download Agreement
                </Button>
                <Button
                  variant="outline"
                  onClick={() => handleAction("view-invoices", selectedLease)}
                >
                  <FileText className="mr-2 h-4 w-4" />
                  View Invoices
                </Button>
                <Button
                  variant="outline"
                  onClick={() => handleAction("contact-manager", selectedLease)}
                >
                  <MessageSquare className="mr-2 h-4 w-4" />
                  Contact Manager
                </Button>
              </div>
            </div>
          )}
        </DialogContent>
      </Dialog>
    </div>
  );
}
