/**
 * PropertyPro - Localization Provider
 * Context provider for localization features
 */

"use client";

import React, {
  createContext,
  useContext,
  useEffect,
  useState,
  useCallback,
  useRef,
} from "react";
import {
  localizationService,
  Currency,
  Locale,
} from "@/lib/services/localization.service";
import { ensureIntlCurrencyPatch } from "@/lib/utils/ensure-intl-currency-patch";

export interface LocalizationContextType {
  currentLocale: string;
  currentCurrency: string;
  locale: Locale | undefined;
  currency: Currency | undefined;
  allLocales: Locale[];
  allCurrencies: Currency[];
  setLocale: (localeCode: string) => void;
  setCurrency: (currencyCode: string) => void;
  formatCurrency: (
    amount: number,
    currencyCode?: string,
    options?: any
  ) => string;
  formatDate: (date: Date | string, options?: any) => string;
  formatTime: (date: Date | string, options?: any) => string;
  formatNumber: (number: number, options?: any) => string;
  formatPercentage: (value: number, options?: any) => string;
  convertCurrency: (
    amount: number,
    fromCurrency: string,
    toCurrency: string
  ) => number;
  isRTL: boolean;
  firstDayOfWeek: number;
  loading: boolean;
}

const LocalizationContext = createContext<LocalizationContextType | undefined>(
  undefined
);

export function useLocalizationContext(): LocalizationContextType {
  const context = useContext(LocalizationContext);
  if (!context) {
    throw new Error(
      "useLocalizationContext must be used within a LocalizationProvider"
    );
  }
  return context;
}

interface LocalizationProviderProps {
  children: React.ReactNode;
  initialLocale?: string;
  initialCurrency?: string;
}

export function LocalizationProvider({
  children,
  initialLocale,
  initialCurrency,
}: LocalizationProviderProps) {
  ensureIntlCurrencyPatch();

  const [currentLocale, setCurrentLocale] = useState(
    initialLocale || localizationService.getCurrentLocale()
  );
  const [currentCurrency, setCurrentCurrency] = useState(
    initialCurrency || localizationService.getCurrentCurrency()
  );
  const [locale, setLocale] = useState(
    localizationService.getLocale(currentLocale)
  );
  const [currency, setCurrency] = useState(
    localizationService.getCurrency(currentCurrency)
  );
  const [loading, setLoading] = useState(true);
  const hasLoadedDisplaySettings = useRef(false);

  const applyCurrency = useCallback(
    (currencyCode: string, persist = false) => {
      if (!currencyCode) {
        return;
      }

      const serviceCurrency = localizationService.getCurrentCurrency();
      if (
        currencyCode === currentCurrency &&
        currencyCode === serviceCurrency
      ) {
        return;
      }

      localizationService.setCurrency(currencyCode);
      setCurrentCurrency(currencyCode);
      setCurrency(localizationService.getCurrency(currencyCode));

      if (persist && typeof window !== "undefined") {
        localStorage.setItem("PropertyPro-currency", currencyCode);
      }
    },
    [currentCurrency]
  );

  // Initialize localization service
  useEffect(() => {
    const initializeLocalization = async () => {
      try {
        // Load user preferences from localStorage
        const savedLocale = localStorage.getItem("PropertyPro-locale");
        const savedCurrency = localStorage.getItem("PropertyPro-currency");

        if (savedLocale && !initialLocale) {
          localizationService.setLocale(savedLocale);
          setCurrentLocale(savedLocale);
          setLocale(localizationService.getLocale(savedLocale));
        } else if (initialLocale) {
          localizationService.setLocale(initialLocale);
          setCurrentLocale(initialLocale);
          setLocale(localizationService.getLocale(initialLocale));
        }

        if (savedCurrency && !initialCurrency) {
          applyCurrency(savedCurrency);
        } else if (initialCurrency) {
          applyCurrency(initialCurrency);
        }

        // Update exchange rates
        await localizationService.updateExchangeRates();
      } catch (error) {
        console.error("Failed to initialize localization:", error);
      } finally {
        setLoading(false);
      }
    };

    initializeLocalization();
  }, [initialLocale, initialCurrency, applyCurrency]);

  const handleSetLocale = useCallback(
    (localeCode: string) => {
      localizationService.setLocale(localeCode);
      setCurrentLocale(localeCode);
      setLocale(localizationService.getLocale(localeCode));

      // Save to localStorage
      localStorage.setItem("PropertyPro-locale", localeCode);

      // Update currency if it changed with locale
      const newCurrency = localizationService.getCurrentCurrency();
      if (newCurrency && newCurrency !== currentCurrency) {
        applyCurrency(newCurrency, true);
      }
    },
    [applyCurrency, currentCurrency]
  );

  const handleSetCurrency = useCallback(
    (currencyCode: string) => {
      applyCurrency(currencyCode, true);
    },
    [applyCurrency]
  );

  useEffect(() => {
    if (typeof window === "undefined") return;

    const handleStorage = (event: StorageEvent) => {
      if (event.key !== "PropertyPro-currency" || !event.newValue) return;
      if (event.newValue === currentCurrency) return;
      applyCurrency(event.newValue, false);
    };

    const handleDisplaySettingsUpdate = (_event: Event) => {
      const latestCurrency = localizationService.getCurrentCurrency();
      if (latestCurrency && latestCurrency !== currentCurrency) {
        applyCurrency(latestCurrency, false);
      }
    };

    window.addEventListener("storage", handleStorage);
    window.addEventListener(
      "pc:display-settings-updated",
      handleDisplaySettingsUpdate
    );

    return () => {
      window.removeEventListener("storage", handleStorage);
      window.removeEventListener(
        "pc:display-settings-updated",
        handleDisplaySettingsUpdate
      );
    };
  }, [applyCurrency, currentCurrency]);

  useEffect(() => {
    if (typeof window === "undefined" || hasLoadedDisplaySettings.current)
      return;

    const controller = new AbortController();

    const loadDisplaySettings = async () => {
      try {
        const response = await fetch(
          "/api/settings/display?includeDefaults=false",
          {
            method: "GET",
            headers: {
              "Content-Type": "application/json",
            },
            cache: "no-store",
            signal: controller.signal,
          }
        );

        if (!response.ok) {
          if (response.status >= 400 && response.status < 500) {
            hasLoadedDisplaySettings.current = true;
          }
          return;
        }

        const raw = await response.json();
        const payload = raw?.data ?? raw;
        const settings =
          payload?.settings ?? payload?.display ?? payload?.data ?? payload;

        if (settings?.currency) {
          applyCurrency(settings.currency, true);
        }

        if (settings?.language) {
          // Display settings store a short language code; try to locate a matching locale.
          const normalized = settings.language.toLowerCase();
          const availableLocales = localizationService.getAllLocales();
          const matchingLocale = availableLocales.find(
            (locale) => locale.code.toLowerCase() === normalized
          );
          const localeByLanguage =
            matchingLocale ||
            availableLocales.find((locale) =>
              locale.code.toLowerCase().startsWith(`${normalized}-`)
            );

          if (localeByLanguage) {
            handleSetLocale(localeByLanguage.code);
          }
        }
        hasLoadedDisplaySettings.current = true;
      } catch (error) {
        if (!controller.signal.aborted) {
          console.warn(
            "Failed to load display settings for localization",
            error
          );
          hasLoadedDisplaySettings.current = true;
        }
      }
    };

    loadDisplaySettings();

    return () => {
      controller.abort();
    };
  }, [applyCurrency, handleSetLocale]);

  const formatCurrency = (
    amount: number,
    currencyCode?: string,
    options?: any
  ) => {
    return localizationService.formatCurrency(amount, currencyCode, options);
  };

  const formatDate = (date: Date | string, options?: any) => {
    return localizationService.formatDate(date, options);
  };

  const formatTime = (date: Date | string, options?: any) => {
    return localizationService.formatTime(date, options);
  };

  const formatNumber = (number: number, options?: any) => {
    return localizationService.formatNumber(number, options);
  };

  const formatPercentage = (value: number, options?: any) => {
    return localizationService.formatPercentage(value, options);
  };

  const convertCurrency = (
    amount: number,
    fromCurrency: string,
    toCurrency: string
  ) => {
    return localizationService.convertCurrency(
      amount,
      fromCurrency,
      toCurrency
    );
  };

  const contextValue: LocalizationContextType = {
    currentLocale,
    currentCurrency,
    locale,
    currency,
    allLocales: localizationService.getAllLocales(),
    allCurrencies: localizationService.getAllCurrencies(),
    setLocale: handleSetLocale,
    setCurrency: handleSetCurrency,
    formatCurrency,
    formatDate,
    formatTime,
    formatNumber,
    formatPercentage,
    convertCurrency,
    isRTL: localizationService.isRTL(),
    firstDayOfWeek: localizationService.getFirstDayOfWeek(),
    loading,
  };

  return (
    <LocalizationContext.Provider value={contextValue}>
      {children}
    </LocalizationContext.Provider>
  );
}

// Export the context for direct use if needed
export { LocalizationContext };
