"use client";

/**
 * PropertyPro - Unit Statistics Component
 * Display statistics for available units
 */

import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import {
  Building2,
  DollarSign,
  Home,
  TrendingUp,
  Users,
  MapPin,
} from "lucide-react";
import { AvailableUnitResponse } from "@/lib/services/property.service";

interface UnitStatsProps {
  units: AvailableUnitResponse[];
  isLoading?: boolean;
}

export default function UnitStats({ units, isLoading }: UnitStatsProps) {
  if (isLoading) {
    return (
      <div className="grid gap-4 md:grid-cols-2 lg:grid-cols-6">
        {Array.from({ length: 6 }).map((_, i) => (
          <Card key={i}>
            <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
              <CardTitle className="text-sm font-medium">
                <div className="h-4 bg-muted rounded animate-pulse w-20" />
              </CardTitle>
              <div className="h-4 w-4 bg-muted rounded animate-pulse" />
            </CardHeader>
            <CardContent>
              <div className="h-8 bg-muted rounded animate-pulse w-16 mb-1" />
              <div className="h-3 bg-muted rounded animate-pulse w-24" />
            </CardContent>
          </Card>
        ))}
      </div>
    );
  }

  // Calculate statistics
  const totalUnits = units.length;
  const averageRent =
    totalUnits > 0
      ? Math.round(
          units.reduce((sum, unit) => sum + unit.rentAmount, 0) / totalUnits
        )
      : 0;

  const unitTypes = units.reduce((acc, unit) => {
    acc[unit.unitType] = (acc[unit.unitType] || 0) + 1;
    return acc;
  }, {} as Record<string, number>);

  const mostCommonUnitType = Object.entries(unitTypes).reduce(
    (max, [type, count]) => (count > max.count ? { type, count } : max),
    { type: "N/A", count: 0 }
  );

  const rentRange =
    totalUnits > 0
      ? {
          min: Math.min(...units.map((unit) => unit.rentAmount)),
          max: Math.max(...units.map((unit) => unit.rentAmount)),
        }
      : { min: 0, max: 0 };

  const uniqueProperties = new Set(units.map((unit) => unit._id)).size;

  const bedroomDistribution = units.reduce((acc, unit) => {
    const bedrooms = unit.bedrooms;
    acc[bedrooms] = (acc[bedrooms] || 0) + 1;
    return acc;
  }, {} as Record<number, number>);

  const mostCommonBedrooms = Object.entries(bedroomDistribution).reduce(
    (max, [bedrooms, count]) =>
      count > max.count ? { bedrooms: Number(bedrooms), count } : max,
    { bedrooms: 0, count: 0 }
  );

  const formatCurrency = (amount: number) => {
    return new Intl.NumberFormat("en-US", {
      style: "currency",
      currency: "USD",
      minimumFractionDigits: 0,
      maximumFractionDigits: 0,
    }).format(amount);
  };

  const stats = [
    {
      title: "Total Available Units",
      value: totalUnits.toLocaleString(),
      description: `Across ${uniqueProperties} ${
        uniqueProperties === 1 ? "property" : "properties"
      }`,
      icon: Home,
      color: "text-blue-600",
      bgColor: "bg-blue-50",
      borderColor: "border-blue-100",
    },
    {
      title: "Average Rent",
      value: formatCurrency(averageRent),
      description: `Range: ${formatCurrency(rentRange.min)} - ${formatCurrency(
        rentRange.max
      )}`,
      icon: DollarSign,
      color: "text-green-600",
      bgColor: "bg-green-50",
      borderColor: "border-green-100",
    },
    {
      title: "Most Common Type",
      value:
        mostCommonUnitType.type.charAt(0).toUpperCase() +
        mostCommonUnitType.type.slice(1),
      description: `${mostCommonUnitType.count} ${
        mostCommonUnitType.count === 1 ? "unit" : "units"
      } available`,
      icon: Building2,
      color: "text-purple-600",
      bgColor: "bg-purple-50",
      borderColor: "border-purple-100",
    },
    {
      title: "Popular Layout",
      value: `${mostCommonBedrooms.bedrooms} ${
        mostCommonBedrooms.bedrooms === 1 ? "Bedroom" : "Bedrooms"
      }`,
      description: `${mostCommonBedrooms.count} ${
        mostCommonBedrooms.count === 1 ? "unit" : "units"
      } available`,
      icon: Users,
      color: "text-orange-600",
      bgColor: "bg-orange-50",
      borderColor: "border-orange-100",
    },
    {
      title: "Unique Properties",
      value: uniqueProperties.toLocaleString(),
      description: `With available units`,
      icon: MapPin,
      color: "text-indigo-600",
      bgColor: "bg-indigo-50",
      borderColor: "border-indigo-100",
    },
    {
      title: "Occupancy Rate",
      value: "Available Now",
      description: "All units ready for rent",
      icon: TrendingUp,
      color: "text-emerald-600",
      bgColor: "bg-emerald-50",
      borderColor: "border-emerald-100",
    },
  ];

  return (
    <div className="grid gap-4 md:grid-cols-2 lg:grid-cols-6">
      {stats.map((stat, index) => {
        const Icon = stat.icon;
        return (
          <Card
            key={index}
            className={`border-0 shadow-sm hover:shadow-md transition-shadow duration-200 ${stat.bgColor} ${stat.borderColor} dark:bg-muted dark:border-gray-700  border gap-2 p-3`}
          >
            <CardHeader className="flex flex-row items-center justify-between px-2 space-y-0">
              <CardTitle className="text-sm font-medium text-gray-700 dark:text-amber-50">
                {stat.title}
              </CardTitle>
              <div
                className={`p-2 rounded-lg ${stat.bgColor} border ${stat.borderColor}`}
              >
                <Icon className={`h-4 w-4 ${stat.color}`} />
              </div>
            </CardHeader>
            <CardContent className="px-2 pb-1">
              <div className={`text-2xl font-bold ${stat.color} mb-1`}>
                {stat.value}
              </div>
              <p className="text-xs text-gray-600 dark:text-gray-400">{stat.description}</p>
            </CardContent>
          </Card>
        );
      })}
    </div>
  );
}
