/**
 * PropertyPro - Property Statistics Component
 * Dashboard statistics for property management
 */

"use client";

import { useMemo } from "react";
import {
  Card,
  CardContent,
  CardDescription,
  CardHeader,
  CardTitle,
} from "@/components/ui/card";
import { Badge } from "@/components/ui/badge";
import {
  Building2,
  Users,
  DollarSign,
  TrendingUp,
  CheckCircle,
  Wrench,
} from "lucide-react";
import { PropertyResponse } from "@/lib/services/property.service";
import { PropertyStatus, PropertyType } from "@/types";
import {
  AnalyticsCard,
  AnalyticsCardGrid,
} from "@/components/analytics/AnalyticsCard";

interface PropertyStatsProps {
  properties: PropertyResponse[];
  isLoading?: boolean;
}

export default function PropertyStats({
  properties,
  isLoading,
}: PropertyStatsProps) {
  const stats = useMemo(() => {
    if (!properties.length) {
      return {
        total: 0,
        available: 0,
        occupied: 0,
        maintenance: 0,
        unavailable: 0,
        averageRent: 0,
        totalRentValue: 0,
        averageSquareFootage: 0,
        houses: 0,
        apartments: 0,
        condos: 0,
        townhouses: 0,
        commercial: 0,
        thisMonthAdded: 0,
        lastMonthAdded: 0,
        totalUnits: 0,
        availableUnits: 0,
        occupiedUnits: 0,
        maintenanceUnits: 0,
      };
    }

    const now = new Date();
    const thisMonth = new Date(now.getFullYear(), now.getMonth(), 1);
    const lastMonth = new Date(now.getFullYear(), now.getMonth() - 1, 1);
    const lastMonthEnd = new Date(now.getFullYear(), now.getMonth(), 0);

    // Status counts for properties
    const available = properties.filter(
      (p) => p.status === PropertyStatus.AVAILABLE
    ).length;
    const occupied = properties.filter(
      (p) => p.status === PropertyStatus.OCCUPIED
    ).length;
    const maintenance = properties.filter(
      (p) => p.status === PropertyStatus.MAINTENANCE
    ).length;
    const unavailable = properties.filter(
      (p) => p.status === PropertyStatus.UNAVAILABLE
    ).length;

    // Unit-based statistics
    let totalUnits = 0;
    let availableUnits = 0;
    let occupiedUnits = 0;
    let maintenanceUnits = 0;

    properties.forEach((property) => {
      if (property.isMultiUnit && property.units) {
        totalUnits += property.units.length;
        property.units.forEach((unit) => {
          switch (unit.status) {
            case PropertyStatus.AVAILABLE:
              availableUnits++;
              break;
            case PropertyStatus.OCCUPIED:
              occupiedUnits++;
              break;
            case PropertyStatus.MAINTENANCE:
              maintenanceUnits++;
              break;
          }
        });
      } else {
        // Single unit properties count as 1 unit
        totalUnits += 1;
        switch (property.status) {
          case PropertyStatus.AVAILABLE:
            availableUnits++;
            break;
          case PropertyStatus.OCCUPIED:
            occupiedUnits++;
            break;
          case PropertyStatus.MAINTENANCE:
            maintenanceUnits++;
            break;
        }
      }
    });

    // Type counts
    const houses = properties.filter(
      (p) => p.type === PropertyType.HOUSE
    ).length;
    const apartments = properties.filter(
      (p) => p.type === PropertyType.APARTMENT
    ).length;
    const condos = properties.filter(
      (p) => p.type === PropertyType.CONDO
    ).length;
    const townhouses = properties.filter(
      (p) => p.type === PropertyType.TOWNHOUSE
    ).length;
    const commercial = properties.filter(
      (p) => p.type === PropertyType.COMMERCIAL
    ).length;

    // Financial calculations
    const totalRentValue = properties.reduce((sum, p) => sum + p.rentAmount, 0);
    const averageRent = Math.round(totalRentValue / properties.length);

    // Square footage
    const totalSquareFootage = properties.reduce(
      (sum, p) => sum + p.squareFootage,
      0
    );
    const averageSquareFootage = Math.round(
      totalSquareFootage / properties.length
    );

    // Monthly additions
    const thisMonthAdded = properties.filter(
      (p) => new Date(p.createdAt) >= thisMonth
    ).length;

    const lastMonthAdded = properties.filter((p) => {
      const createdDate = new Date(p.createdAt);
      return createdDate >= lastMonth && createdDate <= lastMonthEnd;
    }).length;

    return {
      total: properties.length,
      available,
      occupied,
      maintenance,
      unavailable,
      averageRent,
      totalRentValue,
      averageSquareFootage,
      houses,
      apartments,
      condos,
      townhouses,
      commercial,
      thisMonthAdded,
      lastMonthAdded,
      totalUnits,
      availableUnits,
      occupiedUnits,
      maintenanceUnits,
    };
  }, [properties]);

  const formatCurrency = (amount: number) => {
    return new Intl.NumberFormat("en-US", {
      style: "currency",
      currency: "USD",
      maximumFractionDigits: 0,
    }).format(amount);
  };

  if (isLoading) {
    return (
      <AnalyticsCardGrid>
        {[...Array(5)].map((_, i) => (
          <Card key={i} className="hover:shadow-md transition-shadow">
            <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
              <CardTitle className="text-sm font-medium">
                <div className="h-4 bg-muted rounded animate-pulse" />
              </CardTitle>
              <div className="h-4 w-4 bg-muted rounded animate-pulse" />
            </CardHeader>
            <CardContent>
              <div className="h-8 bg-muted rounded animate-pulse mb-2" />
              <div className="h-3 bg-muted rounded animate-pulse w-2/3" />
            </CardContent>
          </Card>
        ))}
      </AnalyticsCardGrid>
    );
  }

  return (
    <AnalyticsCardGrid>
      <AnalyticsCard
        title="Total Properties"
        value={stats.total}
        description="All property listings"
        icon={Building2}
        iconColor="primary"
      />

      <AnalyticsCard
        title="Available Properties"
        value={stats.available}
        description="Ready for rent"
        icon={CheckCircle}
        iconColor="success"
      />

      <AnalyticsCard
        title="Occupied Properties"
        value={stats.occupied}
        description="Currently rented"
        icon={Users}
        iconColor="info"
      />

      <AnalyticsCard
        title="Average Rent"
        value={
          stats.averageRent > 0 ? formatCurrency(stats.averageRent) : "N/A"
        }
        description="Monthly rental rate"
        icon={DollarSign}
        iconColor="success"
      />

      <AnalyticsCard
        title="Under Maintenance"
        value={stats.maintenance}
        description="Needs attention"
        icon={Wrench}
        iconColor="warning"
      />

      <AnalyticsCard
        title="Total Rent Value"
        value={
          stats.totalRentValue > 0
            ? formatCurrency(stats.totalRentValue)
            : "N/A"
        }
        description="Combined monthly potential"
        icon={TrendingUp}
        iconColor="success"
      />

      {/* Unit-based statistics for multi-unit properties */}
      {stats.totalUnits > stats.total && (
        <>
          <AnalyticsCard
            title="Total Units"
            value={stats.totalUnits}
            description="All rental units"
            icon={Building2}
            iconColor="primary"
          />

          <AnalyticsCard
            title="Available Units"
            value={stats.availableUnits}
            description="Ready for tenants"
            icon={CheckCircle}
            iconColor="success"
          />

          <AnalyticsCard
            title="Occupied Units"
            value={stats.occupiedUnits}
            description="Currently rented"
            icon={Users}
            iconColor="info"
          />

          <AnalyticsCard
            title="Units in Maintenance"
            value={stats.maintenanceUnits}
            description="Under repair"
            icon={Wrench}
            iconColor="warning"
          />
        </>
      )}
    </AnalyticsCardGrid>
  );
}
