"use client";

import { forwardRef } from "react";
import { Card, CardContent, CardHeader } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { Separator } from "@/components/ui/separator";
import {
  CheckCircle,
  Download,
  Printer,
  Building2,
  User,
  Calendar,
  CreditCard,
  FileText,
  MapPin,
} from "lucide-react";
import { PaymentStatus, PaymentType, PaymentMethod } from "@/types";

interface PaymentReceiptProps {
  payment: {
    _id: string;
    amount: number;
    type: PaymentType;
    status: PaymentStatus;
    paymentMethod?: PaymentMethod;
    dueDate: string;
    paidDate?: string;
    description?: string;
    notes?: string;
    tenantId: {
      _id: string;
      userId: {
        firstName: string;
        lastName: string;
        email: string;
        phone?: string;
      };
    };
    propertyId: {
      _id: string;
      name: string;
      address: {
        street: string;
        city: string;
        state: string;
        zipCode: string;
      };
    };
    leaseId?: {
      _id: string;
      startDate: string;
      endDate: string;
    };
    createdAt: string;
    updatedAt: string;
  };
  companyInfo?: {
    name: string;
    address: string;
    phone: string;
    email: string;
    website?: string;
  };
  onPrint?: () => void;
  onDownload?: () => void;
  showActions?: boolean;
}

export const PaymentReceipt = forwardRef<HTMLDivElement, PaymentReceiptProps>(
  ({ payment, companyInfo, onPrint, onDownload, showActions = true }, ref) => {
    const formatCurrency = (amount: number) => {
      return new Intl.NumberFormat("en-US", {
        style: "currency",
        currency: "USD",
      }).format(amount);
    };

    const formatDate = (dateString: string) => {
      return new Date(dateString).toLocaleDateString("en-US", {
        year: "numeric",
        month: "long",
        day: "numeric",
      });
    };

    const formatDateTime = (dateString: string) => {
      return new Date(dateString).toLocaleString("en-US", {
        year: "numeric",
        month: "short",
        day: "numeric",
        hour: "2-digit",
        minute: "2-digit",
      });
    };

    const getPaymentTypeLabel = (type: PaymentType) => {
      return type.replace("_", " ").replace(/\b\w/g, (l) => l.toUpperCase());
    };

    const getPaymentMethodLabel = (method?: PaymentMethod) => {
      if (!method) return "Not specified";
      return method.replace("_", " ").replace(/\b\w/g, (l) => l.toUpperCase());
    };

    const defaultCompanyInfo = {
      name: "PropertyPro Management",
      address: "123 Business Ave, Suite 100, San Francisco, CA 94102",
      phone: "(555) 123-4567",
      email: "support@PropertyPro.com",
      website: "www.PropertyPro.com",
    };

    const company = companyInfo || defaultCompanyInfo;

    return (
      <div ref={ref} className="max-w-2xl mx-auto">
        {/* Actions Bar - Only show when not printing */}
        {showActions && (
          <div className="flex justify-end gap-2 mb-4 print:hidden">
            <Button variant="outline" onClick={onPrint}>
              <Printer className="h-4 w-4 mr-2" />
              Print
            </Button>
            <Button variant="outline" onClick={onDownload}>
              <Download className="h-4 w-4 mr-2" />
              Download PDF
            </Button>
          </div>
        )}

        {/* Receipt Content */}
        <Card className="print:shadow-none print:border-none">
          <CardHeader className="text-center pb-6">
            {/* Company Header */}
            <div className="space-y-2">
              <h1 className="text-2xl font-bold text-primary">
                {company.name}
              </h1>
              <p className="text-sm text-muted-foreground">{company.address}</p>
              <div className="flex justify-center gap-4 text-sm text-muted-foreground">
                <span>{company.phone}</span>
                <span>•</span>
                <span>{company.email}</span>
                {company.website && (
                  <>
                    <span>•</span>
                    <span>{company.website}</span>
                  </>
                )}
              </div>
            </div>

            <Separator className="my-6" />

            {/* Receipt Title */}
            <div className="space-y-2">
              <h2 className="text-xl font-semibold flex items-center justify-center gap-2">
                <CheckCircle className="h-5 w-5 text-green-600" />
                Payment Receipt
              </h2>
              <p className="text-sm text-muted-foreground">
                Receipt #{payment._id.slice(-8).toUpperCase()}
              </p>
            </div>
          </CardHeader>

          <CardContent className="space-y-6">
            {/* Payment Status */}
            <div className="text-center">
              <Badge
                variant={
                  payment.status === PaymentStatus.COMPLETED
                    ? "default"
                    : "secondary"
                }
                className="text-sm px-4 py-2"
              >
                {payment.status.charAt(0).toUpperCase() +
                  payment.status.slice(1)}
              </Badge>
            </div>

            {/* Payment Details */}
            <div className="grid gap-6 md:grid-cols-2">
              {/* Tenant Information */}
              <div className="space-y-3">
                <h3 className="font-semibold flex items-center gap-2">
                  <User className="h-4 w-4" />
                  Tenant Information
                </h3>
                <div className="space-y-1 text-sm">
                  <p className="font-medium">
                    {payment?.tenantId?.userId?.firstName ?? ""}{" "}
                    {payment?.tenantId?.userId?.lastName ?? ""}
                  </p>
                  <p className="text-muted-foreground">
                    {payment?.tenantId?.userId?.email ?? ""}
                  </p>
                  {payment?.tenantId?.userId?.phone && (
                    <p className="text-muted-foreground">
                      {payment.tenantId.userId.phone}
                    </p>
                  )}
                </div>
              </div>

              {/* Property Information */}
              <div className="space-y-3">
                <h3 className="font-semibold flex items-center gap-2">
                  <Building2 className="h-4 w-4" />
                  Property Information
                </h3>
                <div className="space-y-1 text-sm">
                  <p className="font-medium">
                    {payment?.propertyId?.name ?? ""}
                  </p>
                  <div className="text-muted-foreground">
                    <p>{payment?.propertyId?.address?.street ?? ""}</p>
                    <p>
                      {payment?.propertyId?.address?.city ?? ""},{" "}
                      {payment?.propertyId?.address?.state ?? ""}{" "}
                      {payment?.propertyId?.address?.zipCode ?? ""}
                    </p>
                  </div>
                </div>
              </div>
            </div>

            <Separator />

            {/* Payment Summary */}
            <div className="space-y-4">
              <h3 className="font-semibold flex items-center gap-2">
                <CreditCard className="h-4 w-4" />
                Payment Summary
              </h3>

              <div className="space-y-3">
                <div className="flex justify-between items-center">
                  <span className="text-muted-foreground">Payment Type:</span>
                  <span className="font-medium">
                    {getPaymentTypeLabel(payment.type)}
                  </span>
                </div>

                <div className="flex justify-between items-center">
                  <span className="text-muted-foreground">Payment Method:</span>
                  <span className="font-medium">
                    {getPaymentMethodLabel(payment.paymentMethod)}
                  </span>
                </div>

                <div className="flex justify-between items-center">
                  <span className="text-muted-foreground">Due Date:</span>
                  <span className="font-medium">
                    {formatDate(payment.dueDate)}
                  </span>
                </div>

                {payment.paidDate && (
                  <div className="flex justify-between items-center">
                    <span className="text-muted-foreground">Paid Date:</span>
                    <span className="font-medium">
                      {formatDate(payment.paidDate)}
                    </span>
                  </div>
                )}

                <div className="flex justify-between items-center">
                  <span className="text-muted-foreground">
                    Transaction Date:
                  </span>
                  <span className="font-medium">
                    {formatDateTime(payment.createdAt)}
                  </span>
                </div>
              </div>

              <Separator />

              {/* Amount */}
              <div className="flex justify-between items-center text-lg">
                <span className="font-semibold">Total Amount:</span>
                <span className="font-bold text-2xl">
                  {formatCurrency(payment.amount)}
                </span>
              </div>
            </div>

            {/* Description */}
            {payment.description && (
              <>
                <Separator />
                <div className="space-y-2">
                  <h3 className="font-semibold flex items-center gap-2">
                    <FileText className="h-4 w-4" />
                    Description
                  </h3>
                  <p className="text-sm text-muted-foreground">
                    {payment.description}
                  </p>
                </div>
              </>
            )}

            {/* Lease Information */}
            {payment.leaseId && (
              <>
                <Separator />
                <div className="space-y-2">
                  <h3 className="font-semibold flex items-center gap-2">
                    <Calendar className="h-4 w-4" />
                    Lease Information
                  </h3>
                  <div className="text-sm space-y-1">
                    <div className="flex justify-between">
                      <span className="text-muted-foreground">
                        Lease Period:
                      </span>
                      <span>
                        {formatDate(payment.leaseId.startDate)} -{" "}
                        {formatDate(payment.leaseId.endDate)}
                      </span>
                    </div>
                  </div>
                </div>
              </>
            )}

            <Separator />

            {/* Footer */}
            <div className="text-center text-xs text-muted-foreground space-y-2">
              <p>Thank you for your payment!</p>
              <p>
                This receipt was generated on{" "}
                {formatDateTime(new Date().toISOString())}
              </p>
              <p>
                For questions about this payment, please contact us at{" "}
                {company.email} or {company.phone}
              </p>
            </div>
          </CardContent>
        </Card>
      </div>
    );
  }
);

PaymentReceipt.displayName = "PaymentReceipt";
