"use client";

import { useState } from "react";
import Link from "next/link";
import {
  Card,
  CardContent,
} from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import {
  DropdownMenu,
  DropdownMenuContent,
  DropdownMenuItem,
  DropdownMenuLabel,
  DropdownMenuSeparator,
  DropdownMenuTrigger,
} from "@/components/ui/dropdown-menu";
import { DeleteConfirmationDialog } from "@/components/ui/confirmation-dialog";
import {
  FileText,
  MoreHorizontal,
  Edit,
  Eye,
  Trash2,
  Calendar,
  DollarSign,
  Home,
  User,
  FileSignature,
  RotateCcw,
  XCircle,
  AlertTriangle,
  CheckCircle,
  Clock,
  Download,
} from "lucide-react";
import { LeaseStatus } from "@/types";
import { LeaseResponse, leaseService } from "@/lib/services/lease.service";
import { toast } from "sonner";
import { LeaseInvoiceModal } from "@/components/invoices";

interface LeaseCardProps {
  lease: LeaseResponse;
  onUpdate?: () => void;
  onDelete?: (id: string) => void;
  showActions?: boolean;
}

export function LeaseCard({
  lease,
  onUpdate,
  onDelete,
  showActions = true,
}: LeaseCardProps) {
  const [isDeleting, setIsDeleting] = useState(false);

  const getStatusColor = (status: LeaseStatus) => {
    switch (status) {
      case LeaseStatus.DRAFT:
        return "outline";
      case LeaseStatus.PENDING:
        return "default";
      case LeaseStatus.ACTIVE:
        return "secondary";
      case LeaseStatus.EXPIRED:
        return "destructive";
      case LeaseStatus.TERMINATED:
        return "destructive";
      default:
        return "outline";
    }
  };

  const getStatusIcon = (status: LeaseStatus) => {
    switch (status) {
      case LeaseStatus.DRAFT:
        return Edit;
      case LeaseStatus.PENDING:
        return Clock;
      case LeaseStatus.ACTIVE:
        return CheckCircle;
      case LeaseStatus.EXPIRED:
        return AlertTriangle;
      case LeaseStatus.TERMINATED:
        return XCircle;
      default:
        return FileText;
    }
  };

  const formatCurrency = (amount: number) => {
    return new Intl.NumberFormat("en-US", {
      style: "currency",
      currency: "USD",
    }).format(amount);
  };

  const formatDate = (dateString: string) => {
    return new Date(dateString).toLocaleDateString("en-US", {
      year: "numeric",
      month: "short",
      day: "numeric",
    });
  };

  const getDaysRemaining = () => {
    const endDate = new Date(lease.endDate);
    const now = new Date();
    const diffTime = endDate.getTime() - now.getTime();
    const diffDays = Math.ceil(diffTime / (1000 * 60 * 60 * 24));
    return diffDays;
  };

  const handleDelete = async () => {
    try {
      setIsDeleting(true);
      await leaseService.deleteLease(lease._id);
      toast.success("Lease deleted successfully");
      onDelete?.(lease._id);
    } catch (error) {
      console.error("Error deleting lease:", error);
      toast.error(
        error instanceof Error ? error.message : "Failed to delete lease"
      );
    } finally {
      setIsDeleting(false);
    }
  };

  const StatusIcon = getStatusIcon(lease.status);
  const daysRemaining = getDaysRemaining();

  return (
    <>
      <Card className="hover:shadow-lg transition-all duration-300 border-border/50 overflow-hidden group p-0">
        {/* Header with gradient background */}
        <div className="relative bg-gradient-to-br from-primary/5 via-primary/3 to-background p-5 border-b">
          <div className="flex items-start justify-between mb-3">
            <div className="flex items-center gap-3">
              <div className="p-2.5 rounded-lg bg-background shadow-sm border">
                <Home className="h-5 w-5 text-primary" />
              </div>
              <div className="space-y-1">
                <div className="flex items-center gap-2">
                  <h3 className="font-semibold text-base text-foreground">
                    {lease.propertyId.name}
                  </h3>
                  {lease.unit && (
                    <Badge variant="secondary" className="text-xs px-2 py-0.5">
                      Unit {lease.unit.unitNumber}
                    </Badge>
                  )}
                </div>
                <p className="text-sm text-muted-foreground line-clamp-1">
                  {lease.propertyId.address.street}, {lease.propertyId.address.city}, {lease.propertyId.address.state}
                </p>
              </div>
            </div>
            <div className="flex items-center gap-2">
              <Badge
                variant={getStatusColor(lease.status) as any}
                className="flex items-center gap-1.5 px-2.5 py-1"
              >
                <StatusIcon className="h-3 w-3" />
                <span className="capitalize">{lease.status}</span>
              </Badge>
              {showActions && (
                <DropdownMenu>
                  <DropdownMenuTrigger asChild>
                    <Button variant="ghost" size="icon" className="h-8 w-8 opacity-0 group-hover:opacity-100 transition-opacity">
                      <MoreHorizontal className="h-4 w-4" />
                    </Button>
                  </DropdownMenuTrigger>
                  <DropdownMenuContent align="end" className="w-48">
                    <DropdownMenuLabel>Actions</DropdownMenuLabel>
                    <DropdownMenuItem asChild>
                      <Link href={`/dashboard/leases/${lease._id}`}>
                        <Eye className="mr-2 h-4 w-4" />
                        View Details
                      </Link>
                    </DropdownMenuItem>
                    {lease.status === LeaseStatus.DRAFT && (
                      <DropdownMenuItem asChild>
                        <Link href={`/dashboard/leases/${lease._id}/edit`}>
                          <Edit className="mr-2 h-4 w-4" />
                          Edit Lease
                        </Link>
                      </DropdownMenuItem>
                    )}
                    {!lease.signedDate &&
                      lease.status === LeaseStatus.DRAFT && (
                        <DropdownMenuItem>
                          <FileSignature className="mr-2 h-4 w-4" />
                          Send for Signature
                        </DropdownMenuItem>
                      )}
                    {lease.status === LeaseStatus.ACTIVE && (
                      <>
                        <DropdownMenuItem>
                          <RotateCcw className="mr-2 h-4 w-4" />
                          Renew Lease
                        </DropdownMenuItem>
                        <DropdownMenuItem>
                          <XCircle className="mr-2 h-4 w-4" />
                          Terminate Lease
                        </DropdownMenuItem>
                      </>
                    )}
                    <DropdownMenuSeparator />
                    <DropdownMenuItem asChild>
                      <Link href={`/dashboard/leases/${lease._id}/invoice`}>
                        <FileText className="mr-2 h-4 w-4" />
                        View Invoice
                      </Link>
                    </DropdownMenuItem>
                    <DropdownMenuItem asChild>
                      <LeaseInvoiceModal
                        lease={lease}
                        trigger={
                          <div className="flex items-center w-full cursor-pointer">
                            <Download className="mr-2 h-4 w-4" />
                            Download Invoice
                          </div>
                        }
                      />
                    </DropdownMenuItem>
                    <DropdownMenuSeparator />
                    <DeleteConfirmationDialog
                      itemName={`${
                        lease.propertyId?.name || "Unknown Property"
                      } - ${
                        lease.tenantId?.firstName && lease.tenantId?.lastName
                          ? `${lease.tenantId.firstName} ${lease.tenantId.lastName}`
                          : "Unknown Tenant"
                      }`}
                      itemType="lease"
                      onConfirm={handleDelete}
                      loading={isDeleting}
                    >
                      <DropdownMenuItem
                        onSelect={(e) => e.preventDefault()}
                        className="text-destructive focus:text-destructive"
                      >
                        <Trash2 className="mr-2 h-4 w-4" />
                        Delete Lease
                      </DropdownMenuItem>
                    </DeleteConfirmationDialog>
                  </DropdownMenuContent>
                </DropdownMenu>
              )}
            </div>
          </div>

          {/* Tenant Info Badge */}
          <div className="flex items-center gap-2 px-3 py-2 bg-background/80 rounded-lg border w-fit">
            <div className="p-1.5 rounded-full bg-primary/10">
              <User className="h-3.5 w-3.5 text-primary" />
            </div>
            <div className="flex flex-col">
              <span className="text-sm font-medium">
                {lease.tenantId?.firstName && lease.tenantId?.lastName
                  ? `${lease.tenantId.firstName} ${lease.tenantId.lastName}`
                  : "Unknown Tenant"}
              </span>
              <span className="text-xs text-muted-foreground">
                {lease.tenantId?.email || "No email"}
              </span>
            </div>
          </div>
        </div>

        <CardContent className="p-5 space-y-4">
          {/* Lease Period with visual timeline */}
          <div className="space-y-2">
            <div className="flex items-center justify-between">
              <div className="flex items-center gap-2">
                <Calendar className="h-4 w-4 text-muted-foreground" />
                <span className="text-sm font-medium">Lease Period</span>
              </div>
              <span
                className={`text-sm font-semibold ${
                  daysRemaining < 30 && daysRemaining > 0
                    ? "text-orange-600"
                    : daysRemaining <= 0
                    ? "text-red-600"
                    : "text-green-600"
                }`}
              >
                {daysRemaining > 0
                  ? `${daysRemaining} days remaining`
                  : daysRemaining === 0
                  ? "Expires today"
                  : "Expired"}
              </span>
            </div>
            <div className="text-xs text-muted-foreground">
              {formatDate(lease.startDate)} - {formatDate(lease.endDate)}
            </div>
          </div>

          {/* Financial Summary */}
          <div className="grid grid-cols-2 gap-3 p-3 bg-muted/30 rounded-lg">
            <div className="space-y-1">
              <div className="flex items-center gap-1.5 text-muted-foreground">
                <DollarSign className="h-3.5 w-3.5" />
                <span className="text-xs font-medium">Monthly Rent</span>
              </div>
              <p className="text-base font-bold text-foreground">
                {formatCurrency(lease.unit?.rentAmount || lease.terms.rentAmount)}
              </p>
            </div>
            <div className="space-y-1">
              <div className="flex items-center gap-1.5 text-muted-foreground">
                <DollarSign className="h-3.5 w-3.5" />
                <span className="text-xs font-medium">Security</span>
              </div>
              <p className="text-base font-bold text-foreground">
                {formatCurrency(lease.terms.securityDeposit)}
              </p>
            </div>
          </div>

          {/* Property Details */}
          <div className="flex flex-wrap gap-2 pt-2">
            {lease.unit ? (
              <>
                <Badge variant="outline" className="text-xs px-2.5 py-1">
                  {lease.unit.bedrooms} bed
                </Badge>
                <Badge variant="outline" className="text-xs px-2.5 py-1">
                  {lease.unit.bathrooms} bath
                </Badge>
                <Badge variant="outline" className="text-xs px-2.5 py-1">
                  {lease.unit.squareFootage.toLocaleString()} sq ft
                </Badge>
                {lease.unit.floor && (
                  <Badge variant="outline" className="text-xs px-2.5 py-1">
                    Floor {lease.unit.floor}
                  </Badge>
                )}
              </>
            ) : (
              <>
                <Badge variant="outline" className="text-xs px-2.5 py-1">
                  {lease.propertyId.bedrooms} bed
                </Badge>
                <Badge variant="outline" className="text-xs px-2.5 py-1">
                  {lease.propertyId.bathrooms} bath
                </Badge>
                <Badge variant="outline" className="text-xs px-2.5 py-1">
                  {lease.propertyId.squareFootage.toLocaleString()} sq ft
                </Badge>
              </>
            )}
          </div>

          {/* Signed Status */}
          {lease.signedDate && (
            <div className="flex items-center gap-2 px-3 py-2 bg-green-50 dark:bg-green-950/20 rounded-lg border border-green-200 dark:border-green-900">
              <CheckCircle className="h-4 w-4 text-green-600 dark:text-green-400" />
              <span className="text-sm font-medium text-green-700 dark:text-green-300">
                Signed on {formatDate(lease.signedDate)}
              </span>
            </div>
          )}

          {/* Action Buttons */}
          <div className="grid grid-cols-2 gap-2 pt-2">
            <Button variant="outline" size="sm" className="w-full" asChild>
              <Link href={`/dashboard/leases/${lease._id}`}>
                <Eye className="mr-2 h-4 w-4" />
                View Details
              </Link>
            </Button>
            <Button variant="outline" size="sm" className="w-full" asChild>
              <Link href={`/dashboard/leases/${lease._id}/invoice`}>
                <FileText className="mr-2 h-4 w-4" />
                Invoice
              </Link>
            </Button>
            <LeaseInvoiceModal
              lease={lease}
              trigger={
                <Button variant="outline" size="sm" className="w-full">
                  <Download className="mr-2 h-4 w-4" />
                  PDF
                </Button>
              }
            />
            {lease.status === LeaseStatus.DRAFT && (
              <Button variant="outline" size="sm" className="w-full" asChild>
                <Link href={`/dashboard/leases/${lease._id}/edit`}>
                  <Edit className="mr-2 h-4 w-4" />
                  Edit
                </Link>
              </Button>
            )}
          </div>
        </CardContent>
      </Card>
    </>
  );
}
