"use client";

import React, { useEffect, useState } from "react";
import {
  Dialog,
  DialogContent,
  DialogDescription,
  DialogHeader,
  DialogTitle,
  DialogTrigger,
} from "@/components/ui/dialog";
import { Button } from "@/components/ui/button";
import { FileText, Receipt } from "lucide-react";
import { LeaseInvoice } from "./LeaseInvoice";
import { LeaseResponse } from "@/lib/services/lease.service";
import { getCompanyInfo, CompanyInfo } from "@/lib/utils/company-info";

export interface LeaseInvoiceModalProps {
  lease: LeaseResponse;
  trigger?: React.ReactNode;
  companyInfo?: {
    name: string;
    address: string;
    phone: string;
    email: string;
    website?: string;
    logo?: string;
  };
  invoiceNumber?: string;
  issueDate?: Date;
  dueDate?: Date;
}

export function LeaseInvoiceModal({
  lease,
  trigger,
  companyInfo: propCompanyInfo,
  invoiceNumber,
  issueDate,
  dueDate,
}: LeaseInvoiceModalProps) {
  const [companyInfo, setCompanyInfo] = useState<CompanyInfo | null>(
    propCompanyInfo || null
  );
  const [isLoadingCompanyInfo, setIsLoadingCompanyInfo] = useState(false);

  useEffect(() => {
    // Only fetch if company info not provided as prop
    if (!propCompanyInfo) {
      setIsLoadingCompanyInfo(true);
      getCompanyInfo()
        .then((info) => {
          if (info) {
            setCompanyInfo(info);
          }
        })
        .catch((error) => {
          console.error("Failed to fetch company info:", error);
        })
        .finally(() => {
          setIsLoadingCompanyInfo(false);
        });
    }
  }, [propCompanyInfo]);

  const defaultTrigger = (
    <Button variant="outline" size="sm" className="flex items-center gap-2">
      <Receipt className="h-4 w-4" />
      Generate Invoice
    </Button>
  );

  return (
    <Dialog>
      <DialogTrigger asChild>{trigger || defaultTrigger}</DialogTrigger>
      <DialogContent className="!w-[90vw] max-h-[90vh] overflow-y-auto">
        <DialogHeader>
          <DialogTitle className="flex items-center gap-2">
            <FileText className="h-5 w-5" />
            Lease Invoice
          </DialogTitle>
          <DialogDescription>
            Professional invoice for lease agreement and terms
          </DialogDescription>
        </DialogHeader>

        <div className="mt-4">
          {isLoadingCompanyInfo ? (
            <div className="flex justify-center items-center py-8">
              <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-primary"></div>
            </div>
          ) : (
            <LeaseInvoice
              lease={lease}
              companyInfo={companyInfo || undefined}
              invoiceNumber={invoiceNumber}
              issueDate={issueDate}
              dueDate={dueDate}
            />
          )}
        </div>
      </DialogContent>
    </Dialog>
  );
}

// Quick action component for lease cards/lists
export interface QuickInvoiceButtonProps {
  lease: LeaseResponse;
  variant?: "default" | "outline" | "ghost";
  size?: "sm" | "default" | "lg";
  className?: string;
}

export function QuickInvoiceButton({
  lease,
  variant = "outline",
  size = "sm",
  className,
}: QuickInvoiceButtonProps) {
  return (
    <LeaseInvoiceModal
      lease={lease}
      trigger={
        <Button variant={variant} size={size} className={className}>
          <Receipt className="h-4 w-4" />
          {size !== "sm" && <span className="ml-2">Invoice</span>}
        </Button>
      }
    />
  );
}
