"use client";

import React, { useState, useEffect, useCallback } from "react";
import {
  Card,
  CardContent,
  CardDescription,
  CardHeader,
  CardTitle,
} from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { Input } from "@/components/ui/input";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select";
import {
  DropdownMenu,
  DropdownMenuContent,
  DropdownMenuItem,
  DropdownMenuLabel,
  DropdownMenuSeparator,
  DropdownMenuTrigger,
} from "@/components/ui/dropdown-menu";
import {
  AlertDialog,
  AlertDialogAction,
  AlertDialogCancel,
  AlertDialogContent,
  AlertDialogDescription,
  AlertDialogFooter,
  AlertDialogHeader,
  AlertDialogTitle,
} from "@/components/ui/alert-dialog";
import {
  Calendar,
  Clock,
  MapPin,
  Users,
  MoreVertical,
  Edit,
  Trash2,
  Eye,
  CheckCircle,
  XCircle,
  AlertCircle,
  Search,
  Filter,
  RefreshCw,
  Plus,
} from "lucide-react";
import { toast } from "sonner";
import { format } from "date-fns";
import { IEvent, EventType, EventStatus, EventPriority } from "@/types";
import { EventListSkeleton } from "./EventListSkeleton";

interface EventListProps {
  onEventEdit?: (event: IEvent) => void;
  onEventView?: (event: IEvent) => void;
  onEventCreate?: () => void;
  className?: string;
}

interface EventFilters {
  search: string;
  type: EventType | "all";
  status: EventStatus | "all";
  priority: EventPriority | "all";
  dateRange: "all" | "today" | "week" | "month";
}

const eventTypeLabels = {
  [EventType.LEASE_RENEWAL]: "Lease Renewal",
  [EventType.PROPERTY_INSPECTION]: "Property Inspection",
  [EventType.MAINTENANCE_APPOINTMENT]: "Maintenance",
  [EventType.PROPERTY_SHOWING]: "Property Showing",
  [EventType.TENANT_MEETING]: "Tenant Meeting",
  [EventType.RENT_COLLECTION]: "Rent Collection",
  [EventType.MOVE_IN]: "Move In",
  [EventType.MOVE_OUT]: "Move Out",
  [EventType.GENERAL]: "General",
};

const eventStatusColors = {
  [EventStatus.SCHEDULED]: "bg-blue-100 text-blue-800",
  [EventStatus.CONFIRMED]: "bg-green-100 text-green-800",
  [EventStatus.IN_PROGRESS]: "bg-yellow-100 text-yellow-800",
  [EventStatus.COMPLETED]: "bg-emerald-100 text-emerald-800",
  [EventStatus.CANCELLED]: "bg-red-100 text-red-800",
  [EventStatus.RESCHEDULED]: "bg-purple-100 text-purple-800",
};

const priorityColors = {
  [EventPriority.LOW]: "bg-gray-100 text-gray-800",
  [EventPriority.MEDIUM]: "bg-blue-100 text-blue-800",
  [EventPriority.HIGH]: "bg-orange-100 text-orange-800",
  [EventPriority.URGENT]: "bg-red-100 text-red-800",
};

export function EventList({
  onEventEdit,
  onEventView,
  onEventCreate,
  className,
}: EventListProps) {
  const [events, setEvents] = useState<IEvent[]>([]);
  const [loading, setLoading] = useState(true);
  const [filters, setFilters] = useState<EventFilters>({
    search: "",
    type: "all",
    status: "all",
    priority: "all",
    dateRange: "all",
  });
  const [deleteDialogOpen, setDeleteDialogOpen] = useState(false);
  const [eventToDelete, setEventToDelete] = useState<IEvent | null>(null);
  const [actionLoading, setActionLoading] = useState<string | null>(null);

  // Add refs to prevent memory leaks and duplicate calls
  const isMountedRef = React.useRef(true);
  const loadTimeoutRef = React.useRef<NodeJS.Timeout | null>(null);

  const loadEvents = useCallback(async () => {
    // Clear any existing timeout
    if (loadTimeoutRef.current) {
      clearTimeout(loadTimeoutRef.current);
    }

    // Debounce the load call
    loadTimeoutRef.current = setTimeout(async () => {
      if (!isMountedRef.current) {
        return;
      }

      try {
        setLoading(true);
        const params = new URLSearchParams();

        if (filters.type !== "all") params.append("type", filters.type);
        if (filters.status !== "all") params.append("status", filters.status);
        if (filters.priority !== "all")
          params.append("priority", filters.priority);
        if (filters.search) params.append("search", filters.search);

        // Add date range filters
        if (filters.dateRange !== "all") {
          const now = new Date();
          let startDate: Date;
          let endDate: Date = new Date(now);
          endDate.setHours(23, 59, 59, 999);

          switch (filters.dateRange) {
            case "today":
              startDate = new Date(now);
              startDate.setHours(0, 0, 0, 0);
              break;
            case "week":
              startDate = new Date(now);
              startDate.setDate(now.getDate() - now.getDay());
              startDate.setHours(0, 0, 0, 0);
              endDate.setDate(startDate.getDate() + 6);
              break;
            case "month":
              startDate = new Date(now.getFullYear(), now.getMonth(), 1);
              endDate = new Date(now.getFullYear(), now.getMonth() + 1, 0);
              endDate.setHours(23, 59, 59, 999);
              break;
            default:
              startDate = new Date(now);
              startDate.setHours(0, 0, 0, 0);
          }

          params.append("startDate", startDate.toISOString());
          params.append("endDate", endDate.toISOString());
        }

        // Add pagination to get more events
        params.append("limit", "100"); // Get up to 100 events


        const response = await fetch(
          `/api/calendar/events?${params.toString()}`
        );

        if (!isMountedRef.current) {
          return;
        }

        if (response.ok) {
          const result = await response.json();

          // Handle different response structures
          let eventsData = [];
          if (result.data && result.data.events) {
            // Paginated response structure
            eventsData = result.data.events;
          } else if (result.data && Array.isArray(result.data)) {
            // Direct array in data
            eventsData = result.data;
          } else if (Array.isArray(result)) {
            // Direct array response
            eventsData = result;
          } else if (result.events) {
            // Events property
            eventsData = result.events;
          }

          if (isMountedRef.current) {
            setEvents(Array.isArray(eventsData) ? eventsData : []);
          }
        } else {
          if (isMountedRef.current) {
            toast.error("Failed to load events");
            setEvents([]);
          }
        }
      } catch (error) {
        console.error("Error loading events:", error);
        if (isMountedRef.current) {
          toast.error("Failed to load events");
          setEvents([]);
        }
      } finally {
        if (isMountedRef.current) {
          setLoading(false);
        }
      }
    }, 250); // 250ms debounce
  }, [
    filters.search,
    filters.type,
    filters.status,
    filters.priority,
    filters.dateRange,
  ]);

  useEffect(() => {
    loadEvents();
  }, [loadEvents]);

  // Cleanup effect
  useEffect(() => {
    isMountedRef.current = true;
    return () => {
      isMountedRef.current = false;
      if (loadTimeoutRef.current) {
        clearTimeout(loadTimeoutRef.current);
      }
    };
  }, []);

  const handleStatusChange = async (
    eventId: string,
    newStatus: EventStatus
  ) => {
    try {
      setActionLoading(eventId);
      const response = await fetch(`/api/calendar/events/${eventId}`, {
        method: "PATCH",
        headers: { "Content-Type": "application/json" },
        body: JSON.stringify({ status: newStatus }),
      });

      if (response.ok) {
        toast.success("Event status updated successfully");
        loadEvents();
      } else {
        toast.error("Failed to update event status");
      }
    } catch (error) {
      console.error("Error updating event status:", error);
      toast.error("Failed to update event status");
    } finally {
      setActionLoading(null);
    }
  };

  const handleDeleteEvent = async () => {
    if (!eventToDelete) return;

    try {
      setActionLoading(eventToDelete._id.toString());
      const response = await fetch(
        `/api/calendar/events/${eventToDelete._id}`,
        {
          method: "DELETE",
        }
      );

      if (response.ok) {
        toast.success("Event deleted successfully");
        loadEvents();
        setDeleteDialogOpen(false);
        setEventToDelete(null);
      } else {
        toast.error("Failed to delete event");
      }
    } catch (error) {
      console.error("Error deleting event:", error);
      toast.error("Failed to delete event");
    } finally {
      setActionLoading(null);
    }
  };

  const filteredEvents = (Array.isArray(events) ? events : []).filter(
    (event) => {
      if (filters.search) {
        const searchLower = filters.search.toLowerCase();
        const locationText = event.location
          ? event.location.type === "physical"
            ? event.location.address || ""
            : event.location.meetingLink || ""
          : "";

        return (
          event.title.toLowerCase().includes(searchLower) ||
          event.description?.toLowerCase().includes(searchLower) ||
          locationText.toLowerCase().includes(searchLower)
        );
      }
      return true;
    }
  );

  const getStatusIcon = (status: EventStatus) => {
    switch (status) {
      case EventStatus.COMPLETED:
        return <CheckCircle className="h-4 w-4" />;
      case EventStatus.CANCELLED:
        return <XCircle className="h-4 w-4" />;
      case EventStatus.IN_PROGRESS:
        return <AlertCircle className="h-4 w-4" />;
      default:
        return <Clock className="h-4 w-4" />;
    }
  };

  if (loading) {
    return <EventListSkeleton />;
  }

  return (
    <div className={`space-y-6 ${className}`}>
      {/* Header and Filters */}
      <div className="flex flex-col gap-4">
        <div className="flex items-center justify-between">
          <div>
            <h2 className="text-2xl font-bold">Event Management</h2>
            <p className="text-muted-foreground">
              View and manage all your calendar events
            </p>
          </div>
          <div className="flex items-center gap-2">
            <Button onClick={loadEvents} variant="outline" size="sm">
              <RefreshCw className="h-4 w-4 mr-2" />
              Refresh
            </Button>
            {onEventCreate && (
              <Button onClick={onEventCreate}>
                <Plus className="h-4 w-4 mr-2" />
                New Event
              </Button>
            )}
          </div>
        </div>

        {/* Filters */}
        <div className="flex flex-wrap items-center gap-4">
          <div className="flex-1 min-w-[200px]">
            <div className="relative">
              <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 h-4 w-4 text-muted-foreground" />
              <Input
                placeholder="Search events..."
                value={filters.search}
                onChange={(e) =>
                  setFilters((prev) => ({ ...prev, search: e.target.value }))
                }
                className="pl-10"
              />
            </div>
          </div>

          <Select
            value={filters.type}
            onValueChange={(value) =>
              setFilters((prev) => ({
                ...prev,
                type: value as EventType | "all",
              }))
            }
          >
            <SelectTrigger className="w-[150px]">
              <SelectValue placeholder="Type" />
            </SelectTrigger>
            <SelectContent>
              <SelectItem value="all">All Types</SelectItem>
              {Object.entries(eventTypeLabels).map(([value, label]) => (
                <SelectItem key={value} value={value}>
                  {label}
                </SelectItem>
              ))}
            </SelectContent>
          </Select>

          <Select
            value={filters.status}
            onValueChange={(value) =>
              setFilters((prev) => ({
                ...prev,
                status: value as EventStatus | "all",
              }))
            }
          >
            <SelectTrigger className="w-[150px]">
              <SelectValue placeholder="Status" />
            </SelectTrigger>
            <SelectContent>
              <SelectItem value="all">All Status</SelectItem>
              {Object.values(EventStatus).map((status) => (
                <SelectItem key={status} value={status}>
                  {status.replace("_", " ")}
                </SelectItem>
              ))}
            </SelectContent>
          </Select>

          <Select
            value={filters.dateRange}
            onValueChange={(value) =>
              setFilters((prev) => ({ ...prev, dateRange: value as any }))
            }
          >
            <SelectTrigger className="w-[150px]">
              <SelectValue placeholder="Date Range" />
            </SelectTrigger>
            <SelectContent>
              <SelectItem value="all">All Time</SelectItem>
              <SelectItem value="today">Today</SelectItem>
              <SelectItem value="week">This Week</SelectItem>
              <SelectItem value="month">This Month</SelectItem>
            </SelectContent>
          </Select>
        </div>
      </div>

      {/* Events List */}
      <div className="space-y-4">
        {loading ? (
          <div className="flex items-center justify-center py-8">
            <RefreshCw className="h-6 w-6 animate-spin mr-2" />
            <span>Loading events...</span>
          </div>
        ) : filteredEvents.length === 0 ? (
          <Card>
            <CardContent className="text-center py-8">
              <Calendar className="h-12 w-12 mx-auto mb-4 opacity-50" />
              <p className="text-muted-foreground">No events found</p>
              <p className="text-sm text-muted-foreground">
                Try adjusting your filters or create a new event
              </p>
            </CardContent>
          </Card>
        ) : (
          <div className="grid gap-4">
            {filteredEvents.map((event) => (
              <Card
                key={event._id.toString()}
                className="hover:shadow-md transition-shadow"
              >
                <CardContent className="p-6">
                  <div className="flex items-start justify-between">
                    <div className="flex-1 space-y-3">
                      {/* Event Header */}
                      <div className="flex items-start gap-3">
                        <div className="flex-1">
                          <div className="flex items-center gap-2 mb-1">
                            <h3 className="font-semibold text-lg">
                              {event.title}
                            </h3>
                            <Badge className={eventStatusColors[event.status]}>
                              {getStatusIcon(event.status)}
                              <span className="ml-1">
                                {event.status.replace("_", " ")}
                              </span>
                            </Badge>
                            <Badge
                              variant="outline"
                              className={priorityColors[event.priority]}
                            >
                              {event.priority}
                            </Badge>
                          </div>
                          <p className="text-sm text-muted-foreground mb-2">
                            {eventTypeLabels[event.type]}
                          </p>
                          {event.description && (
                            <p className="text-sm text-muted-foreground line-clamp-2">
                              {event.description}
                            </p>
                          )}
                        </div>
                      </div>

                      {/* Event Details */}
                      <div className="flex flex-wrap items-center gap-4 text-sm text-muted-foreground">
                        <div className="flex items-center gap-1">
                          <Clock className="h-4 w-4" />
                          <span>
                            {format(
                              new Date(event.startDate),
                              "MMM dd, yyyy 'at' h:mm a"
                            )}
                            {event.endDate && (
                              <span>
                                {" "}
                                - {format(new Date(event.endDate), "h:mm a")}
                              </span>
                            )}
                          </span>
                        </div>

                        {event.location && (
                          <div className="flex items-center gap-1">
                            <MapPin className="h-4 w-4" />
                            <span>
                              {event.location.type === "physical"
                                ? event.location.address
                                : `Online (${
                                    event.location.platform?.replace(
                                      "_",
                                      " "
                                    ) || "Meeting"
                                  })`}
                            </span>
                          </div>
                        )}

                        {event.attendees && event.attendees.length > 0 && (
                          <div className="flex items-center gap-1">
                            <Users className="h-4 w-4" />
                            <span>{event.attendees.length} attendees</span>
                          </div>
                        )}
                      </div>
                    </div>

                    {/* Actions */}
                    <DropdownMenu>
                      <DropdownMenuTrigger asChild>
                        <Button
                          variant="ghost"
                          size="sm"
                          disabled={actionLoading === event._id.toString()}
                        >
                          <MoreVertical className="h-4 w-4" />
                        </Button>
                      </DropdownMenuTrigger>
                      <DropdownMenuContent align="end">
                        <DropdownMenuLabel>Actions</DropdownMenuLabel>
                        <DropdownMenuSeparator />

                        {onEventView && (
                          <DropdownMenuItem onClick={() => onEventView(event)}>
                            <Eye className="h-4 w-4 mr-2" />
                            View Details
                          </DropdownMenuItem>
                        )}

                        {onEventEdit && (
                          <DropdownMenuItem onClick={() => onEventEdit(event)}>
                            <Edit className="h-4 w-4 mr-2" />
                            Edit Event
                          </DropdownMenuItem>
                        )}

                        <DropdownMenuSeparator />

                        {/* Status Actions */}
                        {event.status === EventStatus.SCHEDULED && (
                          <DropdownMenuItem
                            onClick={() =>
                              handleStatusChange(
                                event._id.toString(),
                                EventStatus.CONFIRMED
                              )
                            }
                          >
                            <CheckCircle className="h-4 w-4 mr-2" />
                            Confirm
                          </DropdownMenuItem>
                        )}

                        {[
                          EventStatus.SCHEDULED,
                          EventStatus.CONFIRMED,
                        ].includes(event.status) && (
                          <DropdownMenuItem
                            onClick={() =>
                              handleStatusChange(
                                event._id.toString(),
                                EventStatus.IN_PROGRESS
                              )
                            }
                          >
                            <AlertCircle className="h-4 w-4 mr-2" />
                            Start Event
                          </DropdownMenuItem>
                        )}

                        {event.status === EventStatus.IN_PROGRESS && (
                          <DropdownMenuItem
                            onClick={() =>
                              handleStatusChange(
                                event._id.toString(),
                                EventStatus.COMPLETED
                              )
                            }
                          >
                            <CheckCircle className="h-4 w-4 mr-2" />
                            Complete
                          </DropdownMenuItem>
                        )}

                        {![
                          EventStatus.COMPLETED,
                          EventStatus.CANCELLED,
                        ].includes(event.status) && (
                          <DropdownMenuItem
                            onClick={() =>
                              handleStatusChange(
                                event._id.toString(),
                                EventStatus.CANCELLED
                              )
                            }
                            className="text-red-600"
                          >
                            <XCircle className="h-4 w-4 mr-2" />
                            Cancel
                          </DropdownMenuItem>
                        )}

                        <DropdownMenuSeparator />

                        <DropdownMenuItem
                          onClick={() => {
                            setEventToDelete(event);
                            setDeleteDialogOpen(true);
                          }}
                          className="text-red-600"
                        >
                          <Trash2 className="h-4 w-4 mr-2" />
                          Delete
                        </DropdownMenuItem>
                      </DropdownMenuContent>
                    </DropdownMenu>
                  </div>
                </CardContent>
              </Card>
            ))}
          </div>
        )}
      </div>

      {/* Delete Confirmation Dialog */}
      <AlertDialog open={deleteDialogOpen} onOpenChange={setDeleteDialogOpen}>
        <AlertDialogContent>
          <AlertDialogHeader>
            <AlertDialogTitle>Delete Event</AlertDialogTitle>
            <AlertDialogDescription>
              Are you sure you want to delete "{eventToDelete?.title}"? This
              action cannot be undone.
            </AlertDialogDescription>
          </AlertDialogHeader>
          <AlertDialogFooter>
            <AlertDialogCancel>Cancel</AlertDialogCancel>
            <AlertDialogAction
              onClick={handleDeleteEvent}
              className="bg-red-600 hover:bg-red-700"
              disabled={actionLoading !== null}
            >
              {actionLoading ? "Deleting..." : "Delete"}
            </AlertDialogAction>
          </AlertDialogFooter>
        </AlertDialogContent>
      </AlertDialog>
    </div>
  );
}
