"use client";

import { z } from "zod";
import { useForm } from "react-hook-form";
import { Input } from "@/components/ui/input";
import { Button } from "@/components/ui/button";
import { Textarea } from "@/components/ui/textarea";
import { zodResolver } from "@hookform/resolvers/zod";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select";
import { Switch } from "@/components/ui/switch";
import { DatePicker } from "@/components/ui/date-picker";
import {
  Form,
  FormControl,
  FormDescription,
  FormField,
  FormItem,
  FormLabel,
  FormMessage,
} from "@/components/ui/form";
import {
  EventType,
  EventPriority,
  RecurrenceType,
  LocationType,
  OnlinePlatform,
} from "@/types";
import { toast } from "sonner";

const eventFormSchema = z.object({
  title: z.string().min(1, "Title is required").max(200, "Title too long"),
  description: z.string().max(2000, "Description too long").optional().or(z.literal("")),
  type: z.nativeEnum(EventType),
  priority: z.nativeEnum(EventPriority),
  startDate: z.date({ required_error: "Start date is required" }),
  startTime: z.string().min(1, "Start time is required"),
  endDate: z.date({ required_error: "End date is required" }),
  endTime: z.string().min(1, "End time is required"),
  allDay: z.boolean(),
  locationType: z.nativeEnum(LocationType).optional(),
  locationAddress: z.string().optional().or(z.literal("")),
  onlinePlatform: z.nativeEnum(OnlinePlatform).optional(),
  meetingLink: z.string().url().optional().or(z.literal("")),
  meetingId: z.string().optional().or(z.literal("")),
  passcode: z.string().optional().or(z.literal("")),
  propertyId: z.string().optional().or(z.literal("")),
  unitNumber: z.string().optional().or(z.literal("")),
  tenantId: z.string().optional().or(z.literal("")),
  leaseId: z.string().optional().or(z.literal("")),
  maintenanceRequestId: z.string().optional().or(z.literal("")),
  attendeeEmails: z.string().optional().or(z.literal("")),
  reminderMinutes: z.array(z.number()),
  notes: z.string().optional().or(z.literal("")),
  isRecurring: z.boolean(),
  recurrenceType: z.nativeEnum(RecurrenceType).optional(),
  recurrenceInterval: z.number().min(1).optional(),
  recurrenceEndDate: z.date().optional().nullable(),
  recurrenceOccurrences: z.number().min(1).optional(),
});

type EventFormData = z.infer<typeof eventFormSchema>;

interface EventSubmitData extends Omit<EventFormData, 'startDate' | 'endDate' | 'locationType' | 'locationAddress' | 'onlinePlatform' | 'meetingLink' | 'meetingId' | 'passcode'> {
  startDate: Date;
  endDate: Date;
  location?: {
    type: LocationType;
    address?: string;
    platform?: OnlinePlatform;
    meetingLink?: string;
    meetingId?: string;
    passcode?: string;
  };
}

interface EventFormProps {
  onSubmit: (data: EventSubmitData) => Promise<void>;
  onCancel: () => void;
  initialData?: Partial<EventFormData>;
  loading?: boolean;
}

export default function EventForm({
  onSubmit,
  onCancel,
  initialData,
  loading = false,
}: EventFormProps) {

  const form = useForm<EventFormData>({
    resolver: zodResolver(eventFormSchema),
    defaultValues: {
      title: "",
      description: "",
      type: EventType.GENERAL,
      priority: EventPriority.MEDIUM,
      allDay: false,
      locationType: LocationType.PHYSICAL,
      locationAddress: "",
      onlinePlatform: OnlinePlatform.ZOOM,
      meetingLink: "",
      meetingId: "",
      passcode: "",
      propertyId: undefined,
      unitNumber: "",
      tenantId: undefined,
      leaseId: undefined,
      maintenanceRequestId: undefined,
      attendeeEmails: "",
      reminderMinutes: [15],
      notes: "",
      isRecurring: false,
      recurrenceType: RecurrenceType.NONE,
      recurrenceInterval: 1,
      startTime: "09:00",
      endTime: "10:00",
      ...initialData,
    },
  });

  const handleSubmit = async (data: EventFormData) => {
    try {
      // Combine date and time
      const startDateTime = new Date(data.startDate);
      const endDateTime = new Date(data.endDate);

      if (!data.allDay) {
        const [startHour, startMinute] = data.startTime.split(":").map(Number);
        const [endHour, endMinute] = data.endTime.split(":").map(Number);

        startDateTime.setHours(startHour, startMinute, 0, 0);
        endDateTime.setHours(endHour, endMinute, 0, 0);
      }

      // Validate end time is after start time
      if (endDateTime <= startDateTime) {
        toast.error("End time must be after start time");
        return;
      }

      // Format location data
      const location = data.locationType
        ? {
            type: data.locationType,
            ...(data.locationType === LocationType.PHYSICAL && {
              address: data.locationAddress,
            }),
            ...(data.locationType === LocationType.ONLINE && {
              platform: data.onlinePlatform,
              meetingLink: data.meetingLink,
              meetingId: data.meetingId,
              passcode: data.passcode,
            }),
          }
        : undefined;

      // Destructure to remove location-specific fields from the payload
      // These destructured vars are intentionally unused - they're being excluded from the final payload
      /* eslint-disable @typescript-eslint/no-unused-vars */
      const {
        locationType,
        locationAddress,
        onlinePlatform,
        meetingLink,
        meetingId,
        passcode,
        propertyId,
        tenantId,
        leaseId,
        maintenanceRequestId,
        ...rest
      } = data;
      /* eslint-enable @typescript-eslint/no-unused-vars */

      const formattedData: EventSubmitData = {
        ...rest,
        startDate: startDateTime,
        endDate: endDateTime,
        location,
        // Only include non-empty ObjectId fields
        ...(propertyId && propertyId !== "" && { propertyId }),
        ...(tenantId && tenantId !== "" && { tenantId }),
        ...(leaseId && leaseId !== "" && { leaseId }),
        ...(maintenanceRequestId && maintenanceRequestId !== "" && { maintenanceRequestId }),
      };

      await onSubmit(formattedData);
    } catch (error) {
      console.error("Error submitting form:", error);
      toast.error("Failed to create event");
    }
  };

  const getEventTypeLabel = (type: EventType) => {
    return type.replace(/_/g, " ").replace(/\b\w/g, (l) => l.toUpperCase());
  };

  const getPriorityLabel = (priority: EventPriority) => {
    return priority.charAt(0).toUpperCase() + priority.slice(1);
  };

  return (
    <div className="h-full flex flex-col">
      <Form {...form}>
        <form
          onSubmit={form.handleSubmit(handleSubmit)}
          className="h-full flex flex-col"
        >
          <div className="flex-1 overflow-y-auto pr-2">
            <div className="grid grid-cols-1 lg:grid-cols-2 gap-8 p-1">
              {/* Left Column */}
              <div className="space-y-5">
                {/* Title */}
                <FormField
                  control={form.control}
                  name="title"
                  render={({ field }) => (
                    <FormItem>
                      <FormLabel>Event Title</FormLabel>
                      <FormControl>
                        <Input placeholder="Enter event title" {...field} />
                      </FormControl>
                      <FormMessage />
                    </FormItem>
                  )}
                />

                {/* Type and Priority */}
                <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                  <FormField
                    control={form.control}
                    name="type"
                    render={({ field }) => (
                      <FormItem>
                        <FormLabel>Type</FormLabel>
                        <Select
                          onValueChange={field.onChange}
                          defaultValue={field.value}
                        >
                          <FormControl>
                            <SelectTrigger>
                              <SelectValue placeholder="Select type" />
                            </SelectTrigger>
                          </FormControl>
                          <SelectContent>
                            {Object.values(EventType).map((type) => (
                              <SelectItem key={type} value={type}>
                                {getEventTypeLabel(type)}
                              </SelectItem>
                            ))}
                          </SelectContent>
                        </Select>
                        <FormMessage />
                      </FormItem>
                    )}
                  />

                  <FormField
                    control={form.control}
                    name="priority"
                    render={({ field }) => (
                      <FormItem>
                        <FormLabel>Priority</FormLabel>
                        <Select
                          onValueChange={field.onChange}
                          defaultValue={field.value}
                        >
                          <FormControl>
                            <SelectTrigger>
                              <SelectValue placeholder="Select priority" />
                            </SelectTrigger>
                          </FormControl>
                          <SelectContent>
                            {Object.values(EventPriority).map((priority) => (
                              <SelectItem key={priority} value={priority}>
                                {getPriorityLabel(priority)}
                              </SelectItem>
                            ))}
                          </SelectContent>
                        </Select>
                        <FormMessage />
                      </FormItem>
                    )}
                  />
                </div>

                {/* All Day Toggle */}
                <FormField
                  control={form.control}
                  name="allDay"
                  render={({ field }) => (
                    <FormItem className="flex flex-row items-center justify-between rounded-lg border p-3">
                      <div className="space-y-0.5">
                        <FormLabel className="text-sm font-medium">
                          All Day Event
                        </FormLabel>
                        <FormDescription className="text-xs">
                          This event lasts all day
                        </FormDescription>
                      </div>
                      <FormControl>
                        <Switch
                          checked={field.value}
                          onCheckedChange={field.onChange}
                        />
                      </FormControl>
                    </FormItem>
                  )}
                />

                {/* Date Fields */}
                <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                  <FormField
                    control={form.control}
                    name="startDate"
                    render={({ field }) => (
                      <FormItem>
                        <FormLabel>Start Date</FormLabel>
                        <FormControl>
                          <DatePicker
                            date={field.value}
                            onSelect={field.onChange}
                            placeholder="Pick start date"
                          />
                        </FormControl>
                        <FormMessage />
                      </FormItem>
                    )}
                  />

                  <FormField
                    control={form.control}
                    name="endDate"
                    render={({ field }) => (
                      <FormItem>
                        <FormLabel>End Date</FormLabel>
                        <FormControl>
                          <DatePicker
                            date={field.value}
                            onSelect={field.onChange}
                            placeholder="Pick end date"
                          />
                        </FormControl>
                        <FormMessage />
                      </FormItem>
                    )}
                  />
                </div>

                {/* Time Fields (only show if not all day) */}
                {!form.watch("allDay") && (
                  <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                    <FormField
                      control={form.control}
                      name="startTime"
                      render={({ field }) => (
                        <FormItem>
                          <FormLabel>Start Time</FormLabel>
                          <FormControl>
                            <Input type="time" {...field} />
                          </FormControl>
                          <FormMessage />
                        </FormItem>
                      )}
                    />

                    <FormField
                      control={form.control}
                      name="endTime"
                      render={({ field }) => (
                        <FormItem>
                          <FormLabel>End Time</FormLabel>
                          <FormControl>
                            <Input type="time" {...field} />
                          </FormControl>
                          <FormMessage />
                        </FormItem>
                      )}
                    />
                  </div>
                )}
              </div>

              {/* Right Column */}
              <div className="space-y-5">
                {/* Description */}
                <FormField
                  control={form.control}
                  name="description"
                  render={({ field }) => (
                    <FormItem>
                      <FormLabel>Description</FormLabel>
                      <FormControl>
                        <Textarea
                          placeholder="Event description (optional)"
                          className="min-h-[140px] resize-none"
                          {...field}
                        />
                      </FormControl>
                      <FormMessage />
                    </FormItem>
                  )}
                />

                {/* Location Type */}
                <FormField
                  control={form.control}
                  name="locationType"
                  render={({ field }) => (
                    <FormItem>
                      <FormLabel>Location Type</FormLabel>
                      <Select
                        onValueChange={field.onChange}
                        defaultValue={field.value}
                      >
                        <FormControl>
                          <SelectTrigger>
                            <SelectValue placeholder="Select location type" />
                          </SelectTrigger>
                        </FormControl>
                        <SelectContent>
                          <SelectItem value={LocationType.PHYSICAL}>
                            Physical Location
                          </SelectItem>
                          <SelectItem value={LocationType.ONLINE}>
                            Online Meeting
                          </SelectItem>
                        </SelectContent>
                      </Select>
                      <FormMessage />
                    </FormItem>
                  )}
                />

                {/* Conditional Location Fields */}
                {form.watch("locationType") === LocationType.PHYSICAL && (
                  <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                    <FormField
                      control={form.control}
                      name="locationAddress"
                      render={({ field }) => (
                        <FormItem>
                          <FormLabel>Address</FormLabel>
                          <FormControl>
                            <Input placeholder="Event address" {...field} />
                          </FormControl>
                          <FormMessage />
                        </FormItem>
                      )}
                    />

                    <FormField
                      control={form.control}
                      name="unitNumber"
                      render={({ field }) => (
                        <FormItem>
                          <FormLabel>Unit Number (Optional)</FormLabel>
                          <FormControl>
                            <Input placeholder="Unit number" {...field} />
                          </FormControl>
                          <FormMessage />
                        </FormItem>
                      )}
                    />
                  </div>
                )}

                {form.watch("locationType") === LocationType.ONLINE && (
                  <div className="space-y-4">
                    <FormField
                      control={form.control}
                      name="onlinePlatform"
                      render={({ field }) => (
                        <FormItem>
                          <FormLabel>Platform</FormLabel>
                          <Select
                            onValueChange={field.onChange}
                            defaultValue={field.value}
                          >
                            <FormControl>
                              <SelectTrigger>
                                <SelectValue placeholder="Select platform" />
                              </SelectTrigger>
                            </FormControl>
                            <SelectContent>
                              <SelectItem value={OnlinePlatform.ZOOM}>
                                Zoom
                              </SelectItem>
                              <SelectItem value={OnlinePlatform.GOOGLE_MEET}>
                                Google Meet
                              </SelectItem>
                              <SelectItem
                                value={OnlinePlatform.MICROSOFT_TEAMS}
                              >
                                Microsoft Teams
                              </SelectItem>
                              <SelectItem value={OnlinePlatform.WEBEX}>
                                Webex
                              </SelectItem>
                              <SelectItem value={OnlinePlatform.OTHER}>
                                Other
                              </SelectItem>
                            </SelectContent>
                          </Select>
                          <FormMessage />
                        </FormItem>
                      )}
                    />

                    <FormField
                      control={form.control}
                      name="meetingLink"
                      render={({ field }) => {
                        const platform = form.watch("onlinePlatform");
                        const getPlaceholder = () => {
                          switch (platform) {
                            case OnlinePlatform.ZOOM:
                              return "https://zoom.us/j/123456789";
                            case OnlinePlatform.GOOGLE_MEET:
                              return "https://meet.google.com/abc-defg-hij";
                            case OnlinePlatform.MICROSOFT_TEAMS:
                              return "https://teams.microsoft.com/l/meetup-join/...";
                            case OnlinePlatform.WEBEX:
                              return "https://company.webex.com/meet/...";
                            default:
                              return "Enter meeting link";
                          }
                        };

                        return (
                          <FormItem>
                            <FormLabel>Meeting Link</FormLabel>
                            <FormControl>
                              <Input
                                placeholder={getPlaceholder()}
                                type="url"
                                {...field}
                              />
                            </FormControl>
                            <FormMessage />
                          </FormItem>
                        );
                      }}
                    />

                    <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                      <FormField
                        control={form.control}
                        name="meetingId"
                        render={({ field }) => (
                          <FormItem>
                            <FormLabel>Meeting ID (Optional)</FormLabel>
                            <FormControl>
                              <Input placeholder="Meeting ID" {...field} />
                            </FormControl>
                            <FormMessage />
                          </FormItem>
                        )}
                      />

                      <FormField
                        control={form.control}
                        name="passcode"
                        render={({ field }) => (
                          <FormItem>
                            <FormLabel>Passcode (Optional)</FormLabel>
                            <FormControl>
                              <Input
                                placeholder="Meeting passcode"
                                {...field}
                              />
                            </FormControl>
                            <FormMessage />
                          </FormItem>
                        )}
                      />
                    </div>
                  </div>
                )}

                {/* Attendee Emails */}
                <FormField
                  control={form.control}
                  name="attendeeEmails"
                  render={({ field }) => (
                    <FormItem>
                      <FormLabel>Attendee Emails</FormLabel>
                      <FormControl>
                        <Input
                          placeholder="Enter email addresses separated by commas"
                          {...field}
                        />
                      </FormControl>
                      <FormDescription className="text-xs">
                        Separate multiple email addresses with commas
                      </FormDescription>
                      <FormMessage />
                    </FormItem>
                  )}
                />
              </div>
            </div>
          </div>

          {/* Form Actions */}
          <div className="flex justify-end gap-4 pt-6 border-t mt-6">
            <Button type="button" variant="outline" onClick={onCancel}>
              Cancel
            </Button>
            <Button type="submit" disabled={loading}>
              {loading ? "Creating..." : "Create Event"}
            </Button>
          </div>
        </form>
      </Form>
    </div>
  );
}

export { EventForm };
