"use client";

import { useState, useEffect, useCallback } from "react";
import { useSession } from "next-auth/react";
import Link from "next/link";
import {
  Card,
  CardContent,
  CardDescription,
  CardHeader,
  CardTitle,
} from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { Input } from "@/components/ui/input";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select";
import {
  Table,
  TableBody,
  TableCell,
  TableHead,
  TableHeader,
  TableRow,
} from "@/components/ui/table";
import {
  DropdownMenu,
  DropdownMenuContent,
  DropdownMenuItem,
  DropdownMenuTrigger,
} from "@/components/ui/dropdown-menu";
import { Alert, AlertDescription } from "@/components/ui/alert";
import {
  FileText,
  Search,
  MoreHorizontal,
  Eye,
  Check,
  X,
  RefreshCw,
  AlertCircle,
} from "lucide-react";
import { toast } from "sonner";

const statusOptions = [
  { value: "all", label: "All Status" },
  { value: "pending", label: "Pending" },
  { value: "approved", label: "Approved" },
  { value: "rejected", label: "Rejected" },
];

const mapStatusToFilter = (
  status: string
): "pending" | "approved" | "rejected" => {
  const normalized = status?.toLowerCase?.() ?? "";

  if (
    [
      "submitted",
      "under_review",
      "screening_in_progress",
      "draft",
      "application_submitted",
    ].includes(normalized)
  ) {
    return "pending";
  }

  if (["approved"].includes(normalized)) {
    return "approved";
  }

  if (["rejected", "withdrawn", "terminated"].includes(normalized)) {
    return "rejected";
  }

  return "pending";
};

const matchesStatusFilter = (status: string, filter: string) => {
  if (filter === "all") return true;
  return mapStatusToFilter(status) === filter;
};

const formatLocation = (address?: {
  street?: string;
  city?: string;
  state?: string;
}): string | undefined => {
  if (!address) return undefined;
  const parts = [address.city, address.state].filter(Boolean);
  return parts.length ? parts.join(", ") : undefined;
};

const normalizeDate = (value?: string | Date | null): string | undefined => {
  if (!value) return undefined;
  const date = typeof value === "string" ? new Date(value) : value;
  if (!date || isNaN(date.getTime())) return undefined;
  return date.toISOString();
};

interface ApplicationRow {
  id: string;
  source: "application" | "tenant";
  applicantName: string;
  applicantEmail: string;
  applicantPhone?: string;
  status: string;
  submittedAt?: string;
  reviewedBy?: string;
  reviewDate?: string;
  propertyName?: string;
  propertyLocation?: string;
}

export default function TenantApplicationsPage() {
  const { data: session } = useSession();
  const [applications, setApplications] = useState<ApplicationRow[]>([]);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState<string | null>(null);
  const [searchTerm, setSearchTerm] = useState("");
  const [statusFilter, setStatusFilter] = useState<string>("all");

  const mapApplicationDocuments = useCallback(
    (data: any[]): ApplicationRow[] => {
      if (!Array.isArray(data)) return [];

      return data.map((item) => {
        const personalInfo = item.personalInfo || {};
        const applicant = item.applicantId || {};
        const property = item.propertyId || {};
        const propertyAddress = property.address || {};
        const reviewer = item.reviewedBy || {};

        const applicantName = [
          personalInfo.firstName || applicant.firstName,
          personalInfo.lastName || applicant.lastName,
        ]
          .filter(Boolean)
          .join(" ");

        return {
          id: item._id,
          source: "application",
          applicantName:
            applicantName || personalInfo.fullName || "Unknown Applicant",
          applicantEmail: personalInfo.email || applicant.email || "",
          applicantPhone: personalInfo.phone || applicant.phone,
          status: item.status || "submitted",
          submittedAt:
            normalizeDate(item.submittedAt) ||
            normalizeDate(item.applicationDate) ||
            normalizeDate(item.createdAt),
          reviewedBy:
            reviewer?.firstName || reviewer?.lastName
              ? [reviewer.firstName, reviewer.lastName]
                  .filter(Boolean)
                  .join(" ")
              : undefined,
          reviewDate: normalizeDate(item.reviewedAt || item.reviewDate),
          propertyName: property.name,
          propertyLocation: formatLocation(propertyAddress),
        };
      });
    },
    []
  );

  const mapTenantApplications = useCallback((data: any[]): ApplicationRow[] => {
    if (!Array.isArray(data)) return [];

    return data.map((tenant) => {
      const reviewer = tenant.reviewedBy || {};
      return {
        id: tenant._id,
        source: "tenant",
        applicantName:
          [tenant.firstName, tenant.lastName].filter(Boolean).join(" ") ||
          tenant.fullName ||
          "Unknown Applicant",
        applicantEmail: tenant.email || "",
        applicantPhone: tenant.phone,
        status: tenant.tenantStatus || "application_submitted",
        submittedAt: normalizeDate(tenant.applicationDate || tenant.createdAt),
        reviewedBy:
          reviewer?.firstName || reviewer?.lastName
            ? [reviewer.firstName, reviewer.lastName].filter(Boolean).join(" ")
            : undefined,
        reviewDate: normalizeDate(tenant.lastStatusUpdate),
        propertyName: tenant.currentLeaseId?.propertyId?.name,
        propertyLocation: formatLocation(
          tenant.currentLeaseId?.propertyId?.address
        ),
      };
    });
  }, []);

  const fetchApplications = useCallback(async () => {
    if (!session) return;

    try {
      setLoading(true);
      setError(null);

      const params = new URLSearchParams();
      if (searchTerm) {
        params.append("search", searchTerm);
      }

      const response = await fetch(`/api/applications?${params.toString()}`);
      let rows: ApplicationRow[] = [];

      if (response.ok) {
        const payload = await response.json();
        rows = mapApplicationDocuments(payload.data || []);
      } else if (response.status !== 404) {
        throw new Error("Failed to fetch applications");
      }

      if (rows.length === 0) {
        const tenantParams = new URLSearchParams();
        const tenantStatusQuery =
          statusFilter === "approved"
            ? "approved"
            : statusFilter === "rejected"
            ? "terminated"
            : "pending";
        tenantParams.append("status", tenantStatusQuery);
        if (searchTerm) {
          tenantParams.append("search", searchTerm);
        }

        const tenantResponse = await fetch(
          `/api/tenants?${tenantParams.toString()}`
        );

        if (tenantResponse.ok) {
          const tenantPayload = await tenantResponse.json();
          rows = mapTenantApplications(tenantPayload.data || []);
        }
      }

      const filteredRows = rows.filter((row) =>
        matchesStatusFilter(row.status, statusFilter)
      );

      setApplications(filteredRows);
    } catch (err) {
      const message =
        err instanceof Error ? err.message : "Failed to fetch applications";
      setError(message);
      toast.error(message);
    } finally {
      setLoading(false);
    }
  }, [
    session,
    searchTerm,
    statusFilter,
    mapApplicationDocuments,
    mapTenantApplications,
  ]);

  useEffect(() => {
    fetchApplications();
  }, [fetchApplications]);

  const handleStatusChange = useCallback(
    async (application: ApplicationRow, action: "approve" | "reject") => {
      try {
        if (application.source === "application") {
          const response = await fetch(
            `/api/applications/${application.id}/${action}`,
            {
              method: "POST",
              headers: {
                "Content-Type": "application/json",
              },
              body: JSON.stringify({
                notes: `Application ${action}d by ${session?.user?.name}`,
              }),
            }
          );

          if (!response.ok) {
            throw new Error(`Failed to ${action} application`);
          }
        } else {
          const newStatus = action === "approve" ? "approved" : "terminated";
          const response = await fetch(
            `/api/tenants/${application.id}/status`,
            {
              method: "POST",
              headers: {
                "Content-Type": "application/json",
              },
              body: JSON.stringify({
                newStatus,
                reason: `Application ${action}d via dashboard`,
              }),
            }
          );

          if (!response.ok) {
            throw new Error(`Failed to ${action} application`);
          }
        }

        toast.success(`Application ${action}d successfully`);
        fetchApplications();
      } catch (err) {
        const message =
          err instanceof Error
            ? err.message
            : `Failed to ${action} application`;
        toast.error(message);
      }
    },
    [fetchApplications, session?.user?.name]
  );

  const getStatusBadge = (status: string) => {
    const normalized = status?.toLowerCase?.() ?? "";

    if (normalized === "approved") {
      return (
        <Badge className="bg-green-100 text-green-800 border-green-200">
          Approved
        </Badge>
      );
    }

    if (["rejected", "withdrawn", "terminated"].includes(normalized)) {
      return (
        <Badge className="bg-red-100 text-red-800 border-red-200">
          Rejected
        </Badge>
      );
    }

    if (["under_review"].includes(normalized)) {
      return (
        <Badge className="bg-blue-100 text-blue-800 border-blue-200">
          Under Review
        </Badge>
      );
    }

    return (
      <Badge className="bg-yellow-100 text-yellow-800 border-yellow-200">
        Pending
      </Badge>
    );
  };

  const formatDate = (value?: string) => {
    if (!value) return "N/A";
    const date = new Date(value);
    if (isNaN(date.getTime())) return "N/A";
    return date.toLocaleDateString();
  };

  if (!session) {
    return (
      <div className="flex items-center justify-center min-h-screen">
        <div className="text-center">
          <AlertCircle className="h-12 w-12 text-red-500 mx-auto mb-4" />
          <h2 className="text-xl font-semibold mb-2">Access Denied</h2>
          <p className="text-gray-600">Please sign in to view applications.</p>
        </div>
      </div>
    );
  }

  return (
    <div className="space-y-6">
      <div className="flex items-center justify-between">
        <div>
          <h1 className="text-3xl font-bold tracking-tight">
            Tenant Applications
          </h1>
          <p className="text-muted-foreground">
            Review and manage tenant applications
          </p>
        </div>
        <Button
          variant="outline"
          size="sm"
          onClick={fetchApplications}
          disabled={loading}
        >
          <RefreshCw
            className={`h-4 w-4 mr-2 ${loading ? "animate-spin" : ""}`}
          />
          Refresh
        </Button>
      </div>

      <Card>
        <CardHeader>
          <CardTitle className="text-lg">Filters</CardTitle>
        </CardHeader>
        <CardContent>
          <div className="flex items-center space-x-4">
            <div className="flex-1">
              <div className="relative">
                <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 text-gray-400 h-4 w-4" />
                <Input
                  placeholder="Search applications..."
                  value={searchTerm}
                  onChange={(e) => setSearchTerm(e.target.value)}
                  className="pl-10"
                />
              </div>
            </div>
            <Select value={statusFilter} onValueChange={setStatusFilter}>
              <SelectTrigger className="w-48">
                <SelectValue placeholder="Filter by status" />
              </SelectTrigger>
              <SelectContent>
                {statusOptions.map((option) => (
                  <SelectItem key={option.value} value={option.value}>
                    {option.label}
                  </SelectItem>
                ))}
              </SelectContent>
            </Select>
          </div>
        </CardContent>
      </Card>

      <Card>
        <CardHeader>
          <CardTitle>Applications ({applications.length})</CardTitle>
          <CardDescription>
            A list of all tenant applications and their current status
          </CardDescription>
        </CardHeader>
        <CardContent>
          {error && (
            <Alert className="mb-4">
              <AlertCircle className="h-4 w-4" />
              <AlertDescription>{error}</AlertDescription>
            </Alert>
          )}

          {loading ? (
            <div className="flex items-center justify-center py-8">
              <RefreshCw className="h-6 w-6 animate-spin mr-2" />
              <span>Loading applications...</span>
            </div>
          ) : applications.length === 0 ? (
            <div className="text-center py-8">
              <FileText className="h-12 w-12 text-gray-400 mx-auto mb-4" />
              <h3 className="text-lg font-medium text-gray-900 mb-2">
                No applications found
              </h3>
              <p className="text-gray-600">
                {searchTerm || statusFilter !== "all"
                  ? "No applications match your current filters."
                  : "There are no tenant applications yet."}
              </p>
            </div>
          ) : (
            <div className="overflow-x-auto">
              <Table>
                <TableHeader>
                  <TableRow>
                    <TableHead>Applicant</TableHead>
                    <TableHead>Property</TableHead>
                    <TableHead>Status</TableHead>
                    <TableHead>Application Date</TableHead>
                    <TableHead>Reviewed By</TableHead>
                    <TableHead className="text-right">Actions</TableHead>
                  </TableRow>
                </TableHeader>
                <TableBody>
                  {applications.map((application) => (
                    <TableRow key={application.id}>
                      <TableCell>
                        <div>
                          <div className="font-medium">
                            {application.applicantName}
                          </div>
                          <div className="text-sm text-gray-600">
                            {application.applicantEmail}
                          </div>
                          {application.applicantPhone && (
                            <div className="text-xs text-gray-500">
                              {application.applicantPhone}
                            </div>
                          )}
                        </div>
                      </TableCell>
                      <TableCell>
                        <div>
                          <div className="font-medium">
                            {application.propertyName || "Not specified"}
                          </div>
                          <div className="text-sm text-gray-600">
                            {application.propertyLocation || "N/A"}
                          </div>
                        </div>
                      </TableCell>
                      <TableCell>
                        {getStatusBadge(application.status)}
                      </TableCell>
                      <TableCell>
                        {formatDate(application.submittedAt)}
                      </TableCell>
                      <TableCell>
                        {application.reviewedBy ? (
                          <div>
                            <div className="text-sm">
                              {application.reviewedBy}
                            </div>
                            <div className="text-xs text-gray-600">
                              {formatDate(application.reviewDate)}
                            </div>
                          </div>
                        ) : (
                          <span className="text-gray-400">Not reviewed</span>
                        )}
                      </TableCell>
                      <TableCell className="text-right">
                        <DropdownMenu>
                          <DropdownMenuTrigger asChild>
                            <Button variant="ghost" className="h-8 w-8 p-0">
                              <MoreHorizontal className="h-4 w-4" />
                            </Button>
                          </DropdownMenuTrigger>
                          <DropdownMenuContent align="end">
                            <DropdownMenuItem asChild>
                              <Link
                                href={
                                  application.source === "application"
                                    ? `/dashboard/applications/${application.id}`
                                    : `/dashboard/tenants/${application.id}`
                                }
                              >
                                <Eye className="h-4 w-4 mr-2" />
                                View Details
                              </Link>
                            </DropdownMenuItem>
                            {mapStatusToFilter(application.status) ===
                              "pending" && (
                              <>
                                <DropdownMenuItem
                                  onClick={() =>
                                    handleStatusChange(application, "approve")
                                  }
                                >
                                  <Check className="h-4 w-4 mr-2" />
                                  Approve
                                </DropdownMenuItem>
                                <DropdownMenuItem
                                  onClick={() =>
                                    handleStatusChange(application, "reject")
                                  }
                                >
                                  <X className="h-4 w-4 mr-2" />
                                  Reject
                                </DropdownMenuItem>
                              </>
                            )}
                          </DropdownMenuContent>
                        </DropdownMenu>
                      </TableCell>
                    </TableRow>
                  ))}
                </TableBody>
              </Table>
            </div>
          )}
        </CardContent>
      </Card>
    </div>
  );
}
