"use client";

import Link from "next/link";
import { useRouter } from "next/navigation";
import { useSession } from "next-auth/react";
import { Badge } from "@/components/ui/badge";
import { Input } from "@/components/ui/input";
import { Button } from "@/components/ui/button";
import { Checkbox } from "@/components/ui/checkbox";
import { useState, useEffect, useCallback, useMemo } from "react";
import { Card, CardHeader, CardContent } from "@/components/ui/card";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select";
import {
  Table,
  TableBody,
  TableCell,
  TableHead,
  TableHeader,
  TableRow,
} from "@/components/ui/table";
import {
  DropdownMenu,
  DropdownMenuContent,
  DropdownMenuItem,
  DropdownMenuLabel,
  DropdownMenuSeparator,
  DropdownMenuTrigger,
} from "@/components/ui/dropdown-menu";
import {
  AlertDialog,
  AlertDialogAction,
  AlertDialogCancel,
  AlertDialogContent,
  AlertDialogDescription,
  AlertDialogFooter,
  AlertDialogHeader,
  AlertDialogTitle,
} from "@/components/ui/alert-dialog";
import {
  Building2,
  Plus,
  Search,
  MoreHorizontal,
  Edit,
  Eye,
  Trash2,
  MapPin,
  DollarSign,
  Users,
  Bed,
  Bath,
  Square,
  RefreshCw,
  AlertCircle,
  CheckCircle,
  XCircle,
  Grid3X3,
  List,
  Rows3,
  X,
} from "lucide-react";
import { PropertyType, PropertyStatus, UserRole } from "@/types";
import {
  propertyService,
  PropertyResponse,
  PropertyQueryParams,
} from "@/lib/services/property.service";
import { toast } from "sonner";
import Image from "next/image";
import { getFeaturedImage, hasPropertyImages } from "@/lib/utils/image-utils";
import { PropertyRowCard } from "@/components/properties/PropertyRowCard";
import PropertyStats from "@/components/properties/PropertyStats";

interface PropertyCardProps {
  property: PropertyResponse;
  onEdit: (property: PropertyResponse) => void;
  onDelete: (property: PropertyResponse) => void;
  onView: (property: PropertyResponse) => void;
  deleteLoading: boolean;
}

function PropertyCard({
  property,
  onEdit,
  onDelete,
  onView,
  deleteLoading,
}: PropertyCardProps) {
  const getStatusColor = (status?: PropertyStatus) => {
    switch (status) {
      case PropertyStatus.AVAILABLE:
        return "bg-green-100 dark:bg-green-900 text-green-800 dark:text-green-200 border-green-200 dark:border-green-700";
      case PropertyStatus.OCCUPIED:
        return "bg-blue-100 dark:bg-blue-900 text-blue-800 dark:text-blue-200 border-blue-200 dark:border-blue-700";
      case PropertyStatus.MAINTENANCE:
        return "bg-yellow-100 dark:bg-yellow-900 text-yellow-800 dark:text-yellow-200 border-yellow-200 dark:border-yellow-700";
      case PropertyStatus.UNAVAILABLE:
        return "bg-red-100 dark:bg-red-900 text-red-800 dark:text-red-200 border-red-200 dark:border-red-700";
      default:
        return "bg-gray-100 dark:bg-gray-700 text-gray-800 dark:text-gray-200 border-gray-200 dark:border-gray-600";
    }
  };

  const getTypeIcon = (type?: PropertyType) => {
    switch (type) {
      case PropertyType.APARTMENT:
        return <Building2 className="h-4 w-4" />;
      case PropertyType.HOUSE:
        return <Building2 className="h-4 w-4" />;
      case PropertyType.CONDO:
        return <Building2 className="h-4 w-4" />;
      case PropertyType.TOWNHOUSE:
        return <Building2 className="h-4 w-4" />;
      default:
        return <Building2 className="h-4 w-4" />;
    }
  };

  // Safe image handling using utility functions
  const hasImage = hasPropertyImages(property);
  const featuredImage = getFeaturedImage(property);
  const propertyStatus = property?.status as PropertyStatus | undefined;
  const statusLabel = propertyStatus
    ? propertyStatus.charAt(0).toUpperCase() + propertyStatus.slice(1)
    : "Unknown";
  const propertyType = property?.type as PropertyType | undefined;
  const propertyName = property?.name ?? "Unknown Property";
  const addressParts = [
    property?.address?.street,
    property?.address?.city,
    property?.address?.state,
  ].filter(Boolean);
  const displayAddress =
    [addressParts.join(", "), property?.address?.zipCode]
      .filter(Boolean)
      .join(" ") || "Address unavailable";
  const bedrooms =
    typeof property?.bedrooms === "number" ? property?.bedrooms : 0;
  const bathrooms =
    typeof property?.bathrooms === "number" ? property?.bathrooms : 0;
  const squareFootage =
    typeof property?.squareFootage === "number" ? property?.squareFootage : 0;
  const rentAmount =
    typeof property?.rentAmount === "number" ? property?.rentAmount : 0;
  const totalUnits =
    typeof property?.totalUnits === "number" ? property?.totalUnits : 0;

  // Get unit statistics for multi-unit properties
  const unitStats = getUnitStats(property?.units);
  const rentRange = getRentRange(property?.units);

  return (
    <Card className="group hover:shadow-lg transition-all duration-200 overflow-hidden border border-gray-200 dark:border-gray-700 shadow-sm bg-white dark:bg-gray-800 p-0 gap-0 rounded-lg">
      {/* Featured Image */}
      <div className="relative h-48 overflow-hidden bg-gradient-to-br from-gray-100 to-gray-200 dark:from-gray-700 dark:to-gray-800 rounded-t-lg m-0 p-0">
        {hasImage ? (
          <Image
            src={featuredImage!}
            alt={propertyName}
            fill
            className="object-cover object-center w-full h-full group-hover:scale-105 transition-transform duration-300 m-0 p-0"
            sizes="(max-width: 768px) 100vw, (max-width: 1200px) 50vw, 33vw"
            priority={false}
          />
        ) : (
          <div className="w-full h-full flex items-center justify-center bg-gradient-to-br from-gray-100 to-gray-200 dark:from-gray-700 dark:to-gray-600">
            <Building2 className="h-16 w-16 text-gray-400 dark:text-gray-500" />
          </div>
        )}

        {/* Status Badge */}
        <div className="absolute top-3 left-3">
          <span
            className={`inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium border ${getStatusColor(
              propertyStatus
            )}`}
          >
            {statusLabel}
          </span>
        </div>

        {/* Type Badge */}
        <div className="absolute top-3 right-3">
          <span className="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-white/90 dark:bg-gray-800/90 text-gray-700 dark:text-gray-200 border border-gray-200 dark:border-gray-600">
            {getTypeIcon(propertyType)}
            <span className="ml-1">
              {propertyType
                ? propertyType.charAt(0).toUpperCase() + propertyType.slice(1)
                : "Unknown"}
            </span>
          </span>
        </div>

        {/* Overlay Actions */}
        <div className="absolute inset-0 bg-black/0 group-hover:bg-black/20 transition-colors duration-200 flex items-center justify-center opacity-0 group-hover:opacity-100">
          <div className="flex space-x-2">
            <Button
              size="sm"
              variant="secondary"
              onClick={() => onView(property)}
              className="bg-white/90 hover:bg-white text-gray-900"
            >
              <Eye className="h-4 w-4" />
            </Button>
            <Button
              size="sm"
              variant="secondary"
              onClick={() => onEdit(property)}
              className="bg-white/90 hover:bg-white text-gray-900"
            >
              <Edit className="h-4 w-4" />
            </Button>
          </div>
        </div>
      </div>

      {/* Card Content */}
      <CardContent className="p-4">
        {/* Property Name & Description */}
        <div className="mb-3">
          <div className="flex items-center gap-2 mb-1">
            <h3
              className={`font-semibold text-lg line-clamp-1 ${
                property.deletedAt
                  ? "text-gray-400 dark:text-gray-500 line-through"
                  : "text-gray-900 dark:text-gray-100"
              }`}
            >
              {propertyName}
            </h3>
            {property.deletedAt && (
              <span className="px-2 py-1 text-xs bg-red-100 text-red-800 rounded-full">
                Deleted
              </span>
            )}
          </div>
          {property?.description && (
            <p className="text-sm text-gray-600 dark:text-gray-400 line-clamp-1">
              {property.description}
            </p>
          )}
          {property.deletedAt && (
            <p className="text-xs text-red-500 mt-1">
              Deleted: {new Date(property.deletedAt).toLocaleDateString()}
            </p>
          )}
        </div>

        {/* Location */}
        <div className="flex items-center text-sm text-gray-600 dark:text-gray-400 mb-3">
          <MapPin className="h-4 w-4 mr-1 flex-shrink-0" />
          <span className="line-clamp-1">{displayAddress}</span>
        </div>

        {/* Property Details */}
        <div className="flex items-center justify-between text-sm text-gray-600 dark:text-gray-400 mb-4">
          <div className="flex items-center space-x-4">
            <div className="flex items-center">
              <Bed className="h-4 w-4 mr-1" />
              <span>{bedrooms}</span>
            </div>
            <div className="flex items-center">
              <Bath className="h-4 w-4 mr-1" />
              <span>{bathrooms}</span>
            </div>
            <div className="flex items-center">
              <Square className="h-4 w-4 mr-1" />
              <span>{squareFootage.toLocaleString()} ft²</span>
            </div>
          </div>
        </div>

        {/* Unit Information - Consistent for all properties */}
        <div className="mb-4 p-3 bg-gray-50 dark:bg-gray-700 rounded-lg border border-gray-200 dark:border-gray-600">
          <div className="flex items-center justify-between mb-2">
            <span className="text-sm font-medium text-gray-700 dark:text-gray-300">
              {property?.isMultiUnit && totalUnits > 1
                ? `${totalUnits} Units`
                : "Unit"}
            </span>
            <div className="flex items-center space-x-2 text-xs">
              {property?.isMultiUnit ? (
                <>
                  {unitStats.available > 0 && (
                    <span className="px-2 py-1 bg-green-100 dark:bg-green-900 text-green-700 dark:text-green-300 rounded-full">
                      {unitStats.available} available
                    </span>
                  )}
                  {unitStats.occupied > 0 && (
                    <span className="px-2 py-1 bg-blue-100 dark:bg-blue-900 text-blue-700 dark:text-blue-300 rounded-full">
                      {unitStats.occupied} occupied
                    </span>
                  )}
                  {unitStats.maintenance > 0 && (
                    <span className="px-2 py-1 bg-yellow-100 dark:bg-yellow-900 text-yellow-700 dark:text-yellow-300 rounded-full">
                      {unitStats.maintenance} maintenance
                    </span>
                  )}
                </>
              ) : (
                <span className="px-2 py-1 bg-green-100 dark:bg-green-900 text-green-700 dark:text-green-300 rounded-full">
                  {propertyStatus === PropertyStatus.AVAILABLE
                    ? "available"
                    : propertyStatus?.toLowerCase() || "unknown"}
                </span>
              )}
            </div>
          </div>
          {property?.isMultiUnit && unitStats.types.length > 0 ? (
            <div className="text-xs text-gray-600 dark:text-gray-400">
              Types: {unitStats.types.join(", ")}
            </div>
          ) : (
            <div className="text-xs text-gray-600 dark:text-gray-400">
              Types:{" "}
              {propertyType
                ? propertyType.charAt(0).toUpperCase() + propertyType.slice(1)
                : "Unknown"}
            </div>
          )}
        </div>

        {/* Rent & Owner */}
        <div className="flex items-center justify-between">
          <div className="flex flex-col">
            <div className="flex items-center text-lg font-semibold text-gray-900 dark:text-gray-100">
              <DollarSign className="h-5 w-5 mr-1" />
              <span>
                {property?.isMultiUnit && rentRange
                  ? rentRange
                  : rentAmount.toLocaleString()}
              </span>
              <span className="text-sm font-normal text-gray-600 dark:text-gray-400 ml-1">
                /month
              </span>
            </div>
            {property?.isMultiUnit && unitStats.available > 0 ? (
              <span className="text-xs text-green-600 dark:text-green-400 font-medium mt-1">
                {unitStats.available} units available
              </span>
            ) : (
              !property?.isMultiUnit && (
                <span className="text-xs text-green-600 dark:text-green-400 font-medium mt-1">
                  Single unit
                </span>
              )
            )}
          </div>

          {/* Actions Menu */}
          <DropdownMenu>
            <DropdownMenuTrigger asChild>
              <Button variant="ghost" size="sm" className="h-8 w-8 p-0">
                <MoreHorizontal className="h-4 w-4" />
              </Button>
            </DropdownMenuTrigger>
            <DropdownMenuContent align="end">
              <DropdownMenuItem onClick={() => onView(property)}>
                <Eye className="h-4 w-4 mr-2" />
                View Details
              </DropdownMenuItem>
              <DropdownMenuItem onClick={() => onEdit(property)}>
                <Edit className="h-4 w-4 mr-2" />
                Edit Property
              </DropdownMenuItem>
              <DropdownMenuSeparator />
              {!property.deletedAt ? (
                <DropdownMenuItem
                  onClick={() => onDelete(property)}
                  className="text-red-600 focus:text-red-600"
                  disabled={deleteLoading}
                >
                  <Trash2 className="h-4 w-4 mr-2" />
                  Delete Property
                </DropdownMenuItem>
              ) : (
                <DropdownMenuItem disabled className="text-gray-400">
                  <Trash2 className="h-4 w-4 mr-2" />
                  Already Deleted
                </DropdownMenuItem>
              )}
            </DropdownMenuContent>
          </DropdownMenu>
        </div>
      </CardContent>
    </Card>
  );
}

// ============================================================================
// HELPER FUNCTIONS
// ============================================================================

// Helper function to get unit statistics
const getUnitStats = (units?: PropertyResponse["units"]) => {
  if (!units || units.length === 0) {
    return { available: 0, occupied: 0, maintenance: 0, total: 0, types: [] };
  }

  const stats = {
    available: 0,
    occupied: 0,
    maintenance: 0,
    total: units.length,
    types: [] as string[],
  };

  const typeSet = new Set<string>();

  units.forEach((unit) => {
    switch (unit.status) {
      case PropertyStatus.AVAILABLE:
        stats.available++;
        break;
      case PropertyStatus.OCCUPIED:
        stats.occupied++;
        break;
      case PropertyStatus.MAINTENANCE:
        stats.maintenance++;
        break;
    }
    typeSet.add(unit.unitType);
  });

  stats.types = Array.from(typeSet);
  return stats;
};

// Helper function to get rent range for multi-unit properties
const getRentRange = (units?: PropertyResponse["units"]) => {
  if (!units || units.length === 0) return null;

  const rents = units.map((unit) => unit.rentAmount);
  const minRent = Math.min(...rents);
  const maxRent = Math.max(...rents);

  if (minRent === maxRent) {
    return `$${minRent.toLocaleString()}`;
  }

  return `$${minRent.toLocaleString()} - $${maxRent.toLocaleString()}`;
};

export default function PropertiesPage() {
  const { data: session } = useSession();
  const router = useRouter();
  const [properties, setProperties] = useState<PropertyResponse[]>([]);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState<string | null>(null);
  const [deleteLoading, setDeleteLoading] = useState<string | null>(null);
  const [showDeleteDialog, setShowDeleteDialog] = useState(false);
  const [propertyToDelete, setPropertyToDelete] =
    useState<PropertyResponse | null>(null);
  const [viewMode, setViewMode] = useState<"grid" | "rows" | "list">("grid");

  // Bulk selection state
  const [selectedProperties, setSelectedProperties] = useState<string[]>([]);
  const [bulkDeleteLoading, setBulkDeleteLoading] = useState(false);

  // Filter and pagination state
  const [filters, setFilters] = useState<PropertyQueryParams>({
    page: 1,
    limit: 10,
    sortBy: "createdAt",
    sortOrder: "desc",
  });

  const [pagination, setPagination] = useState({
    page: 1,
    limit: 10,
    total: 0,
    pages: 0,
    hasNext: false,
    hasPrev: false,
  });

  const fetchProperties = useCallback(async () => {
    try {
      setLoading(true);
      setError(null);

      // Adjust filters based on user preferences
      const adjustedFilters = {
        ...filters,
        limit: session?.user?.role === UserRole.ADMIN ? 100 : filters.limit,
      };

      const response = await propertyService.getProperties(adjustedFilters);
      setProperties(response.data);
      setPagination(response.pagination);
    } catch (err) {
      const message =
        err instanceof Error ? err.message : "Failed to fetch properties";
      setError(message);
      toast.error(message);
    } finally {
      setLoading(false);
    }
  }, [filters, session?.user?.role]);

  useEffect(() => {
    fetchProperties();
  }, [fetchProperties]);

  // Show/hide bulk actions based on selection - handled inline now

  // Memoized calculations for better performance
  const isAllSelected = useMemo(() => {
    return (
      selectedProperties.length === properties.length && properties.length > 0
    );
  }, [selectedProperties.length, properties.length]);

  const hasActiveFilters = useMemo(() => {
    return !!(
      filters.search ||
      filters.type ||
      filters.status ||
      filters.unitType
    );
  }, [filters.search, filters.type, filters.status, filters.unitType]);

  const handleSearch = (search: string) => {
    setFilters((prev) => ({ ...prev, search, page: 1 }));
  };

  const handleFilterChange = (
    key: keyof PropertyQueryParams,
    value: string | number | undefined
  ) => {
    setFilters((prev) => ({ ...prev, [key]: value, page: 1 }));
  };

  const handlePageChange = (page: number) => {
    setFilters((prev) => ({ ...prev, page }));
  };

  const handleDeleteClick = (property: PropertyResponse) => {
    setPropertyToDelete(property);
    setShowDeleteDialog(true);
  };

  const handleDeleteConfirm = async () => {
    if (!propertyToDelete) return;

    // Check if property is already deleted
    if (propertyToDelete.deletedAt) {
      toast.error("This property has already been deleted");
      setShowDeleteDialog(false);
      setPropertyToDelete(null);
      return;
    }

    try {
      setDeleteLoading(propertyToDelete._id);
      await propertyService.deleteProperty(propertyToDelete._id);

      // Show success message
      toast.success("Property deleted successfully");

      // Refresh the list
      await fetchProperties();
    } catch (err) {
      const message =
        err instanceof Error ? err.message : "Failed to delete property";
      toast.error(message);
    } finally {
      setDeleteLoading(null);
      setShowDeleteDialog(false);
      setPropertyToDelete(null);
    }
  };

  const handleDeleteCancel = () => {
    setShowDeleteDialog(false);
    setPropertyToDelete(null);
  };

  const handleSelectProperty = (propertyId: string, checked: boolean) => {
    if (checked) {
      setSelectedProperties((prev) => [...prev, propertyId]);
    } else {
      setSelectedProperties((prev) => prev.filter((id) => id !== propertyId));
    }
  };

  const handleSelectAll = (checked: boolean) => {
    if (checked) {
      setSelectedProperties(properties.map((p) => p._id));
    } else {
      setSelectedProperties([]);
    }
  };

  const handleBulkDelete = async () => {
    if (selectedProperties.length === 0) return;

    // Filter out already deleted properties
    const deletableProperties = selectedProperties.filter((id) => {
      const property = properties.find((p) => p._id === id);
      return property && !property.deletedAt;
    });

    if (deletableProperties.length === 0) {
      toast.error("All selected properties have already been deleted");
      return;
    }

    if (deletableProperties.length < selectedProperties.length) {
      const alreadyDeletedCount =
        selectedProperties.length - deletableProperties.length;
      toast.warning(
        `${alreadyDeletedCount} properties are already deleted and will be skipped`
      );
    }

    try {
      setBulkDeleteLoading(true);

      // Use the bulk delete API endpoint with only deletable properties
      const response = await fetch(
        `/api/properties?ids=${deletableProperties.join(",")}`,
        {
          method: "DELETE",
          credentials: "include",
        }
      );

      if (!response.ok) {
        const error = await response.json();
        throw new Error(error.message || "Failed to delete properties");
      }

      toast.success(
        `${deletableProperties.length} properties deleted successfully`
      );
      setSelectedProperties([]);
      await fetchProperties();
    } catch (err) {
      const message =
        err instanceof Error ? err.message : "Failed to delete properties";
      toast.error(message);
    } finally {
      setBulkDeleteLoading(false);
    }
  };

  const handleStatusChange = async (id: string, status: PropertyStatus) => {
    try {
      await propertyService.updatePropertyStatus(id, status);

      // Show success message
      toast.success("Property status updated successfully");

      // Refresh the list
      await fetchProperties();
    } catch (err) {
      const message =
        err instanceof Error ? err.message : "Failed to update property status";
      toast.error(message);
    }
  };

  const getStatusBadgeVariant = (status: PropertyStatus) => {
    switch (status) {
      case PropertyStatus.AVAILABLE:
        return "default";
      case PropertyStatus.OCCUPIED:
        return "secondary";
      case PropertyStatus.MAINTENANCE:
        return "destructive";
      case PropertyStatus.UNAVAILABLE:
        return "outline";
      default:
        return "default";
    }
  };

  const formatCurrency = (amount: number) => {
    return new Intl.NumberFormat("en-US", {
      style: "currency",
      currency: "USD",
    }).format(amount);
  };

  if (!session) {
    return (
      <div className="flex items-center justify-center min-h-screen">
        <div className="text-center">
          <AlertCircle className="h-12 w-12 text-red-500 mx-auto mb-4" />
          <h2 className="text-xl font-semibold mb-2">Access Denied</h2>
          <p className="text-gray-600">Please sign in to view properties.</p>
        </div>
      </div>
    );
  }

  return (
    <div className="space-y-4">
      {/* Header */}
      <div className="flex flex-col sm:flex-row sm:items-center sm:justify-between gap-4">
        <div>
          <h1 className="text-2xl sm:text-3xl font-bold tracking-tight">
            Properties
          </h1>
          <p className="text-muted-foreground text-sm sm:text-base">
            Manage your property portfolio
          </p>
        </div>
        <div className="flex items-center gap-2 sm:space-x-2">
          <Button
            variant="outline"
            size="sm"
            onClick={fetchProperties}
            disabled={loading}
            className="flex-1 sm:flex-none"
          >
            <RefreshCw
              className={`h-4 w-4 ${loading ? "animate-spin" : ""} sm:mr-2`}
            />
            <span className="hidden sm:inline">Refresh</span>
          </Button>
          <Link
            href="/dashboard/properties/new"
            className="flex-1 sm:flex-none"
          >
            <Button className="w-full sm:w-auto">
              <Plus className="h-4 w-4 sm:mr-2" />
              <span className="hidden sm:inline">Add Property</span>
              <span className="sm:hidden">Add</span>
            </Button>
          </Link>
        </div>
      </div>

      {/* Property Statistics */}
      <PropertyStats properties={properties} isLoading={loading} />

      {/* Properties Display with Integrated Filters */}
      <Card>
        <CardHeader className="pb-4">
          {/* Main Header */}
          <div className="flex flex-col lg:flex-row lg:items-center lg:justify-between gap-4 mb-4">
            <div className="flex items-center gap-3">
              <div className="p-2 bg-blue-50 dark:bg-blue-900/30 rounded-lg border border-blue-100 dark:border-blue-800">
                <Building2 className="h-5 w-5 text-blue-600 dark:text-blue-400" />
              </div>
              <div>
                <h2 className="text-lg font-semibold text-gray-900 dark:text-gray-100">
                  Properties
                </h2>
                <p className="text-sm text-gray-500 dark:text-gray-400">
                  Manage your property portfolio
                </p>
              </div>
            </div>

            <div className="flex flex-col sm:flex-row items-stretch sm:items-center gap-3">
              {/* Bulk Actions */}
              {selectedProperties.length > 0 && (
                <div className="flex items-center gap-2 p-2 bg-blue-50 dark:bg-blue-900/30 rounded-lg border border-blue-100 dark:border-blue-800">
                  <span className="text-sm font-medium text-blue-700 dark:text-blue-300">
                    {selectedProperties.length} selected
                  </span>
                  <Button
                    variant="destructive"
                    size="sm"
                    onClick={handleBulkDelete}
                    disabled={bulkDeleteLoading}
                    className="h-8"
                  >
                    <Trash2 className="h-4 w-4 mr-1" />
                    Delete ({selectedProperties.length})
                  </Button>
                  <Button
                    variant="outline"
                    size="sm"
                    onClick={() => setSelectedProperties([])}
                    className="h-8"
                  >
                    Clear
                  </Button>
                </div>
              )}

              {/* View Mode Toggle */}
              <div className="flex items-center border rounded-lg p-1 w-full sm:w-auto">
                <Button
                  variant={viewMode === "grid" ? "default" : "ghost"}
                  size="sm"
                  onClick={() => setViewMode("grid")}
                  className="h-8 flex-1 sm:flex-none sm:px-3"
                >
                  <Grid3X3 className="h-4 w-4" />
                  <span className="ml-1 sm:hidden">Grid</span>
                </Button>
                <Button
                  variant={viewMode === "rows" ? "default" : "ghost"}
                  size="sm"
                  onClick={() => setViewMode("rows")}
                  className="h-8 flex-1 sm:flex-none sm:px-3"
                >
                  <Rows3 className="h-4 w-4" />
                  <span className="ml-1 sm:hidden">Rows</span>
                </Button>
                <Button
                  variant={viewMode === "list" ? "default" : "ghost"}
                  size="sm"
                  onClick={() => setViewMode("list")}
                  className="h-8 flex-1 sm:flex-none sm:px-3"
                >
                  <List className="h-4 w-4" />
                  <span className="ml-1 sm:hidden">List</span>
                </Button>
              </div>
            </div>
          </div>

          {/* Integrated Filters Bar */}
          <div className="flex flex-col lg:flex-row lg:items-center gap-4 p-4 bg-gray-50/50 dark:bg-gray-800/50 rounded-lg border border-gray-200/60 dark:border-gray-700/60">
            {/* Search */}
            <div className="flex-1 min-w-0">
              <div className="relative">
                <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 h-4 w-4 text-gray-400 dark:text-gray-500" />
                <Input
                  placeholder="Search properties..."
                  value={filters.search || ""}
                  onChange={(e) => handleSearch(e.target.value)}
                  className="pl-10 h-10 border-gray-200 dark:border-gray-700 focus:border-blue-400 dark:focus:border-blue-500 focus:ring-1 focus:ring-blue-400 dark:focus:ring-blue-500 bg-white dark:bg-gray-800"
                />
              </div>
            </div>

            {/* Filter Controls */}
            <div className="flex flex-wrap items-center gap-3">
              {/* Property Type */}
              <Select
                value={filters.type || "all"}
                onValueChange={(value) =>
                  handleFilterChange(
                    "type",
                    value === "all" ? undefined : value
                  )
                }
              >
                <SelectTrigger className="w-[140px] h-10 border-gray-200 dark:border-gray-700 bg-white dark:bg-gray-800">
                  <SelectValue placeholder="All Types" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="all">All Types</SelectItem>
                  {Object.values(PropertyType).map((type) => (
                    <SelectItem key={type} value={type}>
                      {type.charAt(0).toUpperCase() + type.slice(1)}
                    </SelectItem>
                  ))}
                </SelectContent>
              </Select>

              {/* Status */}
              <Select
                value={filters.status || "all"}
                onValueChange={(value) =>
                  handleFilterChange(
                    "status",
                    value === "all" ? undefined : value
                  )
                }
              >
                <SelectTrigger className="w-[140px] h-10 border-gray-200 dark:border-gray-700 bg-white dark:bg-gray-800">
                  <SelectValue placeholder="All Statuses" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="all">All Statuses</SelectItem>
                  {Object.values(PropertyStatus).map((status) => (
                    <SelectItem key={status} value={status}>
                      {status.charAt(0).toUpperCase() + status.slice(1)}
                    </SelectItem>
                  ))}
                </SelectContent>
              </Select>

              {/* Sort */}
              <Select
                value={`${filters.sortBy}-${filters.sortOrder}`}
                onValueChange={(value) => {
                  const [sortBy, sortOrder] = value.split("-");
                  setFilters((prev) => ({
                    ...prev,
                    sortBy: sortBy as
                      | "name"
                      | "rentAmount"
                      | "createdAt"
                      | "squareFootage",
                    sortOrder: sortOrder as "asc" | "desc",
                  }));
                }}
              >
                <SelectTrigger className="w-[140px] h-10 border-gray-200 dark:border-gray-700 bg-white dark:bg-gray-800">
                  <SelectValue placeholder="Sort by" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="createdAt-desc">Newest First</SelectItem>
                  <SelectItem value="createdAt-asc">Oldest First</SelectItem>
                  <SelectItem value="name-asc">Name A-Z</SelectItem>
                  <SelectItem value="name-desc">Name Z-A</SelectItem>
                  <SelectItem value="rentAmount-desc">Highest Rent</SelectItem>
                  <SelectItem value="rentAmount-asc">Lowest Rent</SelectItem>
                </SelectContent>
              </Select>

              {/* Unit Type Filter */}
              <Select
                value={filters.unitType || "all"}
                onValueChange={(value) =>
                  handleFilterChange(
                    "unitType",
                    value === "all" ? undefined : value
                  )
                }
              >
                <SelectTrigger className="w-[140px] h-10 border-gray-200 dark:border-gray-700 bg-white dark:bg-gray-800">
                  <SelectValue placeholder="Unit Type" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="all">All Units</SelectItem>
                  <SelectItem value="apartment">Apartment</SelectItem>
                  <SelectItem value="studio">Studio</SelectItem>
                  <SelectItem value="penthouse">Penthouse</SelectItem>
                  <SelectItem value="loft">Loft</SelectItem>
                  <SelectItem value="room">Room</SelectItem>
                </SelectContent>
              </Select>

              {/* Clear Filters */}
              {hasActiveFilters && (
                <Button
                  variant="ghost"
                  size="sm"
                  onClick={() =>
                    setFilters({
                      page: 1,
                      limit: 10,
                      sortBy: "createdAt",
                      sortOrder: "desc",
                    })
                  }
                  className="h-10 px-3 text-gray-500 hover:text-gray-700"
                >
                  <X className="h-4 w-4 mr-1" />
                  Clear
                </Button>
              )}
            </div>
          </div>
        </CardHeader>

        {/* Content */}
        <CardContent>
          {loading ? (
            viewMode === "grid" ? (
              /* Grid Loading Skeleton */
              <div className="grid grid-cols-1 sm:grid-cols-2 md:grid-cols-2 lg:grid-cols-3 xl:grid-cols-4 gap-4 sm:gap-6">
                {Array.from({ length: 8 }).map((_, i) => (
                  <Card key={i} className="overflow-hidden p-0">
                    <div className="relative">
                      <div className="h-48 w-full bg-gray-200 animate-pulse" />
                      <div className="absolute top-2 right-2">
                        <div className="h-6 w-16 bg-gray-300 rounded animate-pulse" />
                      </div>
                    </div>
                    <CardContent className="p-4 space-y-3">
                      <div className="space-y-2">
                        <div className="h-6 w-3/4 bg-gray-200 rounded animate-pulse" />
                        <div className="h-4 w-full bg-gray-200 rounded animate-pulse" />
                      </div>
                      <div className="flex items-center justify-between">
                        <div className="space-y-1">
                          <div className="h-4 w-12 bg-gray-200 rounded animate-pulse" />
                          <div className="h-6 w-20 bg-gray-200 rounded animate-pulse" />
                        </div>
                        <div className="h-8 w-8 bg-gray-200 rounded animate-pulse" />
                      </div>
                    </CardContent>
                  </Card>
                ))}
              </div>
            ) : viewMode === "rows" ? (
              /* Rows Loading Skeleton */
              <div className="space-y-3">
                {Array.from({ length: 6 }).map((_, i) => (
                  <Card key={i} className="overflow-hidden py-0">
                    <CardContent className="p-0">
                      <div className="flex items-center h-20">
                        <div className="w-32 h-20 bg-gray-200 animate-pulse flex-shrink-0" />
                        <div className="flex-1 px-4 py-3 space-y-2">
                          <div className="h-5 w-1/3 bg-gray-200 rounded animate-pulse" />
                          <div className="h-4 w-1/2 bg-gray-200 rounded animate-pulse" />
                        </div>
                        <div className="px-4 space-y-2">
                          <div className="h-4 w-16 bg-gray-200 rounded animate-pulse" />
                          <div className="h-4 w-12 bg-gray-200 rounded animate-pulse" />
                        </div>
                      </div>
                    </CardContent>
                  </Card>
                ))}
              </div>
            ) : (
              /* Table Loading Skeleton */
              <div className="space-y-4">
                {Array.from({ length: 8 }).map((_, i) => (
                  <div
                    key={i}
                    className="flex items-center space-x-4 py-4 border-b border-gray-100"
                  >
                    <div className="w-4 h-4 bg-gray-200 rounded animate-pulse" />
                    <div className="w-10 h-10 bg-gray-200 rounded animate-pulse" />
                    <div className="flex-1 space-y-2">
                      <div className="h-4 w-1/4 bg-gray-200 rounded animate-pulse" />
                      <div className="h-3 w-1/3 bg-gray-200 rounded animate-pulse" />
                    </div>
                    <div className="w-16 h-4 bg-gray-200 rounded animate-pulse" />
                    <div className="w-20 h-4 bg-gray-200 rounded animate-pulse" />
                    <div className="w-24 h-4 bg-gray-200 rounded animate-pulse" />
                    <div className="w-8 h-8 bg-gray-200 rounded animate-pulse" />
                  </div>
                ))}
              </div>
            )
          ) : error ? (
            <div className="flex items-center justify-center py-16">
              <AlertCircle className="h-8 w-8 text-red-500" />
              <span className="ml-2 text-red-600">{error}</span>
              <Button
                variant="outline"
                size="sm"
                onClick={fetchProperties}
                className="ml-4"
              >
                Try Again
              </Button>
            </div>
          ) : properties.length === 0 ? (
            <div className="flex items-center justify-center py-16">
              <Building2 className="h-8 w-8 text-gray-400" />
              <span className="ml-2 text-gray-600">No properties found</span>
              <Link href="/dashboard/properties/new">
                <Button variant="outline" size="sm" className="ml-4">
                  <Plus className="h-4 w-4 mr-2" />
                  Add First Property
                </Button>
              </Link>
            </div>
          ) : viewMode === "grid" ? (
            /* Grid View */
            <div className="grid grid-cols-1 sm:grid-cols-2 md:grid-cols-2 lg:grid-cols-3 xl:grid-cols-4 gap-4 sm:gap-6">
              {properties.map((property) => (
                <PropertyCard
                  key={property._id}
                  property={property}
                  onEdit={(property) =>
                    router.push(`/dashboard/properties/${property._id}/edit`)
                  }
                  onDelete={handleDeleteClick}
                  onView={(property) =>
                    router.push(`/dashboard/properties/${property._id}`)
                  }
                  deleteLoading={deleteLoading === property._id}
                />
              ))}
            </div>
          ) : viewMode === "rows" ? (
            /* Row View */
            <div className="space-y-4">
              {properties.map((property) => (
                <PropertyRowCard
                  key={property._id}
                  property={property}
                  onEdit={(property) =>
                    router.push(`/dashboard/properties/${property._id}/edit`)
                  }
                  onDelete={handleDeleteClick}
                  onView={(property) =>
                    router.push(`/dashboard/properties/${property._id}`)
                  }
                  deleteLoading={deleteLoading === property._id}
                />
              ))}
            </div>
          ) : (
            /* List View - Table */
            <div className="border border-gray-200 dark:border-gray-700 rounded-lg overflow-hidden">
              <Table>
                <TableHeader className="bg-gray-50/50 dark:bg-gray-800/50">
                  <TableRow className="border-b border-gray-200 dark:border-gray-700">
                    <TableHead className="text-left font-medium text-gray-700 dark:text-gray-300 py-3 px-4 w-12">
                      <Checkbox
                        checked={isAllSelected}
                        onCheckedChange={handleSelectAll}
                        aria-label="Select all properties"
                      />
                    </TableHead>
                    <TableHead className="text-left font-medium text-gray-700 dark:text-gray-300 py-3 px-6">
                      Property
                    </TableHead>
                    <TableHead className="text-left font-medium text-gray-700 dark:text-gray-300 py-3 px-4">
                      Status
                    </TableHead>
                    <TableHead className="text-left font-medium text-gray-700 dark:text-gray-300 py-3 px-4">
                      Contact
                    </TableHead>
                    <TableHead className="text-left font-medium text-gray-700 dark:text-gray-300 py-3 px-4">
                      Details
                    </TableHead>
                    <TableHead className="text-left font-medium text-gray-700 dark:text-gray-300 py-3 px-4">
                      Units
                    </TableHead>
                    <TableHead className="text-left font-medium text-gray-700 dark:text-gray-300 py-3 px-4">
                      Rent
                    </TableHead>
                    <TableHead className="text-left font-medium text-gray-700 dark:text-gray-300 py-3 px-4">
                      Owner
                    </TableHead>
                    <TableHead className="text-right font-medium text-gray-700 dark:text-gray-300 py-3 px-6">
                      Actions
                    </TableHead>
                  </TableRow>
                </TableHeader>
                <TableBody>
                  {properties?.map((property, index) => (
                    <TableRow
                      key={property._id}
                      className={`border-b border-gray-100 dark:border-gray-700 hover:bg-gray-50/50 dark:hover:bg-gray-700/50 transition-colors ${
                        index % 2 === 0
                          ? "bg-white dark:bg-gray-800"
                          : "bg-gray-50/20 dark:bg-gray-800/50"
                      }`}
                    >
                      <TableCell className="py-4 px-4">
                        <Checkbox
                          checked={selectedProperties.includes(property._id)}
                          onCheckedChange={(checked) =>
                            handleSelectProperty(
                              property._id,
                              checked as boolean
                            )
                          }
                          disabled={!!property.deletedAt}
                          aria-label={`Select ${property.name}`}
                        />
                      </TableCell>
                      <TableCell className="py-4 px-6">
                        <div className="flex items-center space-x-3">
                          <div className="relative w-10 h-10 rounded-lg overflow-hidden bg-gradient-to-br from-gray-100 to-gray-200 dark:from-gray-700 dark:to-gray-800 flex-shrink-0">
                            {hasPropertyImages(property) ? (
                              <Image
                                src={getFeaturedImage(property)!}
                                alt={property.name}
                                fill
                                className="object-cover object-center w-full h-full m-0 p-0"
                                sizes="40px"
                                priority={false}
                              />
                            ) : (
                              <div className="w-full h-full flex items-center justify-center bg-gradient-to-br from-gray-100 to-gray-200 dark:from-gray-700 dark:to-gray-600">
                                <Building2 className="h-5 w-5 text-gray-400 dark:text-gray-500" />
                              </div>
                            )}
                          </div>
                          <div className="min-w-0 flex-1">
                            <div className="font-medium text-gray-900 dark:text-gray-100">
                              <Link
                                href={`/dashboard/properties/${property._id}`}
                                className={`hover:text-blue-600 dark:hover:text-blue-400 transition-colors ${
                                  property.deletedAt
                                    ? "line-through text-gray-400"
                                    : ""
                                }`}
                              >
                                {property.name}
                              </Link>
                              {property.deletedAt && (
                                <span className="ml-2 px-2 py-1 text-xs bg-red-100 text-red-800 rounded-full">
                                  Deleted
                                </span>
                              )}
                            </div>
                            <div className="text-sm text-gray-500 dark:text-gray-400 truncate">
                              ID: {property._id.slice(-6)}
                              {property.deletedAt && (
                                <span className="ml-2 text-red-500">
                                  (Deleted:{" "}
                                  {new Date(
                                    property.deletedAt
                                  ).toLocaleDateString()}
                                  )
                                </span>
                              )}
                            </div>
                          </div>
                        </div>
                      </TableCell>
                      <TableCell className="py-4 px-4">
                        <div className="flex flex-col space-y-1">
                          <Badge
                            variant={getStatusBadgeVariant(property.status)}
                            className="w-fit"
                          >
                            {property?.status
                              ? property.status.charAt(0).toUpperCase() +
                                property.status.slice(1)
                              : "Unknown"}
                          </Badge>
                          <span className="text-xs text-gray-500 dark:text-gray-400">
                            {property?.type
                              ? property.type.charAt(0).toUpperCase() +
                                property.type.slice(1)
                              : "Unknown"}
                          </span>
                        </div>
                      </TableCell>
                      <TableCell className="py-4 px-4">
                        <div className="flex flex-col space-y-1">
                          <div className="flex items-center text-sm text-gray-900 dark:text-gray-100">
                            <MapPin className="h-3 w-3 mr-1 text-gray-400 dark:text-gray-500" />
                            {[property?.address?.city, property?.address?.state]
                              .filter(Boolean)
                              .join(", ") || "Location unavailable"}
                          </div>
                          <div className="text-xs text-gray-500 dark:text-gray-400">
                            {property?.address?.street ?? ""}
                          </div>
                        </div>
                      </TableCell>
                      <TableCell className="py-4 px-4">
                        <div className="flex flex-col space-y-1">
                          <div className="flex items-center space-x-3 text-sm text-gray-900 dark:text-gray-100">
                            <div className="flex items-center">
                              <Bed className="h-3 w-3 mr-1 text-gray-400 dark:text-gray-500" />
                              {property?.bedrooms ?? 0}
                            </div>
                            <div className="flex items-center">
                              <Bath className="h-3 w-3 mr-1 text-gray-400 dark:text-gray-500" />
                              {property?.bathrooms ?? 0}
                            </div>
                          </div>
                          <div className="text-xs text-gray-500 dark:text-gray-400">
                            {typeof property?.squareFootage === "number"
                              ? property.squareFootage.toLocaleString()
                              : "0"}{" "}
                            sq ft
                          </div>
                        </div>
                      </TableCell>
                      <TableCell className="py-4 px-4">
                        <div className="flex flex-col space-y-1">
                          <div className="text-sm font-medium text-gray-900 dark:text-gray-100">
                            {property.isMultiUnit &&
                            (property.totalUnits ?? 0) > 1
                              ? `${property.totalUnits ?? 0} Units`
                              : "Unit"}
                          </div>
                          <div className="flex items-center space-x-2 text-xs">
                            {property.isMultiUnit ? (
                              (() => {
                                const unitStats = getUnitStats(property?.units);
                                return (
                                  <>
                                    {unitStats.available > 0 && (
                                      <span className="text-green-600 dark:text-green-400">
                                        {unitStats.available} available
                                      </span>
                                    )}
                                    {unitStats.occupied > 0 && (
                                      <span className="text-blue-600 dark:text-blue-400">
                                        {unitStats.occupied} occupied
                                      </span>
                                    )}
                                  </>
                                );
                              })()
                            ) : (
                              <span className="text-green-600 dark:text-green-400">
                                {property?.status === PropertyStatus.AVAILABLE
                                  ? "available"
                                  : property?.status?.toLowerCase() ||
                                    "unknown"}
                              </span>
                            )}
                          </div>
                        </div>
                      </TableCell>
                      <TableCell className="py-4 px-4">
                        <div className="flex flex-col space-y-1">
                          <div className="font-medium text-gray-900 dark:text-gray-100">
                            {property.isMultiUnit &&
                            getRentRange(property?.units)
                              ? getRentRange(property?.units)
                              : formatCurrency(property?.rentAmount ?? 0)}
                          </div>
                          <div className="text-xs text-gray-500 dark:text-gray-400">
                            /month
                          </div>
                        </div>
                      </TableCell>
                      <TableCell className="py-4 px-4">
                        <div className="flex flex-col space-y-1">
                          <div className="text-sm text-gray-900 dark:text-gray-100">
                            {property.ownerId?.firstName ?? ""}{" "}
                            {property.ownerId?.lastName ?? ""}
                          </div>
                          <div className="text-xs text-gray-500 dark:text-gray-400">
                            Property Owner
                          </div>
                        </div>
                      </TableCell>
                      <TableCell className="text-right py-4 px-6">
                        <DropdownMenu>
                          <DropdownMenuTrigger asChild>
                            <Button
                              variant="ghost"
                              className="h-8 w-8 p-0 hover:bg-gray-100 dark:hover:bg-gray-700 transition-colors"
                            >
                              <MoreHorizontal className="h-4 w-4 text-gray-500 dark:text-gray-400" />
                            </Button>
                          </DropdownMenuTrigger>
                          <DropdownMenuContent align="end">
                            <DropdownMenuLabel>Actions</DropdownMenuLabel>
                            <DropdownMenuItem asChild>
                              <Link
                                href={`/dashboard/properties/${
                                  property?._id ?? ""
                                }`}
                              >
                                <Eye className="mr-2 h-4 w-4" />
                                View Details
                              </Link>
                            </DropdownMenuItem>
                            <DropdownMenuItem asChild>
                              <Link
                                href={`/dashboard/properties/${
                                  property?._id ?? ""
                                }/edit`}
                              >
                                <Edit className="mr-2 h-4 w-4" />
                                Edit Property
                              </Link>
                            </DropdownMenuItem>
                            <DropdownMenuSeparator />

                            {/* Status Change Options */}
                            {property.status !== PropertyStatus.AVAILABLE && (
                              <DropdownMenuItem
                                onClick={() =>
                                  handleStatusChange(
                                    property._id,
                                    PropertyStatus.AVAILABLE
                                  )
                                }
                              >
                                <CheckCircle className="mr-2 h-4 w-4 text-green-600" />
                                Mark Available
                              </DropdownMenuItem>
                            )}
                            {property.status !== PropertyStatus.OCCUPIED && (
                              <DropdownMenuItem
                                onClick={() =>
                                  handleStatusChange(
                                    property._id,
                                    PropertyStatus.OCCUPIED
                                  )
                                }
                              >
                                <Users className="mr-2 h-4 w-4 text-blue-600" />
                                Mark Occupied
                              </DropdownMenuItem>
                            )}
                            {property.status !== PropertyStatus.MAINTENANCE && (
                              <DropdownMenuItem
                                onClick={() =>
                                  handleStatusChange(
                                    property._id,
                                    PropertyStatus.MAINTENANCE
                                  )
                                }
                              >
                                <AlertCircle className="mr-2 h-4 w-4 text-yellow-600" />
                                Mark Maintenance
                              </DropdownMenuItem>
                            )}
                            {property.status !== PropertyStatus.UNAVAILABLE && (
                              <DropdownMenuItem
                                onClick={() =>
                                  handleStatusChange(
                                    property._id,
                                    PropertyStatus.UNAVAILABLE
                                  )
                                }
                              >
                                <XCircle className="mr-2 h-4 w-4 text-red-600" />
                                Mark Unavailable
                              </DropdownMenuItem>
                            )}

                            <DropdownMenuSeparator />
                            {!property.deletedAt ? (
                              <DropdownMenuItem
                                className="text-red-600"
                                onClick={() => handleDeleteClick(property)}
                                disabled={deleteLoading === property._id}
                              >
                                <Trash2 className="mr-2 h-4 w-4" />
                                {deleteLoading === property._id
                                  ? "Deleting..."
                                  : "Delete Property"}
                              </DropdownMenuItem>
                            ) : (
                              <DropdownMenuItem
                                disabled
                                className="text-gray-400"
                              >
                                <Trash2 className="mr-2 h-4 w-4" />
                                Already Deleted
                              </DropdownMenuItem>
                            )}
                          </DropdownMenuContent>
                        </DropdownMenu>
                      </TableCell>
                    </TableRow>
                  ))}
                </TableBody>
              </Table>
            </div>
          )}

          {/* Pagination */}
          {pagination.pages > 1 && (
            <div className="flex flex-col sm:flex-row sm:items-center sm:justify-between gap-4">
              <div className="text-sm text-muted-foreground text-center sm:text-left">
                Showing {(pagination.page - 1) * pagination.limit + 1} to{" "}
                {Math.min(pagination.page * pagination.limit, pagination.total)}{" "}
                of {pagination.total} properties
              </div>
              <div className="flex items-center justify-center sm:justify-end space-x-2">
                <Button
                  variant="outline"
                  size="sm"
                  onClick={() => handlePageChange(pagination.page - 1)}
                  disabled={!pagination.hasPrev}
                  className="px-2 sm:px-4"
                >
                  <span className="hidden sm:inline">Previous</span>
                  <span className="sm:hidden">Prev</span>
                </Button>
                <div className="flex items-center space-x-1">
                  {Array.from(
                    { length: Math.min(3, pagination.pages) },
                    (_, i) => {
                      const page = i + 1;
                      return (
                        <Button
                          key={page}
                          variant={
                            page === pagination.page ? "default" : "outline"
                          }
                          size="sm"
                          onClick={() => handlePageChange(page)}
                          className="w-8 h-8 p-0"
                        >
                          {page}
                        </Button>
                      );
                    }
                  )}
                </div>
                <Button
                  variant="outline"
                  size="sm"
                  onClick={() => handlePageChange(pagination.page + 1)}
                  disabled={!pagination.hasNext}
                  className="px-2 sm:px-4"
                >
                  <span className="hidden sm:inline">Next</span>
                  <span className="sm:hidden">Next</span>
                </Button>
              </div>
            </div>
          )}
        </CardContent>
      </Card>

      {/* Delete Confirmation Dialog */}
      <AlertDialog open={showDeleteDialog} onOpenChange={setShowDeleteDialog}>
        <AlertDialogContent>
          <AlertDialogHeader>
            <AlertDialogTitle>Delete Property</AlertDialogTitle>
            <AlertDialogDescription>
              Are you sure you want to delete &quot;{propertyToDelete?.name}
              &quot;? This action cannot be undone. All associated data
              including leases, payments, and maintenance requests will be
              affected.
            </AlertDialogDescription>
          </AlertDialogHeader>
          <AlertDialogFooter>
            <AlertDialogCancel
              onClick={handleDeleteCancel}
              disabled={deleteLoading === propertyToDelete?._id}
            >
              Cancel
            </AlertDialogCancel>
            <AlertDialogAction
              onClick={handleDeleteConfirm}
              disabled={deleteLoading === propertyToDelete?._id}
              className="bg-red-600 hover:bg-red-700 focus:ring-red-600"
            >
              {deleteLoading === propertyToDelete?._id
                ? "Deleting..."
                : "Delete Property"}
            </AlertDialogAction>
          </AlertDialogFooter>
        </AlertDialogContent>
      </AlertDialog>
    </div>
  );
}
