"use client";

import { useState } from "react";
import { useRouter } from "next/navigation";
import { toast } from "sonner";
import { EnhancedPropertyForm } from "@/components/properties/PropertyForm";
import { Button } from "@/components/ui/button";
import { Building2, ArrowLeft } from "lucide-react";
import Link from "next/link";

export default function EnhancedNewPropertyPage() {
  const router = useRouter();
  const [isLoading, setIsLoading] = useState(false);

  const handlePropertySubmit = async (data: any) => {
    setIsLoading(true);
    try {
      const response = await fetch("/api/properties", {
        method: "POST",
        headers: {
          "Content-Type": "application/json",
        },
        body: JSON.stringify(data),
      });

      const result = await response.json();

      if (!response.ok) {
        // Extract detailed error information
        const errorDetails = result.details || result.error || result.message;
        const errorMessage =
          typeof errorDetails === "string"
            ? errorDetails
            : JSON.stringify(errorDetails);

        throw new Error(errorMessage || "Failed to create property");
      }

      // The API returns data in result.data
      const property = result.data;
      toast.success("Property created successfully!", {
        description: `${data.name} has been added to your portfolio.`,
        duration: 4000,
      });

      // Redirect to property details page
      router.push(`/dashboard/properties/${property._id}`);
    } catch (error) {
      const errorMessage =
        error instanceof Error
          ? error.message
          : "Failed to create property. Please try again.";

      toast.error("Unable to Create Property", {
        description: errorMessage,
        duration: 5000,
        action: {
          label: "Dismiss",
          onClick: () => {},
        },
      });
    } finally {
      setIsLoading(false);
    }
  };

  return (
    <div className="space-y-6 p-6">
      {/* Header */}
      <div className="flex items-center justify-between">
        <div className="space-y-1">
          <div>
            <h1 className="text-3xl font-bold tracking-tight flex items-center gap-3">
              <Building2 className="h-8 w-8" />
              Create New Property
            </h1>
            <p className="text-muted-foreground">
              Add a new property to your portfolio with comprehensive details
              and integrated unit management
            </p>
          </div>
        </div>
        <div className="flex items-center gap-2 border rounded-lg">
          <Link href="/dashboard/properties">
            <Button variant="ghost" size="sm">
              <ArrowLeft className="h-4 w-4 mr-2" />
              Back to Properties
            </Button>
          </Link>
        </div>
      </div>
      <EnhancedPropertyForm
        onSubmit={handlePropertySubmit}
        isLoading={isLoading}
        mode="create"
      />
    </div>
  );
}
