"use client";

import Link from "next/link";
import { toast } from "sonner";
import Image from "next/image";
import { Input } from "@/components/ui/input";
import { Button } from "@/components/ui/button";
import { useState, useEffect, useCallback } from "react";
import { Card, CardContent, CardHeader } from "@/components/ui/card";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select";
import {
  Building2,
  Plus,
  Search,
  Eye,
  MapPin,
  DollarSign,
  Bed,
  Bath,
  Square,
  RefreshCw,
  AlertCircle,
  CheckCircle,
  Grid3X3,
  List,
  Rows3,
  MoreHorizontal,
  X,
  Home,
} from "lucide-react";
import {
  DropdownMenu,
  DropdownMenuContent,
  DropdownMenuItem,
  DropdownMenuTrigger,
} from "@/components/ui/dropdown-menu";
import {
  Table,
  TableBody,
  TableCell,
  TableHead,
  TableHeader,
  TableRow,
} from "@/components/ui/table";
import { PropertyType } from "@/types";
import {
  propertyService,
  AvailableUnitResponse,
  PropertyQueryParams,
} from "@/lib/services/property.service";
import UnitStats from "@/components/properties/UnitStats";
import UnitDetailsModal from "@/components/properties/UnitDetailsModal";
import { getFeaturedImage, hasPropertyImages } from "@/lib/utils/image-utils";

interface UnitCardProps {
  unit: AvailableUnitResponse;
  onViewDetails: (propertyId: string, unitId: string) => void;
}

function UnitCard({ unit, onViewDetails }: UnitCardProps) {
  // Safe image handling using utility functions
  const hasImage = hasPropertyImages(unit);
  const featuredImage = getFeaturedImage(unit);

  return (
    <Card className="group hover:shadow-lg py-0 gap-0 transition-all duration-200 overflow-hidden border-0 shadow-sm">
      {/* Featured Image */}
      <div className="relative h-48 overflow-hidden bg-gray-100 dark:bg-gray-800">
        {hasImage ? (
          <Image
            src={featuredImage!}
            alt={`${unit.name} - Unit ${unit.unitNumber}`}
            fill
            className="object-cover group-hover:scale-105 transition-transform duration-300"
            sizes="(max-width: 768px) 100vw, (max-width: 1200px) 50vw, 33vw"
          />
        ) : (
          <div className="w-full h-full flex items-center justify-center bg-gradient-to-br from-gray-100 to-gray-200 dark:from-gray-800 dark:to-gray-900">
            <Building2 className="h-16 w-16 text-gray-400 dark:text-gray-600" />
          </div>
        )}

        {/* Status Badge */}
        <div className="absolute top-3 left-3">
          <span className="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium border bg-green-100 dark:bg-green-950/30 text-green-800 dark:text-green-300 border-green-200 dark:border-green-800">
            <CheckCircle className="h-3 w-3 mr-1" />
            Available
          </span>
        </div>

        {/* Unit Type Badge */}
        <div className="absolute top-3 right-3">
          <span className="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-blue-100 dark:bg-blue-950/30 text-blue-800 dark:text-blue-300 border border-blue-200 dark:border-blue-800">
            <Home className="h-3 w-3 mr-1" />
            <span>
              {unit.unitType.charAt(0).toUpperCase() + unit.unitType.slice(1)}
            </span>
          </span>
        </div>

        {/* Overlay Actions */}
        <div className="absolute inset-0 bg-black/0 group-hover:bg-black/20 transition-colors duration-200 flex items-center justify-center opacity-0 group-hover:opacity-100">
          <div className="flex space-x-2">
            <Link
              href={`/dashboard/properties/${unit._id}/units/${unit.unitId}`}
            >
              <Button
                size="sm"
                variant="secondary"
                className="bg-white/90 hover:bg-white text-gray-900"
              >
                <Eye className="h-4 w-4" />
              </Button>
            </Link>
          </div>
        </div>
      </div>

      {/* Card Content */}
      <CardContent className="p-4">
        {/* Property & Unit Name */}
        <div className="mb-3">
          <h3 className="font-semibold text-lg text-gray-900 dark:text-gray-100 mb-1 line-clamp-1">
            {unit.name} - Unit {unit.unitNumber}
          </h3>
          {unit.description && (
            <p className="text-sm text-gray-600 dark:text-gray-400 line-clamp-2">
              {unit.description}
            </p>
          )}
        </div>

        {/* Location */}
        <div className="flex items-center text-sm text-gray-600 dark:text-gray-400 mb-3">
          <MapPin className="h-4 w-4 mr-1 flex-shrink-0" />
          <span className="line-clamp-1">
            {unit.address.city}, {unit.address.state}
          </span>
        </div>

        {/* Unit Details */}
        <div className="flex items-center justify-between text-sm text-gray-600 dark:text-gray-400 mb-4">
          <div className="flex items-center space-x-4">
            <div className="flex items-center">
              <Bed className="h-4 w-4 mr-1" />
              <span>{unit.bedrooms}</span>
            </div>
            <div className="flex items-center">
              <Bath className="h-4 w-4 mr-1" />
              <span>{unit.bathrooms}</span>
            </div>
            <div className="flex items-center">
              <Square className="h-4 w-4 mr-1" />
              <span>{unit.squareFootage.toLocaleString()} ft²</span>
            </div>
          </div>
        </div>

        {/* Floor Info */}
        {unit.floor !== undefined && (
          <div className="flex items-center text-sm text-gray-600 dark:text-gray-400 mb-3">
            <span className="text-xs bg-gray-100 dark:bg-gray-800 px-2 py-1 rounded">
              Floor {unit.floor}
            </span>
          </div>
        )}

        {/* Rent & Actions */}
        <div className="flex items-center justify-between">
          <div className="flex items-center text-lg font-semibold text-gray-900 dark:text-gray-100">
            <DollarSign className="h-5 w-5 mr-1" />
            <span>{unit.rentAmount.toLocaleString()}</span>
            <span className="text-sm font-normal text-gray-600 dark:text-gray-400 ml-1">
              /month
            </span>
          </div>

          {/* Actions Menu */}
          <DropdownMenu>
            <DropdownMenuTrigger asChild>
              <Button variant="ghost" size="sm" className="h-8 w-8 p-0">
                <MoreHorizontal className="h-4 w-4" />
              </Button>
            </DropdownMenuTrigger>
            <DropdownMenuContent align="end">
              <DropdownMenuItem
                onClick={() => onViewDetails(unit._id, unit.unitId)}
              >
                <Eye className="h-4 w-4 mr-2" />
                View Unit Details
              </DropdownMenuItem>
              <DropdownMenuItem asChild>
                <Link href={`/dashboard/properties/${unit._id}`}>
                  <Building2 className="h-4 w-4 mr-2" />
                  View Property
                </Link>
              </DropdownMenuItem>
            </DropdownMenuContent>
          </DropdownMenu>
        </div>
      </CardContent>
    </Card>
  );
}

export default function AvailablePropertiesPage() {
  const [availableUnits, setAvailableUnits] = useState<AvailableUnitResponse[]>(
    []
  );
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState<string | null>(null);
  const [viewMode, setViewMode] = useState<"grid" | "rows" | "list">("grid");

  // Unit Details Dialog State
  const [selectedUnit, setSelectedUnit] = useState<{
    propertyId: string;
    unitId: string;
  } | null>(null);
  const [unitDetailsOpen, setUnitDetailsOpen] = useState(false);

  // Filter states
  const [filters, setFilters] = useState<PropertyQueryParams>({
    page: 1,
    limit: 12,
    sortBy: "createdAt",
    sortOrder: "desc",
    search: "",
  });

  const [pagination, setPagination] = useState({
    page: 1,
    limit: 12,
    total: 0,
    pages: 0,
    hasNext: false,
    hasPrev: false,
  });

  const fetchAvailableUnits = useCallback(async () => {
    try {
      setLoading(true);
      setError(null);
      const response = await propertyService.getAvailableProperties(filters);
      setAvailableUnits(response.data);
      setPagination(response.pagination);
    } catch (err) {
      const message =
        err instanceof Error ? err.message : "Failed to fetch available units";
      setError(message);
      toast.error(message);
    } finally {
      setLoading(false);
    }
  }, [filters]);

  useEffect(() => {
    fetchAvailableUnits();
  }, [fetchAvailableUnits]);

  const handleSearch = (search: string) => {
    setFilters((prev) => ({ ...prev, search, page: 1 }));
  };

  const handleFilterChange = (
    key: keyof PropertyQueryParams,
    value: string | number | undefined
  ) => {
    setFilters((prev) => ({ ...prev, [key]: value, page: 1 }));
  };

  const handlePageChange = (page: number) => {
    setFilters((prev) => ({ ...prev, page }));
  };

  // Handle opening unit details dialog
  const handleViewUnitDetails = (propertyId: string, unitId: string) => {
    setSelectedUnit({ propertyId, unitId });
    setUnitDetailsOpen(true);
  };

  // Handle unit updates
  const handleUnitUpdated = () => {
    fetchAvailableUnits();
  };

  // Handle unit deletion
  const handleUnitDeleted = () => {
    fetchAvailableUnits();
    setUnitDetailsOpen(false);
    setSelectedUnit(null);
  };

  const formatCurrency = (amount: number) => {
    return new Intl.NumberFormat("en-US", {
      style: "currency",
      currency: "USD",
    }).format(amount);
  };

  return (
    <div className="space-y-6">
      {/* Header */}
      <div className="flex items-center justify-between">
        <div>
          <h1 className="text-3xl font-bold tracking-tight">Available Units</h1>
          <p className="text-muted-foreground">
            Individual units currently available for rent
          </p>
        </div>
        <div className="flex items-center space-x-2">
          <Button
            variant="outline"
            size="sm"
            onClick={fetchAvailableUnits}
            disabled={loading}
          >
            <RefreshCw
              className={`h-4 w-4 mr-2 ${loading ? "animate-spin" : ""}`}
            />
            Refresh
          </Button>
          <Link href="/dashboard/properties/new">
            <Button>
              <Plus className="h-4 w-4 mr-2" />
              Add Property
            </Button>
          </Link>
        </div>
      </div>

      {/* Unit Statistics */}
      <UnitStats units={availableUnits} isLoading={loading} />

      {/* Available Units with Integrated Filters */}
      <Card>
        <CardHeader className="pb-4">
          {/* Main Header */}
          <div className="flex flex-col lg:flex-row lg:items-center lg:justify-between gap-4 mb-4">
            <div className="flex items-center gap-3">
              <div className="p-2 bg-green-50 dark:bg-green-900/30 rounded-lg border border-green-100 dark:border-green-800">
                <CheckCircle className="h-5 w-5 text-green-600 dark:text-green-400" />
              </div>
              <div>
                <h2 className="text-lg font-semibold text-gray-900 dark:text-gray-100">
                  Available Units
                </h2>
                <p className="text-sm text-gray-500 dark:text-gray-400">
                  Individual units currently available for rent
                </p>
              </div>
            </div>

            <div className="flex flex-col sm:flex-row items-stretch sm:items-center gap-3">
              {/* View Mode Toggle */}
              <div className="flex items-center border rounded-lg p-1 w-full sm:w-auto">
                <Button
                  variant={viewMode === "grid" ? "default" : "ghost"}
                  size="sm"
                  onClick={() => setViewMode("grid")}
                  className="h-8 flex-1 sm:flex-none sm:px-3"
                >
                  <Grid3X3 className="h-4 w-4" />
                  <span className="ml-1 sm:hidden">Grid</span>
                </Button>
                <Button
                  variant={viewMode === "rows" ? "default" : "ghost"}
                  size="sm"
                  onClick={() => setViewMode("rows")}
                  className="h-8 flex-1 sm:flex-none sm:px-3"
                >
                  <Rows3 className="h-4 w-4" />
                  <span className="ml-1 sm:hidden">Rows</span>
                </Button>
                <Button
                  variant={viewMode === "list" ? "default" : "ghost"}
                  size="sm"
                  onClick={() => setViewMode("list")}
                  className="h-8 flex-1 sm:flex-none sm:px-3"
                >
                  <List className="h-4 w-4" />
                  <span className="ml-1 sm:hidden">List</span>
                </Button>
              </div>
            </div>
          </div>

          {/* Integrated Filters Bar */}
          <div className="flex flex-col lg:flex-row lg:items-center gap-4 p-4 bg-gray-50/50 dark:bg-gray-800/50 rounded-lg border border-gray-200/60 dark:border-gray-700/60">
            {/* Search */}
            <div className="flex-1 min-w-0">
              <div className="relative">
                <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 h-4 w-4 text-gray-400 dark:text-gray-500" />
                <Input
                  placeholder="Search available units..."
                  value={filters.search || ""}
                  onChange={(e) => handleSearch(e.target.value)}
                  className="pl-10 h-10 border-gray-200 dark:border-gray-700 focus:border-blue-400 dark:focus:border-blue-500 focus:ring-1 focus:ring-blue-400 dark:focus:ring-blue-500 bg-white dark:bg-gray-800"
                />
              </div>
            </div>

            {/* Filter Controls */}
            <div className="flex flex-wrap items-center gap-3">
              {/* Property Type */}
              <Select
                value={filters.type || "all"}
                onValueChange={(value) =>
                  handleFilterChange(
                    "type",
                    value === "all" ? undefined : value
                  )
                }
              >
                <SelectTrigger className="w-[140px] h-10 border-gray-200 dark:border-gray-700 bg-white dark:bg-gray-800">
                  <SelectValue placeholder="All Types" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="all">All Types</SelectItem>
                  {Object.values(PropertyType).map((type) => (
                    <SelectItem key={type} value={type}>
                      {type.charAt(0).toUpperCase() + type.slice(1)}
                    </SelectItem>
                  ))}
                </SelectContent>
              </Select>

              {/* Bedrooms */}
              <Select
                value={filters.bedrooms?.toString() || "all"}
                onValueChange={(value) =>
                  handleFilterChange(
                    "bedrooms",
                    value === "all" ? undefined : parseInt(value)
                  )
                }
              >
                <SelectTrigger className="w-[120px] h-10 border-gray-200 dark:border-gray-700 bg-white dark:bg-gray-800">
                  <SelectValue placeholder="Bedrooms" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="all">Any Beds</SelectItem>
                  <SelectItem value="1">1 Bed</SelectItem>
                  <SelectItem value="2">2 Beds</SelectItem>
                  <SelectItem value="3">3 Beds</SelectItem>
                  <SelectItem value="4">4 Beds</SelectItem>
                  <SelectItem value="5">5+ Beds</SelectItem>
                </SelectContent>
              </Select>

              {/* Bathrooms */}
              <Select
                value={filters.bathrooms?.toString() || "all"}
                onValueChange={(value) =>
                  handleFilterChange(
                    "bathrooms",
                    value === "all" ? undefined : parseInt(value)
                  )
                }
              >
                <SelectTrigger className="w-[120px] h-10 border-gray-200 dark:border-gray-700 bg-white dark:bg-gray-800">
                  <SelectValue placeholder="Bathrooms" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="all">Any Baths</SelectItem>
                  <SelectItem value="1">1 Bath</SelectItem>
                  <SelectItem value="2">2 Baths</SelectItem>
                  <SelectItem value="3">3 Baths</SelectItem>
                  <SelectItem value="4">4+ Baths</SelectItem>
                </SelectContent>
              </Select>

              {/* Unit Type Filter */}
              <Select
                value={filters.unitType || "all"}
                onValueChange={(value) =>
                  handleFilterChange(
                    "unitType",
                    value === "all" ? undefined : value
                  )
                }
              >
                <SelectTrigger className="w-[140px] h-10 border-gray-200 dark:border-gray-700 bg-white dark:bg-gray-800">
                  <SelectValue placeholder="Unit Type" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="all">All Units</SelectItem>
                  <SelectItem value="apartment">Apartment</SelectItem>
                  <SelectItem value="studio">Studio</SelectItem>
                  <SelectItem value="penthouse">Penthouse</SelectItem>
                  <SelectItem value="loft">Loft</SelectItem>
                  <SelectItem value="room">Room</SelectItem>
                </SelectContent>
              </Select>

              {/* Clear Filters */}
              {(filters.search ||
                filters.type ||
                filters.bedrooms ||
                filters.bathrooms ||
                filters.minRent ||
                filters.maxRent ||
                filters.unitType) && (
                <Button
                  variant="ghost"
                  size="sm"
                  onClick={() =>
                    setFilters({
                      page: 1,
                      limit: 12,
                      sortBy: "createdAt",
                      sortOrder: "desc",
                      search: "",
                    })
                  }
                  className="h-10 px-3 text-gray-500 dark:text-gray-400 hover:text-gray-700 dark:hover:text-gray-300"
                >
                  <X className="h-4 w-4 mr-1" />
                  Clear
                </Button>
              )}
            </div>
          </div>
        </CardHeader>

        {/* Content */}
        <CardContent>
          {/* Content */}
          {loading ? (
            <div className="grid gap-6 md:grid-cols-2 lg:grid-cols-3">
              {Array.from({ length: 6 }).map((_, i) => (
                <Card key={i} className="overflow-hidden">
                  <div className="h-48 bg-muted animate-pulse" />
                  <CardContent className="p-4 space-y-3">
                    <div className="h-6 bg-muted rounded animate-pulse w-3/4" />
                    <div className="h-4 bg-muted rounded animate-pulse w-full" />
                    <div className="h-4 bg-muted rounded animate-pulse w-2/3" />
                    <div className="flex items-center justify-between">
                      <div className="h-5 bg-muted rounded animate-pulse w-20" />
                      <div className="h-8 bg-muted rounded animate-pulse w-8" />
                    </div>
                  </CardContent>
                </Card>
              ))}
            </div>
          ) : error ? (
            <div className="flex items-center justify-center py-16">
              <AlertCircle className="h-8 w-8 text-red-500 dark:text-red-400" />
              <span className="ml-2 text-red-600 dark:text-red-400">
                {error}
              </span>
              <Button
                variant="outline"
                size="sm"
                onClick={fetchAvailableUnits}
                className="ml-4"
              >
                Try Again
              </Button>
            </div>
          ) : availableUnits.length === 0 ? (
            <div className="flex items-center justify-center py-16">
              <Building2 className="h-8 w-8 text-gray-400 dark:text-gray-600" />
              <span className="ml-2 text-gray-600 dark:text-gray-400">
                No available units found
              </span>
              <Link href="/dashboard/properties/new">
                <Button className="ml-4">
                  <Plus className="h-4 w-4 mr-2" />
                  Add Property
                </Button>
              </Link>
            </div>
          ) : viewMode === "grid" ? (
            /* Grid View */
            <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 xl:grid-cols-4 gap-6">
              {availableUnits.map((unit) => (
                <UnitCard
                  key={`${unit._id}-${unit.unitId}`}
                  unit={unit}
                  onViewDetails={handleViewUnitDetails}
                />
              ))}
            </div>
          ) : viewMode === "rows" ? (
            /* Row View */
            <div className="space-y-3">
              {availableUnits.map((unit) => (
                <UnitCard
                  key={`${unit._id}-${unit.unitId}`}
                  unit={unit}
                  onViewDetails={handleViewUnitDetails}
                />
              ))}
            </div>
          ) : (
            /* List View - Table */
            <div className="border border-gray-200 dark:border-gray-700 rounded-lg overflow-hidden">
              <Table>
                <TableHeader className="bg-gray-50/50 dark:bg-gray-800/50">
                  <TableRow className="border-b border-gray-200 dark:border-gray-700">
                    <TableHead className="text-left font-medium text-gray-700 dark:text-gray-300 py-3 px-6">
                      Unit
                    </TableHead>
                    <TableHead className="text-left font-medium text-gray-700 dark:text-gray-300 py-3 px-4">
                      Property
                    </TableHead>
                    <TableHead className="text-left font-medium text-gray-700 dark:text-gray-300 py-3 px-4">
                      Location
                    </TableHead>
                    <TableHead className="text-left font-medium text-gray-700 dark:text-gray-300 py-3 px-4">
                      Details
                    </TableHead>
                    <TableHead className="text-left font-medium text-gray-700 dark:text-gray-300 py-3 px-4">
                      Rent
                    </TableHead>
                    <TableHead className="text-right font-medium text-gray-700 dark:text-gray-300 py-3 px-6">
                      Actions
                    </TableHead>
                  </TableRow>
                </TableHeader>
                <TableBody>
                  {availableUnits.map((unit, index) => (
                    <TableRow
                      key={`${unit._id}-${unit.unitId}`}
                      className={`border-b border-gray-100 dark:border-gray-800 hover:bg-gray-50/50 dark:hover:bg-gray-800/50 transition-colors ${
                        index % 2 === 0
                          ? "bg-white dark:bg-gray-900/20"
                          : "bg-gray-50/20 dark:bg-gray-800/20"
                      }`}
                    >
                      <TableCell className="py-4 px-6">
                        <div className="flex items-center space-x-3">
                          <div className="relative w-10 h-10 rounded-lg overflow-hidden bg-gray-100 dark:bg-gray-800 flex-shrink-0">
                            {hasPropertyImages(unit) ? (
                              <Image
                                src={getFeaturedImage(unit)!}
                                alt={`${unit.name} - Unit ${unit.unitNumber}`}
                                fill
                                className="object-cover"
                                sizes="40px"
                              />
                            ) : (
                              <div className="w-full h-full flex items-center justify-center bg-gradient-to-br from-gray-100 to-gray-200 dark:from-gray-800 dark:to-gray-900">
                                <Building2 className="h-5 w-5 text-gray-400 dark:text-gray-600" />
                              </div>
                            )}
                          </div>
                          <div className="min-w-0 flex-1">
                            <div className="font-medium text-gray-900 dark:text-gray-100">
                              <Link
                                href={`/dashboard/properties/${unit._id}/units/${unit.unitId}`}
                                className="hover:text-blue-600 dark:hover:text-blue-400 transition-colors"
                              >
                                Unit {unit.unitNumber}
                              </Link>
                            </div>
                            <div className="text-sm text-gray-500 dark:text-gray-400 truncate">
                              {unit.unitType.charAt(0).toUpperCase() +
                                unit.unitType.slice(1)}
                              {unit.floor !== undefined &&
                                ` • Floor ${unit.floor}`}
                            </div>
                          </div>
                        </div>
                      </TableCell>
                      <TableCell className="py-4 px-4">
                        <div className="flex flex-col space-y-1">
                          <div className="font-medium text-gray-900 dark:text-gray-100">
                            <Link
                              href={`/dashboard/properties/${unit._id}`}
                              className="hover:text-blue-600 dark:hover:text-blue-400 transition-colors"
                            >
                              {unit.name}
                            </Link>
                          </div>
                          <span className="text-xs text-gray-500 dark:text-gray-400">
                            {unit.type.charAt(0).toUpperCase() +
                              unit.type.slice(1)}
                          </span>
                        </div>
                      </TableCell>
                      <TableCell className="py-4 px-4">
                        <div className="flex flex-col space-y-1">
                          <div className="flex items-center text-sm text-gray-900 dark:text-gray-100">
                            <MapPin className="h-3 w-3 mr-1 text-gray-400 dark:text-gray-500" />
                            {unit.address.city}, {unit.address.state}
                          </div>
                          <div className="text-xs text-gray-500 dark:text-gray-400">
                            {unit.address.street}
                          </div>
                        </div>
                      </TableCell>
                      <TableCell className="py-4 px-4">
                        <div className="flex flex-col space-y-1">
                          <div className="flex items-center space-x-3 text-sm text-gray-900 dark:text-gray-100">
                            <div className="flex items-center">
                              <Bed className="h-3 w-3 mr-1 text-gray-400 dark:text-gray-500" />
                              {unit.bedrooms}
                            </div>
                            <div className="flex items-center">
                              <Bath className="h-3 w-3 mr-1 text-gray-400 dark:text-gray-500" />
                              {unit.bathrooms}
                            </div>
                          </div>
                          <div className="text-xs text-gray-500 dark:text-gray-400">
                            {unit.squareFootage.toLocaleString()} sq ft
                          </div>
                        </div>
                      </TableCell>
                      <TableCell className="py-4 px-4">
                        <div className="flex flex-col space-y-1">
                          <div className="font-medium text-gray-900 dark:text-gray-100">
                            {formatCurrency(unit.rentAmount)}
                          </div>
                          <div className="text-xs text-gray-500 dark:text-gray-400">
                            /month
                          </div>
                        </div>
                      </TableCell>
                      <TableCell className="text-right py-4 px-6">
                        <DropdownMenu>
                          <DropdownMenuTrigger asChild>
                            <Button
                              variant="ghost"
                              size="sm"
                              className="h-8 w-8 p-0 hover:bg-gray-100 transition-colors"
                            >
                              <MoreHorizontal className="h-4 w-4 text-gray-500" />
                            </Button>
                          </DropdownMenuTrigger>
                          <DropdownMenuContent align="end">
                            <DropdownMenuItem
                              onClick={() =>
                                handleViewUnitDetails(unit._id, unit.unitId)
                              }
                            >
                              <Eye className="h-4 w-4 mr-2" />
                              View Unit Details
                            </DropdownMenuItem>
                            <DropdownMenuItem asChild>
                              <Link href={`/dashboard/properties/${unit._id}`}>
                                <Building2 className="h-4 w-4 mr-2" />
                                View Property
                              </Link>
                            </DropdownMenuItem>
                          </DropdownMenuContent>
                        </DropdownMenu>
                      </TableCell>
                    </TableRow>
                  ))}
                </TableBody>
              </Table>
            </div>
          )}
        </CardContent>
      </Card>

      {/* Pagination */}
      {pagination.pages > 1 && (
        <div className="flex justify-center items-center space-x-2">
          <Button
            variant="outline"
            size="sm"
            onClick={() => handlePageChange(pagination.page - 1)}
            disabled={!pagination.hasPrev}
          >
            Previous
          </Button>
          <span className="text-sm text-gray-600 dark:text-gray-400">
            Page {pagination.page} of {pagination.pages}
          </span>
          <Button
            variant="outline"
            size="sm"
            onClick={() => handlePageChange(pagination.page + 1)}
            disabled={!pagination.hasNext}
          >
            Next
          </Button>
        </div>
      )}

      {/* Unit Details Modal */}
      {selectedUnit && (
        <UnitDetailsModal
          open={unitDetailsOpen}
          onOpenChange={setUnitDetailsOpen}
          propertyId={selectedUnit.propertyId}
          unitId={selectedUnit.unitId}
          onUnitUpdated={handleUnitUpdated}
          onUnitDeleted={handleUnitDeleted}
        />
      )}
    </div>
  );
}
