"use client";

import { useState, useEffect } from "react";
import { useRouter } from "next/navigation";
import { useSession } from "next-auth/react";
import { PaymentForm } from "@/components/forms/payment-form";
import { toast } from "sonner";
import {
  Card,
  CardContent,
  CardDescription,
  CardHeader,
  CardTitle,
} from "@/components/ui/card";
import { Skeleton } from "@/components/ui/skeleton";
import { CreditCard, ArrowLeft } from "lucide-react";
import { Button } from "@/components/ui/button";
import Link from "next/link";

interface Tenant {
  id: string;
  name: string;
  email: string;
}

interface Property {
  id: string;
  name: string;
  address: string;
}

interface Lease {
  id: string;
  propertyName: string;
  tenantName: string;
}

export default function NewPaymentPage() {
  const router = useRouter();
  const { data: session } = useSession();

  const [isLoading, setIsLoading] = useState(false);
  const [isDataLoading, setIsDataLoading] = useState(true);
  const [tenants, setTenants] = useState<Tenant[]>([]);
  const [properties, setProperties] = useState<Property[]>([]);
  const [leases, setLeases] = useState<Lease[]>([]);

  // Fetch required data for form
  useEffect(() => {
    const fetchData = async () => {
      try {
        setIsDataLoading(true);

        // Fetch tenants, properties, and leases in parallel
        const [tenantsRes, propertiesRes, leasesRes] = await Promise.all([
          fetch("/api/tenants"),
          fetch("/api/properties"),
          fetch("/api/leases"),
        ]);

        if (tenantsRes.ok) {
          const tenantsData = await tenantsRes.json();
          setTenants(
            tenantsData.data?.map((tenant: any) => ({
              id: tenant._id,
              name: `${tenant.userId.firstName} ${tenant.userId.lastName}`,
              email: tenant.userId.email,
            })) || []
          );
        }

        if (propertiesRes.ok) {
          const propertiesData = await propertiesRes.json();
          setProperties(
            propertiesData.data?.map((property: any) => ({
              id: property._id,
              name: property.name,
              address: `${property.address.street}, ${property.address.city}, ${property.address.state}`,
            })) || []
          );
        }

        if (leasesRes.ok) {
          const leasesData = await leasesRes.json();
          setLeases(
            leasesData.data?.map((lease: any) => ({
              id: lease._id,
              propertyName: lease.propertyId?.name || "Unknown Property",
              tenantName: lease.tenantId?.userId
                ? `${lease.tenantId.userId.firstName} ${lease.tenantId.userId.lastName}`
                : "Unknown Tenant",
            })) || []
          );
        }
      } catch (error) {
        toast.error(
          error instanceof Error
            ? error.message
            : "Failed to load form data. Please try again."
        );
      } finally {
        setIsDataLoading(false);
      }
    };

    if (session) {
      fetchData();
    }
  }, [session, toast]);

  const handleSubmit = async (data: any) => {
    try {
      setIsLoading(true);

      // Convert string date to Date object
      const paymentData = {
        ...data,
        dueDate: new Date(data.dueDate),
      };

      const response = await fetch("/api/payments", {
        method: "POST",
        headers: {
          "Content-Type": "application/json",
        },
        body: JSON.stringify(paymentData),
      });

      const result = await response.json();

      if (response.ok) {
        toast.success("Payment created successfully.");
        router.push("/dashboard/payments");
      } else {
        throw new Error(result.error || "Failed to create payment");
      }
    } catch (error) {
      toast.error(
        error instanceof Error
          ? error.message
          : "Failed to create payment. Please try again."
      );
    } finally {
      setIsLoading(false);
    }
  };

  const handleCancel = () => {
    router.push("/dashboard/payments");
  };

  if (isDataLoading) {
    return (
      <div className="space-y-6">
        {/* Header Skeleton */}
        <div className="flex items-center justify-between">
          <div className="space-y-2">
            <Skeleton className="h-8 w-48" />
            <Skeleton className="h-4 w-64" />
          </div>
          <Skeleton className="h-10 w-24" />
        </div>

        {/* Form Skeleton */}
        <Card>
          <CardHeader>
            <Skeleton className="h-6 w-40" />
            <Skeleton className="h-4 w-64" />
          </CardHeader>
          <CardContent className="space-y-4">
            {Array.from({ length: 6 }).map((_, i) => (
              <div key={i} className="space-y-2">
                <Skeleton className="h-4 w-24" />
                <Skeleton className="h-10 w-full" />
              </div>
            ))}
          </CardContent>
        </Card>
      </div>
    );
  }

  return (
    <div className="space-y-6">
      {/* Header */}
      <div className="flex items-center justify-between">
        <div className="space-y-1">
          <div className="flex items-center gap-2">
            <Link href="/dashboard/payments">
              <Button variant="ghost" size="sm">
                <ArrowLeft className="h-4 w-4 mr-2" />
                Back to Payments
              </Button>
            </Link>
          </div>
          <h1 className="text-3xl font-bold tracking-tight flex items-center gap-2">
            <CreditCard className="h-8 w-8" />
            Create Payment
          </h1>
          <p className="text-muted-foreground">
            Create a new payment record for rent, fees, or other charges
          </p>
        </div>
      </div>

      {/* Payment Form */}
      <PaymentForm
        onSubmit={handleSubmit}
        onCancel={handleCancel}
        isLoading={isLoading}
        tenants={tenants}
        properties={properties}
        leases={leases}
      />
    </div>
  );
}
