"use client";

import Link from "next/link";
import dynamic from "next/dynamic";
import { useRouter } from "next/navigation";
import { useSession } from "next-auth/react";
import { useCallback, useEffect, useState } from "react";
import {
  Card,
  CardContent,
  CardDescription,
  CardHeader,
  CardTitle,
} from "@/components/ui/card";
import { Badge } from "@/components/ui/badge";
import { Button } from "@/components/ui/button";
import { Progress } from "@/components/ui/progress";
import { ScrollArea } from "@/components/ui/scroll-area";
import { Alert, AlertDescription, AlertTitle } from "@/components/ui/alert";
import {
  AnalyticsCard,
  AnalyticsCardGrid,
} from "@/components/analytics/AnalyticsCard";
import { ResponsiveLayout } from "@/components/layout/responsive-layout";
import { DashboardSkeleton } from "@/components/ui/skeleton-layouts";
import { DashboardOverviewResponse } from "@/types/dashboard";
import {
  Building2,
  Home,
  Users,
  DollarSign,
  Wrench,
  AlertTriangle,
  Calendar,
  CreditCard,
  FileText,
  Activity,
  Target,
  BarChart3,
  PieChart,
  LineChart,
  RefreshCw,
  ChevronRight,
  UserCheck,
  ClipboardList,
} from "lucide-react";
import {
  PieChart as RechartsPieChart,
  Pie,
  Cell,
  XAxis,
  YAxis,
  CartesianGrid,
  Tooltip,
  ResponsiveContainer,
  Area,
  AreaChart,
} from "recharts";
import { UserRole } from "@/types";

const formatCurrency = (amount?: number, maximumFractionDigits = 0) => {
  const value =
    typeof amount === "number" && !Number.isNaN(amount) ? amount : 0;
  return new Intl.NumberFormat("en-US", {
    style: "currency",
    currency: "USD",
    minimumFractionDigits: 0,
    maximumFractionDigits,
  }).format(value);
};

const formatPercent = (value?: number) => {
  if (value === null || value === undefined || Number.isNaN(value)) {
    return "0%";
  }
  return `${value.toFixed(1)}%`;
};

const formatTimeAgo = (input?: Date | string | null) => {
  if (!input) {
    return "Just now";
  }

  const date = typeof input === "string" ? new Date(input) : input;
  if (!(date instanceof Date) || Number.isNaN(date.getTime())) {
    return "Just now";
  }

  const now = new Date();
  const diffInMinutes = Math.floor(
    (now.getTime() - date.getTime()) / (1000 * 60)
  );

  if (diffInMinutes < 60) {
    return `${diffInMinutes}m ago`;
  } else if (diffInMinutes < 1440) {
    return `${Math.floor(diffInMinutes / 60)}h ago`;
  } else {
    return `${Math.floor(diffInMinutes / 1440)}d ago`;
  }
};

const getActivityIcon = (type: string) => {
  switch (type) {
    case "payment":
      return DollarSign;
    case "maintenance":
      return Wrench;
    case "lease":
      return FileText;
    case "event":
      return Calendar;
    case "application":
      return UserCheck;
    default:
      return Activity;
  }
};

const getActivityColor = (type: string, status?: string) => {
  if (status === "completed") return "text-success";
  if (status === "pending") return "text-warning";
  if (status === "sent") return "text-info";
  if (status === "overdue" || status === "late") return "text-error";

  switch (type) {
    case "payment":
      return "text-success";
    case "maintenance":
      return "text-warning";
    case "lease":
      return "text-primary";
    case "event":
      return "text-info";
    case "application":
      return "text-info";
    default:
      return "text-muted-foreground";
  }
};

const getPriorityColor = (priority: string) => {
  switch (priority) {
    case "high":
      return "text-error";
    case "urgent":
      return "text-error";
    case "medium":
      return "text-warning";
    case "low":
      return "text-success";
    default:
      return "text-muted-foreground";
  }
};

const getAlertStyles = (type: string) => {
  switch (type) {
    case "payment":
      return {
        border: "border-yellow-400",
        bg: "bg-yellow-50/30",
        iconColor: "text-orange-500",
        textColor: "text-yellow-600",
        badgeBg: "bg-white/80",
      };
    case "maintenance":
      return {
        border: "border-red-400",
        bg: "bg-red-50/30",
        iconColor: "text-red-500",
        textColor: "text-red-600",
        badgeBg: "bg-white/80",
      };
    case "lease":
      return {
        border: "border-cyan-400",
        bg: "bg-cyan-50/30",
        iconColor: "text-cyan-500",
        textColor: "text-cyan-600",
        badgeBg: "bg-white/80",
      };
    default:
      return {
        border: "border-gray-300",
        bg: "bg-gray-50",
        iconColor: "text-gray-600",
        textColor: "text-gray-800",
        badgeBg: "bg-white/80",
      };
  }
};

export default function DashboardPage() {
  const { data: session, status } = useSession();
  const router = useRouter();
  const [dashboardData, setDashboardData] =
    useState<DashboardOverviewResponse | null>(null);
  const [isLoading, setIsLoading] = useState(true);
  const [isRefreshing, setIsRefreshing] = useState(false);
  const [error, setError] = useState<string | null>(null);
  const user = session?.user;
  const userRole = user?.role;
  const isTenant = userRole === UserRole.TENANT;
  const isDashboardAuthorized =
    userRole === UserRole.ADMIN || userRole === UserRole.MANAGER;

  const loadDashboardData = useCallback(
    async ({ isRefresh = false }: { isRefresh?: boolean } = {}) => {
      if (!isDashboardAuthorized) {
        setError(null);
        setDashboardData(null);
        if (isRefresh) {
          setIsRefreshing(false);
        } else {
          setIsLoading(false);
        }
        return;
      }

      if (isRefresh) {
        setIsRefreshing(true);
      } else {
        setIsLoading(true);
      }

      setError(null);

      try {
        const controller = new AbortController();
        const timeoutId = setTimeout(() => controller.abort(), 30000); // 30s timeout

        const response = await fetch("/api/dashboard/overview", {
          signal: controller.signal,
          headers: {
            "Cache-Control": isRefresh ? "no-cache" : "max-age=300",
          },
        });

        clearTimeout(timeoutId);

        if (!response.ok) {
          const errorText = await response.text().catch(() => "Unknown error");
          throw new Error(
            `Dashboard load failed: ${response.status} ${errorText}`
          );
        }

        const payload = await response.json();
        const data = (payload?.data ?? payload) as DashboardOverviewResponse;

        if (!data?.overview) {
          throw new Error("Invalid dashboard response format");
        }

        setDashboardData(data);
      } catch (err) {
        // Ignore errors from aborted requests
        if (err instanceof Error && err.name === "AbortError") {
          return;
        }

        setError(
          err instanceof Error
            ? err.message
            : "Unable to load dashboard overview"
        );
      } finally {
        if (isRefresh) {
          setIsRefreshing(false);
        } else {
          setIsLoading(false);
        }
      }
    },
    [isDashboardAuthorized]
  );

  useEffect(() => {
    if (status !== "authenticated") {
      return;
    }

    if (!isDashboardAuthorized) {
      setIsLoading(false);
      return;
    }

    loadDashboardData();
  }, [status, isDashboardAuthorized, loadDashboardData]);

  // Get greeting based on time of day
  const getGreeting = () => {
    const hour = new Date().getHours();
    if (hour < 12) return "Good morning";
    if (hour < 17) return "Good afternoon";
    return "Good evening";
  };

  const handleRefresh = useCallback(() => {
    loadDashboardData({ isRefresh: true });
  }, [loadDashboardData]);

  const handleExport = useCallback(async () => {
    try {
      setIsRefreshing(true);

      const controller = new AbortController();
      const timeoutId = setTimeout(() => controller.abort(), 60000); // 60s timeout for export

      const response = await fetch("/api/dashboard/export", {
        method: "POST",
        headers: { "Content-Type": "application/json" },
        signal: controller.signal,
        body: JSON.stringify({
          format: "json",
          includeDetails: false,
          dateRange: {
            start: new Date(new Date().getFullYear(), 0, 1).toISOString(),
            end: new Date().toISOString(),
          },
        }),
      });

      clearTimeout(timeoutId);

      if (!response.ok) {
        const errorText = await response.text().catch(() => "Unknown error");
        throw new Error(`Export failed: ${response.status} ${errorText}`);
      }

      const result = await response.json();

      if (!result.data) {
        throw new Error("Invalid export data received");
      }

      const blob = new Blob([JSON.stringify(result.data, null, 2)], {
        type: "application/json",
      });

      const url = URL.createObjectURL(blob);
      const link = document.createElement("a");
      link.href = url;
      link.download = `dashboard-export-${
        new Date().toISOString().split("T")[0]
      }.json`;

      document.body.appendChild(link);
      link.click();
      document.body.removeChild(link);
      URL.revokeObjectURL(url);
    } catch (error) {
      // Ignore errors from aborted requests
      if (error instanceof Error && error.name === "AbortError") {
        return;
      }

      setError(error instanceof Error ? error.message : "Export failed");
    } finally {
      setIsRefreshing(false);
    }
  }, []);

  // Navigation handlers for alert cards
  const handleAlertClick = useCallback(
    (alertType: string) => {
      switch (alertType) {
        case "payment":
          router.push("/dashboard/payments?filter=overdue");
          break;
        case "maintenance":
          router.push("/dashboard/maintenance?filter=urgent");
          break;
        case "lease":
          router.push("/dashboard/leases/expiring");
          break;
        case "tenant":
          router.push("/dashboard/tenants/applications");
          break;
        default:
          break;
      }
    },
    [router]
  );

  // Navigation handler for upcoming tasks
  const handleTaskClick = useCallback(
    (task: any) => {
      const taskId = task.id;

      if (taskId.startsWith("lease-")) {
        router.push("/dashboard/leases/expiring");
      } else if (taskId.startsWith("maintenance-")) {
        const maintenanceId = taskId.replace("maintenance-", "");
        router.push(`/dashboard/maintenance/${maintenanceId}`);
      } else if (task.type === "lease_renewal") {
        router.push("/dashboard/leases/expiring");
      } else if (task.type === "maintenance") {
        router.push("/dashboard/maintenance");
      } else {
        router.push("/dashboard/calendar");
      }
    },
    [router]
  );

  if (isTenant) {
    // Import the TenantDashboard component dynamically to avoid SSR issues
    const TenantDashboard = dynamic(
      () => import("@/components/tenant/TenantDashboard"),
      {
        ssr: false,
        loading: () => (
          <ResponsiveLayout>
            <div className="space-y-6">
              <div className="flex items-center justify-between">
                <div>
                  <div className="h-8 w-64 bg-gray-200 rounded animate-pulse mb-2" />
                  <div className="h-4 w-48 bg-gray-200 rounded animate-pulse" />
                </div>
              </div>
              <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-4">
                {[...Array(4)].map((_, i) => (
                  <Card key={i}>
                    <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                      <div className="h-4 w-24 bg-gray-200 rounded animate-pulse" />
                      <div className="h-4 w-4 bg-gray-200 rounded animate-pulse" />
                    </CardHeader>
                    <CardContent>
                      <div className="h-8 w-16 bg-gray-200 rounded animate-pulse mb-1" />
                      <div className="h-3 w-20 bg-gray-200 rounded animate-pulse" />
                    </CardContent>
                  </Card>
                ))}
              </div>
            </div>
          </ResponsiveLayout>
        ),
      }
    );

    return (
      <ResponsiveLayout>
        <TenantDashboard />
      </ResponsiveLayout>
    );
  }

  if (!dashboardData && isLoading) {
    return (
      <ResponsiveLayout>
        <DashboardSkeleton />
      </ResponsiveLayout>
    );
  }

  if (!dashboardData && error) {
    return (
      <ResponsiveLayout className="space-y-6">
        <Alert variant="destructive">
          <AlertTriangle className="h-4 w-4" />
          <AlertTitle>Dashboard unavailable</AlertTitle>
          <AlertDescription>
            {error} — please try refreshing the page.
          </AlertDescription>
        </Alert>
        <Button onClick={() => loadDashboardData()} className="w-min">
          Retry
        </Button>
      </ResponsiveLayout>
    );
  }

  const overview = dashboardData?.overview;
  const maintenance = overview?.maintenanceRequests;
  const payments = overview?.payments;
  const revenueTrend = dashboardData?.trends?.revenue ?? [];
  const propertyDistribution = dashboardData?.propertyTypes ?? [];
  const alerts = dashboardData?.alerts ?? [];
  const recentActivities = dashboardData?.recentActivities ?? [];
  const upcomingTasks = dashboardData?.upcomingTasks ?? [];

  const vacantUnits =
    (overview?.totalUnits ?? 0) - (overview?.occupiedUnits ?? 0);
  const vacancyRate = overview?.totalUnits
    ? (vacantUnits / overview.totalUnits) * 100
    : 0;

  const latestTrendPoint =
    revenueTrend.length > 0 ? revenueTrend[revenueTrend.length - 1] : null;
  const currentRevenueValue =
    latestTrendPoint?.totalRevenue ?? overview?.monthlyRevenue ?? 0;
  const currentExpenseValue = latestTrendPoint?.totalExpenses ?? 0;

  // Manager/Admin Dashboard
  return (
    <ResponsiveLayout className="space-y-6">
      {/* Header Section */}
      <div className="flex flex-col lg:flex-row lg:items-center lg:justify-between gap-4">
        <div>
          <h1 className="text-3xl font-bold tracking-tight">
            {getGreeting()}, {user?.firstName}!
          </h1>
          <p className="text-muted-foreground">
            Here&apos;s what&apos;s happening with your property portfolio
          </p>
        </div>
        <div className="flex items-center gap-3">
          <Button
            variant="outline"
            size="sm"
            onClick={handleRefresh}
            disabled={isRefreshing}
            className="gap-2"
          >
            <RefreshCw
              className={`h-4 w-4 ${isRefreshing ? "animate-spin" : ""}`}
            />
            Refresh
          </Button>
          <Link href="/dashboard/analytics">
            <Button size="sm" className="gap-2">
              <BarChart3 className="h-4 w-4" />
              Analytics
            </Button>
          </Link>
        </div>
      </div>

      {error && (
        <Alert variant="destructive">
          <AlertTriangle className="h-4 w-4" />
          <AlertTitle>Some data may be out of date</AlertTitle>
          <AlertDescription>
            {error} — try refreshing to fetch the latest metrics.
          </AlertDescription>
        </Alert>
      )}

      {/* Alerts Section - Always show the main 3 alerts */}
      <div className="grid gap-4  md:grid-cols-3">
        {alerts?.slice(0, 3).map((alert) => {
          const styles = getAlertStyles(alert.type);
          return (
            <div
              key={alert.id}
              className={`rounded-xl border-l-4 border-1 ${styles.border} ${styles.bg} px-4 py-2 cursor-pointer hover:shadow-lg transition-all duration-200 hover:scale-[1.02]`}
              onClick={() => handleAlertClick(alert.type)}
            >
              <div className="flex items-start justify-between gap-3">
                <div className="flex-1">
                  <div className="flex items-center gap-2 mb-1">
                    <AlertTriangle className={`h-4 w-4 ${styles.iconColor}`} />
                    <h4 className={`font-medium text-sm ${styles.textColor}`}>
                      {alert.title}
                    </h4>
                    <span
                      className={`text-md font-bold ${styles.textColor} ml-auto`}
                    >
                      {alert.count}
                    </span>
                  </div>
                  <p className={`text-xs ${styles.textColor} opacity-80`}>
                    {alert.message}
                  </p>
                </div>
                <ChevronRight
                  className={`h-4 w-4 ${styles.iconColor} shrink-0 mt-1`}
                />
              </div>
            </div>
          );
        })}
      </div>

      {/* Main KPI Cards */}
      <AnalyticsCardGrid>
        <AnalyticsCard
          title="Total Properties"
          value={overview?.totalProperties ?? 0}
          description="Active properties in portfolio"
          icon={Building2}
          iconColor="primary"
        />
        <AnalyticsCard
          title="Occupancy Rate"
          value={formatPercent(overview?.occupancyRate)}
          description={`${overview?.occupiedUnits ?? 0} of ${
            overview?.totalUnits ?? 0
          } units occupied`}
          icon={Home}
          iconColor="success"
        />
        <AnalyticsCard
          title="Monthly Revenue"
          value={formatCurrency(overview?.monthlyRevenue)}
          description="Current month collected"
          icon={DollarSign}
          iconColor="success"
        />
        <AnalyticsCard
          title="Collection Rate"
          value={formatPercent(overview?.collectionRate)}
          description="Payment collection efficiency"
          icon={Target}
          iconColor="info"
        />
        <AnalyticsCard
          title="Active Tenants"
          value={overview?.activeTenants ?? 0}
          description={`${
            overview?.pendingApplications ?? 0
          } pending applications`}
          icon={Users}
          iconColor="primary"
        />
        <AnalyticsCard
          title="Maintenance Requests"
          value={maintenance?.open ?? 0}
          description={`${maintenance?.urgent ?? 0} urgent`}
          icon={Wrench}
          iconColor="warning"
        />
        <AnalyticsCard
          title="Vacant Units"
          value={vacantUnits}
          description={`${formatPercent(vacancyRate)} vacancy rate`}
          icon={Home}
          iconColor="error"
        />
        <AnalyticsCard
          title="Average Rent"
          value={formatCurrency(overview?.averageRent)}
          description="Per unit monthly average"
          icon={DollarSign}
          iconColor="success"
        />
        <AnalyticsCard
          title="Lease Renewals"
          value={overview?.expiringLeases ?? 0}
          description="Due in next 30 days"
          icon={FileText}
          iconColor="warning"
        />
        <AnalyticsCard
          title="Recent Events"
          value={recentActivities.length}
          description={`${
            recentActivities.filter(
              (activity) =>
                activity.priority === "high" || activity.priority === "urgent"
            ).length
          } urgent activities`}
          icon={Activity}
          iconColor="info"
        />
      </AnalyticsCardGrid>

      {/* Main Content Grid */}
      <div className="grid gap-6 lg:grid-cols-3">
        {/* Left Column - Charts and Analytics */}
        <div className="lg:col-span-2 space-y-6">
          {/* Revenue & Expenses Trends */}
          <Card className="hover-lift">
            <CardHeader>
              <div className="flex items-center justify-between">
                <div>
                  <CardTitle className="flex items-center gap-2">
                    <LineChart className="h-5 w-5 text-primary" />
                    Revenue & Expenses
                  </CardTitle>
                  <CardDescription className="text-muted-foreground">
                    (+5.2%) revenue growth this year
                  </CardDescription>
                </div>
                <div className="flex items-center gap-2">
                  <select className="text-sm border rounded px-2 py-1 bg-background">
                    <option>2024</option>
                    <option>2023</option>
                    <option>2022</option>
                  </select>
                </div>
              </div>
            </CardHeader>
            <CardContent>
              {/* Legend */}
              <div className="flex items-center gap-6 mb-6">
                <div className="flex items-center gap-2">
                  <div className="w-3 h-3 rounded-full bg-[#10b981]"></div>
                  <span className="text-sm text-muted-foreground">
                    Total Revenue
                  </span>
                  <span className="text-lg font-semibold">
                    {formatCurrency(currentRevenueValue)}
                  </span>
                </div>
                <div className="flex items-center gap-2">
                  <div className="w-3 h-3 rounded-full bg-[#f59e0b]"></div>
                  <span className="text-sm text-muted-foreground">
                    Total Expenses
                  </span>
                  <span className="text-lg font-semibold">
                    {formatCurrency(currentExpenseValue)}
                  </span>
                </div>
              </div>

              <ResponsiveContainer width="100%" height={300}>
                <AreaChart data={revenueTrend}>
                  <defs>
                    <linearGradient
                      id="incomeGradient"
                      x1="0"
                      y1="0"
                      x2="0"
                      y2="1"
                    >
                      <stop offset="5%" stopColor="#10b981" stopOpacity={0.3} />
                      <stop
                        offset="95%"
                        stopColor="#10b981"
                        stopOpacity={0.05}
                      />
                    </linearGradient>
                    <linearGradient
                      id="expenseGradient"
                      x1="0"
                      y1="0"
                      x2="0"
                      y2="1"
                    >
                      <stop offset="5%" stopColor="#f59e0b" stopOpacity={0.3} />
                      <stop
                        offset="95%"
                        stopColor="#f59e0b"
                        stopOpacity={0.05}
                      />
                    </linearGradient>
                  </defs>
                  <CartesianGrid strokeDasharray="3 3" stroke="#f1f5f9" />
                  <XAxis
                    dataKey="month"
                    axisLine={false}
                    tickLine={false}
                    tick={{ fontSize: 12, fill: "#64748b" }}
                  />
                  <YAxis
                    axisLine={false}
                    tickLine={false}
                    tick={{ fontSize: 12, fill: "#64748b" }}
                    tickFormatter={(value) => `${value / 1000}k`}
                  />
                  <Tooltip
                    content={({ active, payload, label }) => {
                      if (active && payload && payload.length) {
                        return (
                          <div className="bg-white p-3 border rounded-lg shadow-lg">
                            <p className="text-sm font-medium mb-2">{label}</p>
                            {payload.map((entry, index) => (
                              <div
                                key={index}
                                className="flex items-center gap-2 text-sm"
                              >
                                <div
                                  className="w-2 h-2 rounded-full"
                                  style={{ backgroundColor: entry.color }}
                                ></div>
                                <span className="text-muted-foreground">
                                  {entry.name}:
                                </span>
                                <span className="font-semibold">
                                  {formatCurrency(entry.value as number)}
                                </span>
                              </div>
                            ))}
                          </div>
                        );
                      }
                      return null;
                    }}
                  />
                  <Area
                    type="monotone"
                    dataKey="totalRevenue"
                    stroke="#10b981"
                    strokeWidth={2}
                    fill="url(#incomeGradient)"
                    name="Total Revenue"
                    dot={false}
                  />
                  <Area
                    type="monotone"
                    dataKey="totalExpenses"
                    stroke="#f59e0b"
                    strokeWidth={2}
                    fill="url(#expenseGradient)"
                    name="Total Expenses"
                    dot={false}
                  />
                </AreaChart>
              </ResponsiveContainer>
            </CardContent>
          </Card>

          {/* Property Type Distribution and Payment Status */}
          <div className="grid gap-6 md:grid-cols-2">
            {/* Property Distribution */}
            <Card className="hover-lift">
              <CardHeader>
                <CardTitle className="flex items-center gap-2">
                  <PieChart className="h-5 w-5 text-primary" />
                  Property Distribution
                </CardTitle>
              </CardHeader>
              <CardContent>
                <div className="relative">
                  <ResponsiveContainer width="100%" height={250}>
                    <RechartsPieChart>
                      <Pie
                        data={propertyDistribution as any}
                        cx="50%"
                        cy="50%"
                        innerRadius={60}
                        outerRadius={90}
                        paddingAngle={2}
                        dataKey="value"
                        stroke="none"
                      >
                        {propertyDistribution.map((entry, index) => (
                          <Cell key={`cell-${index}`} fill={entry.color} />
                        ))}
                      </Pie>
                      <Tooltip
                        content={({ active, payload }) => {
                          if (active && payload && payload.length) {
                            const data = payload[0].payload;
                            const percentage = overview?.totalProperties
                              ? Math.round(
                                  (data.value / overview.totalProperties) * 100
                                )
                              : 0;
                            return (
                              <div className="bg-white p-2 border rounded shadow-lg">
                                <p className="text-sm font-medium">
                                  {data.name}: {data.value} ({percentage}%)
                                </p>
                              </div>
                            );
                          }
                          return null;
                        }}
                      />
                    </RechartsPieChart>
                  </ResponsiveContainer>

                  {/* Center Total */}
                  <div className="absolute inset-0 flex items-center justify-center">
                    <div className="text-center">
                      <div className="text-xs text-muted-foreground">Total</div>
                      <div className="text-2xl font-bold">
                        {overview?.totalProperties ?? 0}
                      </div>
                    </div>
                  </div>
                </div>

                {/* Legend */}
                <div className="flex justify-center gap-6 mt-4">
                  {propertyDistribution.map((entry, index) => (
                    <div key={index} className="flex items-center gap-2">
                      <div
                        className="w-3 h-3 rounded-full"
                        style={{ backgroundColor: entry.color }}
                      ></div>
                      <span className="text-sm text-muted-foreground">
                        {entry.name}
                      </span>
                    </div>
                  ))}
                </div>
              </CardContent>
            </Card>

            {/* Payment Status */}
            <Card className="hover-lift">
              <CardHeader>
                <CardTitle className="flex items-center gap-2">
                  <CreditCard className="h-5 w-5 text-success" />
                  Payment Status
                </CardTitle>
                <CardDescription>
                  Current month payment collection
                </CardDescription>
              </CardHeader>
              <CardContent className="space-y-4">
                <div className="space-y-3">
                  <div className="flex items-center justify-between">
                    <span className="text-sm font-medium">Collected</span>
                    <span className="text-sm font-bold text-success">
                      {formatCurrency(payments?.collected)}
                    </span>
                  </div>
                  <Progress
                    value={
                      payments?.totalDue
                        ? (payments.collected / payments.totalDue) * 100
                        : 0
                    }
                    className="h-2"
                  />
                </div>

                <div className="grid grid-cols-2 gap-4 pt-2">
                  <div className="text-center p-3 bg-warning/10 rounded-lg">
                    <div className="text-lg font-bold text-warning">
                      {formatCurrency(payments?.pending)}
                    </div>
                    <div className="text-xs text-muted-foreground">Pending</div>
                  </div>
                  <div className="text-center p-3 bg-error/10 rounded-lg">
                    <div className="text-lg font-bold text-error">
                      {formatCurrency(payments?.overdue)}
                    </div>
                    <div className="text-xs text-muted-foreground">Overdue</div>
                  </div>
                </div>
              </CardContent>
            </Card>
          </div>
        </div>

        {/* Right Column - Activities and Tasks */}
        <div className="space-y-6">
          {/* Recent Activities */}
          <Card className="hover-lift">
            <CardHeader>
              <div className="flex items-center justify-between">
                <CardTitle className="flex items-center gap-2">
                  <Activity className="h-5 w-5 text-primary" />
                  Recent Activity
                </CardTitle>
              </div>
              <CardDescription>
                Latest updates across your portfolio
              </CardDescription>
            </CardHeader>
            <CardContent>
              <ScrollArea className="h-80">
                <div className="space-y-4">
                  {recentActivities.length === 0 && (
                    <p className="text-sm text-muted-foreground">
                      No recent activity yet.
                    </p>
                  )}
                  {recentActivities.map((activity) => {
                    const IconComponent = getActivityIcon(activity.type);
                    return (
                      <div
                        key={activity.id}
                        className="flex items-start gap-3 p-3 rounded-lg hover:bg-muted/50 transition-colors"
                      >
                        <div
                          className={`p-2 rounded-lg bg-muted ${getActivityColor(
                            activity.type,
                            activity.status
                          )}`}
                        >
                          <IconComponent className="h-4 w-4" />
                        </div>
                        <div className="flex-1 min-w-0">
                          <p className="text-sm font-medium leading-tight">
                            {activity.description}
                          </p>
                          <div className="flex items-center gap-2 mt-1">
                            <span className="text-xs text-muted-foreground">
                              {formatTimeAgo(activity.timestamp)}
                            </span>
                            {activity.amount && (
                              <Badge variant="outline" className="text-xs">
                                {formatCurrency(activity.amount)}
                              </Badge>
                            )}
                            {activity.priority && (
                              <Badge
                                variant="outline"
                                className={`text-xs ${getPriorityColor(
                                  activity.priority
                                )}`}
                              >
                                {activity.priority}
                              </Badge>
                            )}
                          </div>
                        </div>
                      </div>
                    );
                  })}
                </div>
              </ScrollArea>
            </CardContent>
          </Card>

          {/* Upcoming Tasks */}
          <Card className="hover-lift">
            <CardHeader>
              <div className="flex items-center justify-between">
                <CardTitle className="flex items-center gap-2">
                  <ClipboardList className="h-5 w-5 text-warning" />
                  Upcoming Tasks
                </CardTitle>
              </div>
              <CardDescription>
                Important tasks requiring attention
              </CardDescription>
            </CardHeader>
            <CardContent>
              <div className="space-y-3">
                {upcomingTasks.length === 0 && (
                  <p className="text-sm text-muted-foreground">
                    Nothing scheduled right now.
                  </p>
                )}
                {upcomingTasks.slice(0, 4).map((task) => {
                  const dueDate = new Date(task.dueDate);
                  const dueDateLabel = Number.isNaN(dueDate.getTime())
                    ? "Date pending"
                    : dueDate.toLocaleDateString();

                  return (
                    <div
                      key={task.id}
                      className="flex items-center gap-3 p-3 rounded-lg border border-border/50 hover:border-border hover:shadow-sm cursor-pointer transition-all"
                      onClick={() => handleTaskClick(task)}
                    >
                      <div className="flex-1 min-w-0">
                        <p className="text-sm font-medium leading-tight">
                          {task.title}
                        </p>
                        <div className="flex items-center gap-2 mt-1">
                          <Calendar className="h-3 w-3 text-muted-foreground" />
                          <span className="text-xs text-muted-foreground">
                            {dueDateLabel}
                          </span>
                          <Badge
                            variant="outline"
                            className={`text-xs ${getPriorityColor(
                              task.priority
                            )}`}
                          >
                            {task.priority}
                          </Badge>
                        </div>
                      </div>
                      <Button variant="ghost" size="sm" className="h-8 w-8 p-0">
                        <ChevronRight className="h-4 w-4" />
                      </Button>
                    </div>
                  );
                })}
              </div>
            </CardContent>
          </Card>
        </div>
      </div>
    </ResponsiveLayout>
  );
}
