"use client";

import Link from "next/link";
import { useState, useEffect } from "react";
import { useRouter } from "next/navigation";
import { useSession } from "next-auth/react";
import {
  Card,
  CardContent,
  CardDescription,
  CardHeader,
  CardTitle,
} from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { Input } from "@/components/ui/input";
import { Checkbox } from "@/components/ui/checkbox";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select";
import {
  Table,
  TableBody,
  TableCell,
  TableHead,
  TableHeader,
  TableRow,
} from "@/components/ui/table";

import {
  Wrench,
  Plus,
  Search,
  Calendar,
  AlertTriangle,
  CheckCircle,
  Clock,
  User,
  Building2,
  DollarSign,
  Play,
  Square,
  X,
  Image,
  TrendingUp,
  Filter,
  ArrowUpDown,
  ArrowUp,
  ArrowDown,
  Grid3X3,
  List,
} from "lucide-react";
import {
  MaintenancePriority,
  MaintenanceStatus,
  UserRole,
} from "@/types";
import { toast } from "sonner";
import { MaintenanceStatusManager } from "@/components/maintenance/maintenance-status-manager";
import { useMaintenanceStaff } from "@/hooks/use-maintenance-staff";

interface MaintenanceRequestWithPopulated {
  _id: string;
  title: string;
  description: string;
  category: string;
  priority: MaintenancePriority;
  status: MaintenanceStatus;
  estimatedCost?: number;
  actualCost?: number;
  scheduledDate?: Date;
  completedDate?: Date;
  createdAt: Date;
  updatedAt: Date;
  images: string[];
  property: {
    name: string;
    address: {
      street: string;
      city: string;
      state: string;
      zipCode: string;
    };
  };
  unit?: {
    _id: string;
    unitNumber: string;
    unitType: string;
    status: string;
  };
  tenant: {
    user: {
      firstName: string;
      lastName: string;
      email: string;
      phone: string;
    };
  };
  assignedTo?: {
    firstName: string;
    lastName: string;
    email: string;
    phone: string;
  };
}

export default function MaintenancePage() {
  const router = useRouter();
  const { data: session } = useSession();
  const [searchTerm, setSearchTerm] = useState("");
  const [statusFilter, setStatusFilter] = useState<string>("ALL_STATUSES");
  const [priorityFilter, setPriorityFilter] =
    useState<string>("ALL_PRIORITIES");
  const [categoryFilter, setCategoryFilter] =
    useState<string>("ALL_CATEGORIES");
  const [dateFilter, setDateFilter] = useState<string>("ALL_DATES");
  const [assignedFilter, setAssignedFilter] = useState<string>("ALL_ASSIGNED");
  const [overdueFilter, setOverdueFilter] = useState<boolean>(false);
  const [sortBy, setSortBy] = useState<string>("createdAt");
  const [sortOrder, setSortOrder] = useState<"asc" | "desc">("desc");
  const [viewMode, setViewMode] = useState<"table" | "cards">("table");
  const [maintenanceRequests, setMaintenanceRequests] = useState<
    MaintenanceRequestWithPopulated[]
  >([]);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState<string | null>(null);

  // Fetch available maintenance staff for assignment
  const { staff: availableStaff } = useMaintenanceStaff();

  // Redirect tenants to their my-requests page
  useEffect(() => {
    if (session?.user?.role === UserRole.TENANT) {
      toast.info("Redirecting to your maintenance requests...");
      router.push("/dashboard/maintenance/my-requests");
    }
  }, [session, router]);

  useEffect(() => {
    // Only fetch if not a tenant (tenants are redirected)
    if (session && session.user?.role !== UserRole.TENANT) {
      fetchMaintenanceRequests();
    }
  }, [session]);

  // Handle status updates
  const handleStatusUpdate = (
    requestId: string,
    newStatus: MaintenanceStatus
  ) => {
    setMaintenanceRequests((prev) =>
      prev.map((request) =>
        request._id === requestId ? { ...request, status: newStatus } : request
      )
    );
  };

  // Handle request updates (refresh data)
  const handleRequestUpdate = () => {
    fetchMaintenanceRequests();
  };

  const fetchMaintenanceRequests = async () => {
    try {
      setLoading(true);
      setError(null);

      const response = await fetch("/api/maintenance?limit=100");

      if (!response.ok) {
        throw new Error("Failed to fetch maintenance requests");
      }

      const data = await response.json();
      setMaintenanceRequests(data.data || []);
    } catch (error: any) {
      const errorMessage =
        error.message || "Failed to load maintenance requests";
      setError(errorMessage);
      toast.error(errorMessage);
    } finally {
      setLoading(false);
    }
  };

  const getDaysOverdue = (
    createdAt: Date | string | undefined,
    priority: MaintenancePriority | undefined,
    status: MaintenanceStatus | undefined
  ) => {
    // Return 0 if any required data is missing
    if (!createdAt || !priority || !status) {
      return 0;
    }

    if (
      status === MaintenanceStatus.COMPLETED ||
      status === MaintenanceStatus.CANCELLED
    ) {
      return 0;
    }

    try {
      const now = new Date();
      const createdDate =
        typeof createdAt === "string" ? new Date(createdAt) : createdAt;

      // Check if date is valid
      if (isNaN(createdDate.getTime())) {
        return 0;
      }

      const diffTime = now.getTime() - createdDate.getTime();
      const diffDays = Math.ceil(diffTime / (1000 * 60 * 60 * 24));

      // Define SLA days based on priority
      const slaDays = {
        [MaintenancePriority.EMERGENCY]: 1,
        [MaintenancePriority.HIGH]: 3,
        [MaintenancePriority.MEDIUM]: 7,
        [MaintenancePriority.LOW]: 14,
      };

      return Math.max(0, diffDays - (slaDays[priority] || 0));
    } catch (error) {
      // Silently handle calculation errors
      return 0;
    }
  };

  const filteredAndSortedRequests = maintenanceRequests
    .filter((request: MaintenanceRequestWithPopulated) => {
      const matchesSearch =
        (request?.title?.toLowerCase() || "").includes(
          searchTerm.toLowerCase()
        ) ||
        (request?.description?.toLowerCase() || "").includes(
          searchTerm.toLowerCase()
        ) ||
        (request?.category?.toLowerCase() || "").includes(
          searchTerm.toLowerCase()
        ) ||
        (request?.property?.name?.toLowerCase() || "").includes(
          searchTerm.toLowerCase()
        ) ||
        (request?.tenant?.user?.firstName?.toLowerCase() || "").includes(
          searchTerm.toLowerCase()
        ) ||
        (request?.tenant?.user?.lastName?.toLowerCase() || "").includes(
          searchTerm.toLowerCase()
        );

      const matchesStatus =
        statusFilter === "ALL_STATUSES" || request?.status === statusFilter;
      const matchesPriority =
        priorityFilter === "ALL_PRIORITIES" ||
        request?.priority === priorityFilter;
      const matchesCategory =
        categoryFilter === "ALL_CATEGORIES" ||
        request?.category === categoryFilter;

      // Date filter
      const matchesDate = (() => {
        if (dateFilter === "ALL_DATES") return true;
        const now = new Date();
        const requestDate = new Date(request?.createdAt || "");

        switch (dateFilter) {
          case "TODAY":
            return requestDate.toDateString() === now.toDateString();
          case "WEEK":
            const weekAgo = new Date(now.getTime() - 7 * 24 * 60 * 60 * 1000);
            return requestDate >= weekAgo;
          case "MONTH":
            const monthAgo = new Date(now.getTime() - 30 * 24 * 60 * 60 * 1000);
            return requestDate >= monthAgo;
          default:
            return true;
        }
      })();

      // Assignment filter
      const matchesAssigned = (() => {
        if (assignedFilter === "ALL_ASSIGNED") return true;
        if (assignedFilter === "ASSIGNED") return !!request?.assignedTo;
        if (assignedFilter === "UNASSIGNED") return !request?.assignedTo;
        return true;
      })();

      // Overdue filter
      const matchesOverdue =
        !overdueFilter ||
        getDaysOverdue(request?.createdAt, request?.priority, request?.status) >
          0;

      return (
        matchesSearch &&
        matchesStatus &&
        matchesPriority &&
        matchesCategory &&
        matchesDate &&
        matchesAssigned &&
        matchesOverdue
      );
    })
    .sort((a, b) => {
      let aValue: any, bValue: any;

      switch (sortBy) {
        case "title":
          aValue = a?.title || "";
          bValue = b?.title || "";
          break;
        case "priority":
          const priorityOrder = {
            [MaintenancePriority.EMERGENCY]: 4,
            [MaintenancePriority.HIGH]: 3,
            [MaintenancePriority.MEDIUM]: 2,
            [MaintenancePriority.LOW]: 1,
          };
          aValue = priorityOrder[a?.priority as MaintenancePriority] || 0;
          bValue = priorityOrder[b?.priority as MaintenancePriority] || 0;
          break;
        case "status":
          aValue = a?.status || "";
          bValue = b?.status || "";
          break;
        case "createdAt":
          aValue = new Date(a?.createdAt || "").getTime();
          bValue = new Date(b?.createdAt || "").getTime();
          break;
        case "cost":
          aValue = a?.estimatedCost || a?.actualCost || 0;
          bValue = b?.estimatedCost || b?.actualCost || 0;
          break;
        default:
          aValue = new Date(a?.createdAt || "").getTime();
          bValue = new Date(b?.createdAt || "").getTime();
      }

      if (sortOrder === "asc") {
        return aValue > bValue ? 1 : -1;
      } else {
        return aValue < bValue ? 1 : -1;
      }
    });

  // Use the filtered and sorted requests for display
  const filteredRequests = filteredAndSortedRequests;

  const getPriorityColor = (priority: MaintenancePriority | undefined) => {
    if (!priority) return "outline";

    switch (priority) {
      case MaintenancePriority.EMERGENCY:
        return "destructive";
      case MaintenancePriority.HIGH:
        return "destructive";
      case MaintenancePriority.MEDIUM:
        return "default";
      case MaintenancePriority.LOW:
        return "secondary";
      default:
        return "outline";
    }
  };

  const getStatusColor = (status: MaintenanceStatus | undefined) => {
    if (!status) return "outline";

    switch (status) {
      case MaintenanceStatus.SUBMITTED:
        return "default";
      case MaintenanceStatus.ASSIGNED:
        return "secondary";
      case MaintenanceStatus.IN_PROGRESS:
        return "default";
      case MaintenanceStatus.COMPLETED:
        return "secondary";
      case MaintenanceStatus.CANCELLED:
        return "outline";
      default:
        return "outline";
    }
  };

  const getStatusIcon = (status: MaintenanceStatus | undefined) => {
    if (!status) return Clock;

    switch (status) {
      case MaintenanceStatus.SUBMITTED:
        return Clock;
      case MaintenanceStatus.ASSIGNED:
        return User;
      case MaintenanceStatus.IN_PROGRESS:
        return Play;
      case MaintenanceStatus.COMPLETED:
        return CheckCircle;
      case MaintenanceStatus.CANCELLED:
        return X;
      default:
        return Clock;
    }
  };

  const formatCurrency = (amount: number | undefined) => {
    if (amount === undefined || amount === null) return "-";
    return new Intl.NumberFormat("en-US", {
      style: "currency",
      currency: "USD",
    }).format(amount);
  };

  const formatDate = (date: Date | string | undefined) => {
    if (!date) return "N/A";

    try {
      const dateObj = typeof date === "string" ? new Date(date) : date;
      if (isNaN(dateObj.getTime())) return "N/A";

      return dateObj.toLocaleDateString("en-US", {
        year: "numeric",
        month: "short",
        day: "numeric",
        hour: "2-digit",
        minute: "2-digit",
      });
    } catch (error) {
      // Silently handle date formatting errors
      return "N/A";
    }
  };

  // Calculate statistics
  const totalRequests = filteredRequests.length;
  const emergencyRequests = filteredRequests.filter(
    (r: MaintenanceRequestWithPopulated) =>
      r?.priority === MaintenancePriority.EMERGENCY
  ).length;
  const inProgressRequests = filteredRequests.filter(
    (r: MaintenanceRequestWithPopulated) =>
      r?.status === MaintenanceStatus.IN_PROGRESS
  ).length;
  const overdueRequests = filteredRequests.filter(
    (r: MaintenanceRequestWithPopulated) =>
      getDaysOverdue(r?.createdAt, r?.priority, r?.status) > 0
  ).length;
  const totalEstimatedCost = filteredRequests.reduce(
    (sum: number, r: MaintenanceRequestWithPopulated) =>
      sum + (r?.estimatedCost || 0),
    0
  );

  // Additional analytics
  const completedRequests = filteredRequests.filter(
    (r: MaintenanceRequestWithPopulated) =>
      r?.status === MaintenanceStatus.COMPLETED
  ).length;
  const pendingRequests = filteredRequests.filter(
    (r: MaintenanceRequestWithPopulated) =>
      r?.status === MaintenanceStatus.SUBMITTED ||
      r?.status === MaintenanceStatus.ASSIGNED
  ).length;

  // Calculate completion rate
  const completionRate =
    totalRequests > 0
      ? Math.round((completedRequests / totalRequests) * 100)
      : 0;

  // Calculate total actual cost
  const totalActualCost = filteredRequests.reduce(
    (sum: number, r: MaintenanceRequestWithPopulated) =>
      sum + (r?.actualCost || 0),
    0
  );

  // Calculate average cost
  const avgCost =
    completedRequests > 0 ? totalActualCost / completedRequests : 0;

  if (error) {
    return (
      <div className="space-y-6">
        <div className="flex items-center justify-between">
          <div>
            <h1 className="text-3xl font-bold tracking-tight">Maintenance</h1>
            <p className="text-muted-foreground">
              Manage maintenance requests and work orders
            </p>
          </div>
          <Link href="/dashboard/maintenance/new">
            <Button>
              <Plus className="mr-2 h-4 w-4" />
              New Request
            </Button>
          </Link>
        </div>
        <div className="flex flex-col items-center justify-center min-h-[400px] space-y-4">
          <AlertTriangle className="h-12 w-12 text-muted-foreground" />
          <h2 className="text-xl font-semibold">
            Failed to Load Maintenance Requests
          </h2>
          <p className="text-muted-foreground text-center">{error}</p>
          <Button onClick={fetchMaintenanceRequests}>Try Again</Button>
        </div>
      </div>
    );
  }

  return (
    <div className="space-y-6">
      {/* Header */}
      <div className="flex items-center justify-between">
        <div>
          <h1 className="text-3xl font-bold tracking-tight">Maintenance</h1>
          <p className="text-muted-foreground">
            Manage maintenance requests and work orders
          </p>
        </div>
        <div className="flex items-center space-x-2">
          <Link href="/dashboard/maintenance/new">
            <Button>
              <Plus className="mr-2 h-4 w-4" />
              New Request
            </Button>
          </Link>
        </div>
      </div>

      {/* Stats Cards */}
      <div className="grid gap-4 md:grid-cols-5">
        <Card className="gap-0">
          <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
            <CardTitle className="text-sm font-medium">
              Total Requests
            </CardTitle>
            <Wrench className="h-4 w-4 text-muted-foreground" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold">{totalRequests}</div>
            <p className="text-xs text-muted-foreground">
              Active maintenance requests
            </p>
          </CardContent>
        </Card>
        <Card className="gap-0">
          <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
            <CardTitle className="text-sm font-medium">Emergency</CardTitle>
            <AlertTriangle className="h-4 w-4 text-red-600" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold text-red-600">
              {emergencyRequests}
            </div>
            <p className="text-xs text-muted-foreground">
              Urgent attention needed
            </p>
          </CardContent>
        </Card>
        <Card className="gap-0">
          <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
            <CardTitle className="text-sm font-medium">In Progress</CardTitle>
            <Play className="h-4 w-4 text-blue-600" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold text-blue-600">
              {inProgressRequests}
            </div>
            <p className="text-xs text-muted-foreground">
              Currently being worked on
            </p>
          </CardContent>
        </Card>
        <Card className="gap-0">
          <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
            <CardTitle className="text-sm font-medium">Overdue</CardTitle>
            <Clock className="h-4 w-4 text-orange-600" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold text-orange-600">
              {overdueRequests}
            </div>
            <p className="text-xs text-muted-foreground">Past SLA deadline</p>
          </CardContent>
        </Card>
        <Card className="gap-0">
          <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
            <CardTitle className="text-sm font-medium">Est. Cost</CardTitle>
            <DollarSign className="h-4 w-4 text-muted-foreground" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold">
              {formatCurrency(totalEstimatedCost)}
            </div>
            <p className="text-xs text-muted-foreground">
              Total estimated costs
            </p>
          </CardContent>
        </Card>
      </div>

      {/* Additional Analytics Cards */}
      <div className="grid gap-4 md:grid-cols-4">
        <Card className="gap-0">
          <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
            <CardTitle className="text-sm font-medium">Completed</CardTitle>
            <CheckCircle className="h-4 w-4 text-green-600" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold text-green-600">
              {completedRequests}
            </div>
            <p className="text-xs text-muted-foreground">
              Successfully resolved
            </p>
          </CardContent>
        </Card>
        <Card className="gap-0">
          <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
            <CardTitle className="text-sm font-medium">Pending</CardTitle>
            <Clock className="h-4 w-4 text-yellow-600" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold text-yellow-600">
              {pendingRequests}
            </div>
            <p className="text-xs text-muted-foreground">Awaiting assignment</p>
          </CardContent>
        </Card>
        <Card className="gap-0">
          <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
            <CardTitle className="text-sm font-medium">
              Completion Rate
            </CardTitle>
            <TrendingUp className="h-4 w-4 text-green-600" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold text-green-600">
              {completionRate}%
            </div>
            <p className="text-xs text-muted-foreground">
              Overall success rate
            </p>
          </CardContent>
        </Card>
        <Card className="gap-0">
          <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
            <CardTitle className="text-sm font-medium">Avg Cost</CardTitle>
            <DollarSign className="h-4 w-4 text-blue-600" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold text-blue-600">
              {formatCurrency(avgCost)}
            </div>
            <p className="text-xs text-muted-foreground">
              Per completed request
            </p>
          </CardContent>
        </Card>
      </div>

      {/* Advanced Filters */}
      <Card className="gap-2">
        <CardHeader>
          <CardTitle className="flex items-center gap-2">
            <Filter className="h-5 w-5" />
            Filter & Search Maintenance Requests
          </CardTitle>
        </CardHeader>
        <CardContent>
          <div className="space-y-4">
            {/* Search */}
            <div className="relative">
              <Search className="absolute left-3 top-1/2 h-4 w-4 -translate-y-1/2 text-muted-foreground" />
              <Input
                placeholder="Search by title, description, category, property, or tenant..."
                value={searchTerm}
                onChange={(e) => setSearchTerm(e.target.value)}
                className="pl-10"
              />
            </div>

            {/* Primary Filters */}
            <div className="grid gap-4 md:grid-cols-4">
              <Select value={statusFilter} onValueChange={setStatusFilter}>
                <SelectTrigger>
                  <SelectValue placeholder="Filter by status" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="ALL_STATUSES">All Statuses</SelectItem>
                  <SelectItem value={MaintenanceStatus.SUBMITTED}>
                    Submitted
                  </SelectItem>
                  <SelectItem value={MaintenanceStatus.ASSIGNED}>
                    Assigned
                  </SelectItem>
                  <SelectItem value={MaintenanceStatus.IN_PROGRESS}>
                    In Progress
                  </SelectItem>
                  <SelectItem value={MaintenanceStatus.COMPLETED}>
                    Completed
                  </SelectItem>
                  <SelectItem value={MaintenanceStatus.CANCELLED}>
                    Cancelled
                  </SelectItem>
                </SelectContent>
              </Select>

              <Select value={priorityFilter} onValueChange={setPriorityFilter}>
                <SelectTrigger>
                  <SelectValue placeholder="Filter by priority" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="ALL_PRIORITIES">All Priorities</SelectItem>
                  <SelectItem value={MaintenancePriority.EMERGENCY}>
                    Emergency
                  </SelectItem>
                  <SelectItem value={MaintenancePriority.HIGH}>High</SelectItem>
                  <SelectItem value={MaintenancePriority.MEDIUM}>
                    Medium
                  </SelectItem>
                  <SelectItem value={MaintenancePriority.LOW}>Low</SelectItem>
                </SelectContent>
              </Select>

              <Select value={categoryFilter} onValueChange={setCategoryFilter}>
                <SelectTrigger>
                  <SelectValue placeholder="Filter by category" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="ALL_CATEGORIES">All Categories</SelectItem>
                  <SelectItem value="Plumbing">Plumbing</SelectItem>
                  <SelectItem value="HVAC">HVAC</SelectItem>
                  <SelectItem value="Electrical">Electrical</SelectItem>
                  <SelectItem value="General Repair">General Repair</SelectItem>
                  <SelectItem value="Appliances">Appliances</SelectItem>
                  <SelectItem value="Flooring">Flooring</SelectItem>
                  <SelectItem value="Painting">Painting</SelectItem>
                  <SelectItem value="Roofing">Roofing</SelectItem>
                  <SelectItem value="Windows">Windows</SelectItem>
                  <SelectItem value="Doors">Doors</SelectItem>
                  <SelectItem value="Landscaping">Landscaping</SelectItem>
                  <SelectItem value="Pest Control">Pest Control</SelectItem>
                  <SelectItem value="Security">Security</SelectItem>
                  <SelectItem value="Emergency">Emergency</SelectItem>
                  <SelectItem value="Other">Other</SelectItem>
                </SelectContent>
              </Select>

              <Select value={dateFilter} onValueChange={setDateFilter}>
                <SelectTrigger>
                  <SelectValue placeholder="Filter by date" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="ALL_DATES">All Dates</SelectItem>
                  <SelectItem value="TODAY">Today</SelectItem>
                  <SelectItem value="WEEK">This Week</SelectItem>
                  <SelectItem value="MONTH">This Month</SelectItem>
                </SelectContent>
              </Select>
            </div>

            {/* Secondary Filters */}
            <div className="flex flex-wrap items-center gap-4">
              <Select value={assignedFilter} onValueChange={setAssignedFilter}>
                <SelectTrigger className="w-[180px]">
                  <SelectValue placeholder="Assignment status" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="ALL_ASSIGNED">All Requests</SelectItem>
                  <SelectItem value="ASSIGNED">Assigned</SelectItem>
                  <SelectItem value="UNASSIGNED">Unassigned</SelectItem>
                </SelectContent>
              </Select>

              <div className="flex items-center space-x-2">
                <Checkbox
                  id="overdue"
                  checked={overdueFilter}
                  onCheckedChange={(checked) =>
                    setOverdueFilter(checked as boolean)
                  }
                />
                <label
                  htmlFor="overdue"
                  className="text-sm font-medium leading-none peer-disabled:cursor-not-allowed peer-disabled:opacity-70"
                >
                  Show only overdue
                </label>
              </div>

              {/* Sort Options */}
              <div className="flex items-center gap-2">
                <Select value={sortBy} onValueChange={setSortBy}>
                  <SelectTrigger className="w-[140px]">
                    <SelectValue placeholder="Sort by" />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="createdAt">Date Created</SelectItem>
                    <SelectItem value="title">Title</SelectItem>
                    <SelectItem value="priority">Priority</SelectItem>
                    <SelectItem value="status">Status</SelectItem>
                    <SelectItem value="cost">Cost</SelectItem>
                  </SelectContent>
                </Select>

                <Button
                  variant="outline"
                  size="sm"
                  onClick={() =>
                    setSortOrder(sortOrder === "asc" ? "desc" : "asc")
                  }
                  className="px-3"
                >
                  {sortOrder === "asc" ? (
                    <ArrowUp className="h-4 w-4" />
                  ) : (
                    <ArrowDown className="h-4 w-4" />
                  )}
                </Button>
              </div>

              {/* Clear Filters */}
              <Button
                variant="outline"
                size="sm"
                onClick={() => {
                  setSearchTerm("");
                  setStatusFilter("ALL_STATUSES");
                  setPriorityFilter("ALL_PRIORITIES");
                  setCategoryFilter("ALL_CATEGORIES");
                  setDateFilter("ALL_DATES");
                  setAssignedFilter("ALL_ASSIGNED");
                  setOverdueFilter(false);
                  setSortBy("createdAt");
                  setSortOrder("desc");
                }}
              >
                Clear Filters
              </Button>
            </div>
          </div>
        </CardContent>
      </Card>

      {/* Maintenance Requests */}
      <Card className="gap-2">
        <CardHeader>
          <div className="flex items-center justify-between">
            <div>
              <CardTitle>
                Maintenance Requests ({filteredRequests.length})
              </CardTitle>
              <CardDescription>
                A list of all maintenance requests and their current status
              </CardDescription>
            </div>
            <div className="flex items-center gap-2">
              <Button
                variant={viewMode === "table" ? "default" : "outline"}
                size="sm"
                onClick={() => setViewMode("table")}
                className="hidden md:flex"
              >
                <List className="h-4 w-4 mr-2" />
                Table
              </Button>
              <Button
                variant={viewMode === "cards" ? "default" : "outline"}
                size="sm"
                onClick={() => setViewMode("cards")}
              >
                <Grid3X3 className="h-4 w-4 mr-2" />
                Cards
              </Button>
            </div>
          </div>
        </CardHeader>
        <CardContent>
          {viewMode === "table" ? (
            <div className="hidden md:block max-w-full">
              <Table
                containerClassName="md:overflow-x-hidden md:overflow-y-visible md:[&::-webkit-scrollbar]:hidden"
                className="table-fixed"
              >
                <TableHeader className="bg-muted/50">
                  <TableRow>
                    <TableHead className="min-w-[200px]">Request</TableHead>
                    <TableHead className="min-w-[120px] hidden lg:table-cell">
                      Property
                    </TableHead>
                    <TableHead className="min-w-[120px] hidden xl:table-cell">
                      Tenant
                    </TableHead>
                    <TableHead className="w-[80px]">Priority</TableHead>
                    <TableHead className="w-[100px]">Status</TableHead>
                    <TableHead className="min-w-[120px] hidden lg:table-cell">
                      Assigned To
                    </TableHead>
                    <TableHead className="w-[80px] hidden xl:table-cell">
                      Cost
                    </TableHead>
                    <TableHead className="w-[100px] hidden lg:table-cell">
                      Created
                    </TableHead>
                    <TableHead className="w-[80px] hidden xl:table-cell">
                      Overdue
                    </TableHead>
                    <TableHead className="w-[60px] text-right">
                      Actions
                    </TableHead>
                  </TableRow>
                </TableHeader>
                <TableBody>
                  {filteredRequests.map((request) => {
                    const StatusIcon = getStatusIcon(request?.status);
                    const daysOverdue = getDaysOverdue(
                      request?.createdAt,
                      request?.priority,
                      request?.status
                    );

                    return (
                      <TableRow key={request._id}>
                        <TableCell className="max-w-[200px]">
                          <div className="space-y-1">
                            <div className="font-medium text-sm line-clamp-2">
                              {request?.title || "N/A"}
                            </div>
                            <div className="text-xs text-muted-foreground line-clamp-1">
                              {request?.category || "N/A"}
                            </div>
                            {(request?.images?.length || 0) > 0 && (
                              <div className="flex items-center text-xs text-muted-foreground">
                                <Image className="h-3 w-3 mr-1" />
                                {request?.images?.length || 0} image
                                {(request?.images?.length || 0) > 1 ? "s" : ""}
                              </div>
                            )}
                          </div>
                        </TableCell>
                        <TableCell className="hidden lg:table-cell max-w-[120px]">
                          <div>
                            <div className="font-medium text-sm truncate">
                              {request?.property?.name || "N/A"}
                            </div>
                            {request?.unit && (
                              <div className="text-xs text-blue-600 truncate">
                                Unit {request.unit.unitNumber} (
                                {request.unit.unitType})
                              </div>
                            )}
                            <div className="text-xs text-muted-foreground truncate">
                              {request?.property?.address?.city || "N/A"},{" "}
                              {request?.property?.address?.state || "N/A"}
                            </div>
                          </div>
                        </TableCell>
                        <TableCell className="hidden xl:table-cell max-w-[120px]">
                          <div>
                            <div className="font-medium text-sm truncate">
                              {request?.tenant?.user?.firstName || "N/A"}{" "}
                              {request?.tenant?.user?.lastName || ""}
                            </div>
                            <div className="text-xs text-muted-foreground truncate">
                              {request?.tenant?.user?.email || "N/A"}
                            </div>
                          </div>
                        </TableCell>
                        <TableCell className="w-[80px]">
                          <Badge
                            variant={getPriorityColor(request?.priority)}
                            className="capitalize text-xs"
                          >
                            {request?.priority ===
                              MaintenancePriority.EMERGENCY && (
                              <AlertTriangle className="h-3 w-3 mr-1" />
                            )}
                            <span className="truncate">
                              {request?.priority ===
                              MaintenancePriority.EMERGENCY
                                ? "Emerg"
                                : request?.priority || "N/A"}
                            </span>
                          </Badge>
                        </TableCell>
                        <TableCell className="w-[100px]">
                          <Badge
                            variant={getStatusColor(request?.status)}
                            className="flex items-center gap-1 w-fit text-xs"
                          >
                            <StatusIcon className="h-3 w-3" />
                            <span className="truncate">
                              {request?.status?.replace("_", " ") || "N/A"}
                            </span>
                          </Badge>
                        </TableCell>
                        <TableCell className="hidden lg:table-cell max-w-[120px]">
                          {request?.assignedTo ? (
                            <div>
                              <div className="font-medium text-sm truncate">
                                {request?.assignedTo?.firstName || "N/A"}{" "}
                                {request?.assignedTo?.lastName || ""}
                              </div>
                              <div className="text-xs text-muted-foreground truncate">
                                {request?.assignedTo?.email || "N/A"}
                              </div>
                            </div>
                          ) : (
                            <span className="text-muted-foreground text-sm">
                              Unassigned
                            </span>
                          )}
                        </TableCell>
                        <TableCell className="hidden xl:table-cell w-[80px]">
                          <div className="space-y-1">
                            {request?.estimatedCost && (
                              <div className="text-xs">
                                Est: {formatCurrency(request.estimatedCost)}
                              </div>
                            )}
                            {request?.actualCost && (
                              <div className="text-xs font-medium">
                                {formatCurrency(request.actualCost)}
                              </div>
                            )}
                            {!request?.estimatedCost &&
                              !request?.actualCost && (
                                <span className="text-muted-foreground text-xs">
                                  -
                                </span>
                              )}
                          </div>
                        </TableCell>
                        <TableCell className="hidden lg:table-cell w-[100px]">
                          <div className="flex items-center space-x-1 text-xs">
                            <Calendar className="h-3 w-3 text-muted-foreground" />
                            <span className="truncate">
                              {formatDate(request?.createdAt)}
                            </span>
                          </div>
                        </TableCell>
                        <TableCell className="hidden xl:table-cell w-[80px]">
                          {daysOverdue > 0 ? (
                            <div className="text-xs text-red-600 font-medium">
                              {daysOverdue}d
                            </div>
                          ) : (
                            <span className="text-muted-foreground text-xs">
                              -
                            </span>
                          )}
                        </TableCell>
                        <TableCell className="text-right w-[60px]">
                          <MaintenanceStatusManager
                            request={request as any}
                            onStatusUpdate={handleStatusUpdate}
                            onRequestUpdate={handleRequestUpdate}
                            availableStaff={availableStaff}
                          />
                        </TableCell>
                      </TableRow>
                    );
                  })}
                </TableBody>
              </Table>
            </div>
          ) : (
            <div className="grid gap-4 md:grid-cols-2 lg:grid-cols-3">
              {filteredRequests.map((request) => {
                const StatusIcon = getStatusIcon(request?.status);
                const daysOverdue = getDaysOverdue(
                  request?.createdAt,
                  request?.priority,
                  request?.status
                );

                return (
                  <Card
                    key={request._id}
                    className="hover:shadow-md transition-shadow"
                  >
                    <CardHeader className="pb-3">
                      <div className="flex items-start justify-between">
                        <div className="space-y-1 flex-1">
                          <CardTitle className="text-base line-clamp-2">
                            {request?.title || "N/A"}
                          </CardTitle>
                          <div className="flex items-center gap-2">
                            <Badge
                              variant={
                                getPriorityColor(request?.priority) as any
                              }
                              className="text-xs"
                            >
                              {request?.priority || "N/A"}
                            </Badge>
                            <Badge
                              variant={getStatusColor(request?.status) as any}
                              className="flex items-center gap-1 text-xs"
                            >
                              <StatusIcon className="h-3 w-3" />
                              {request?.status?.replace("_", " ") || "N/A"}
                            </Badge>
                          </div>
                        </div>
                        <MaintenanceStatusManager
                          request={request as any}
                          onStatusUpdate={handleStatusUpdate}
                          onRequestUpdate={handleRequestUpdate}
                          availableStaff={availableStaff}
                        />
                      </div>
                    </CardHeader>
                    <CardContent className="pt-0 space-y-3">
                      <div className="text-sm text-muted-foreground line-clamp-2">
                        {request?.description || "No description"}
                      </div>

                      <div className="grid grid-cols-2 gap-3 text-sm">
                        <div>
                          <div className="font-medium text-muted-foreground">
                            Property
                          </div>
                          <div className="truncate">
                            {request?.property?.name || "N/A"}
                          </div>
                        </div>
                        <div>
                          <div className="font-medium text-muted-foreground">
                            Tenant
                          </div>
                          <div className="truncate">
                            {request?.tenant?.user?.firstName || "N/A"}{" "}
                            {request?.tenant?.user?.lastName || ""}
                          </div>
                        </div>
                        <div>
                          <div className="font-medium text-muted-foreground">
                            Category
                          </div>
                          <div className="truncate">
                            {request?.category || "N/A"}
                          </div>
                        </div>
                        <div>
                          <div className="font-medium text-muted-foreground">
                            Cost
                          </div>
                          <div className="truncate">
                            {formatCurrency(
                              request?.estimatedCost || request?.actualCost
                            )}
                          </div>
                        </div>
                      </div>

                      {request?.assignedTo && (
                        <div className="flex items-center gap-2 text-sm">
                          <User className="h-4 w-4 text-muted-foreground" />
                          <span className="text-muted-foreground">
                            Assigned to:
                          </span>
                          <span>
                            {request.assignedTo.firstName}{" "}
                            {request.assignedTo.lastName}
                          </span>
                        </div>
                      )}

                      <div className="flex items-center justify-between text-xs text-muted-foreground">
                        <div className="flex items-center gap-1">
                          <Calendar className="h-3 w-3" />
                          {formatDate(request?.createdAt)}
                        </div>
                        {daysOverdue > 0 && (
                          <Badge variant="destructive" className="text-xs">
                            {daysOverdue} days overdue
                          </Badge>
                        )}
                      </div>
                    </CardContent>
                  </Card>
                );
              })}
            </div>
          )}

          {/* Mobile Table View */}
          {viewMode === "table" && (
            <div className="md:hidden">
              <div className="space-y-4">
                {filteredRequests.map((request) => {
                  const StatusIcon = getStatusIcon(request?.status);
                  const daysOverdue = getDaysOverdue(
                    request?.createdAt,
                    request?.priority,
                    request?.status
                  );

                  return (
                    <Card key={request._id} className="p-4">
                      <div className="space-y-3">
                        <div className="flex items-start justify-between">
                          <div className="space-y-1 flex-1">
                            <h3 className="font-medium line-clamp-2">
                              {request?.title || "N/A"}
                            </h3>
                            <div className="flex items-center gap-2">
                              <Badge
                                variant={
                                  getPriorityColor(request?.priority) as any
                                }
                                className="text-xs"
                              >
                                {request?.priority || "N/A"}
                              </Badge>
                              <Badge
                                variant={getStatusColor(request?.status) as any}
                                className="flex items-center gap-1 text-xs"
                              >
                                <StatusIcon className="h-3 w-3" />
                                {request?.status?.replace("_", " ") || "N/A"}
                              </Badge>
                            </div>
                          </div>
                          <MaintenanceStatusManager
                            request={request as any}
                            onStatusUpdate={handleStatusUpdate}
                            onRequestUpdate={handleRequestUpdate}
                            availableStaff={availableStaff}
                          />
                        </div>

                        <div className="grid grid-cols-2 gap-2 text-sm">
                          <div>
                            <span className="text-muted-foreground">
                              Property:
                            </span>
                            <div className="truncate">
                              {request?.property?.name || "N/A"}
                              {request?.unit && (
                                <div className="text-xs text-blue-600">
                                  Unit {request.unit.unitNumber}
                                </div>
                              )}
                            </div>
                          </div>
                          <div>
                            <span className="text-muted-foreground">
                              Tenant:
                            </span>
                            <div className="truncate">
                              {request?.tenant?.user?.firstName || "N/A"}{" "}
                              {request?.tenant?.user?.lastName || ""}
                            </div>
                          </div>
                          <div>
                            <span className="text-muted-foreground">
                              Created:
                            </span>
                            <div>{formatDate(request?.createdAt)}</div>
                          </div>
                          <div>
                            <span className="text-muted-foreground">Cost:</span>
                            <div>
                              {formatCurrency(
                                request?.estimatedCost || request?.actualCost
                              )}
                            </div>
                          </div>
                        </div>

                        {daysOverdue > 0 && (
                          <Badge
                            variant="destructive"
                            className="text-xs w-fit"
                          >
                            {daysOverdue} days overdue
                          </Badge>
                        )}
                      </div>
                    </Card>
                  );
                })}
              </div>
            </div>
          )}

          {filteredRequests.length === 0 && !loading && (
            <div className="flex flex-col items-center justify-center py-12 space-y-4">
              <Wrench className="h-12 w-12 text-muted-foreground" />
              <h3 className="text-lg font-semibold">
                No maintenance requests found
              </h3>
              <p className="text-muted-foreground text-center">
                {searchTerm ||
                statusFilter !== "ALL_STATUSES" ||
                priorityFilter !== "ALL_PRIORITIES" ||
                categoryFilter !== "ALL_CATEGORIES"
                  ? "Try adjusting your filters to see more results."
                  : "Get started by creating your first maintenance request."}
              </p>
              {!searchTerm &&
                statusFilter === "ALL_STATUSES" &&
                priorityFilter === "ALL_PRIORITIES" &&
                categoryFilter === "ALL_CATEGORIES" && (
                  <Link href="/dashboard/maintenance/new">
                    <Button>
                      <Plus className="mr-2 h-4 w-4" />
                      Create First Request
                    </Button>
                  </Link>
                )}
            </div>
          )}
        </CardContent>
      </Card>
    </div>
  );
}
