"use client";

import { toast } from "sonner";
import { UserRole } from "@/types";
import { ArrowLeft } from "lucide-react";
import { useState, useEffect } from "react";
import { useRouter } from "next/navigation";
import { useSession } from "next-auth/react";
import { Button } from "@/components/ui/button";
import { MaintenanceRequestForm } from "@/components/forms/maintenance-request-form";
import TenantMaintenanceRequestForm from "@/components/tenant/TenantMaintenanceRequestForm";

interface Property {
  id: string;
  name: string;
  address: string;
  isMultiUnit?: boolean;
  units?: Array<{
    _id: string;
    unitNumber: string;
    unitType: string;
    status: string;
  }>;
}

interface PropertyApiResponse {
  _id: string;
  name: string;
  address: {
    street: string;
    city: string;
    state: string;
  };
  isMultiUnit?: boolean;
  units?: Array<{
    _id: string;
    unitNumber: string;
    unitType: string;
    status: string;
  }>;
}

interface Tenant {
  id: string;
  name: string;
  email: string;
  propertyName?: string;
  phone?: string;
  tenantStatus?: string;
}

interface TenantApiResponse {
  _id: string;
  firstName: string;
  lastName: string;
  email: string;
  phone?: string;
  tenantStatus?: string;
  currentLeaseId?: {
    propertyId?: {
      name: string;
    };
  };
}

interface TechnicianApiResponse {
  _id: string;
  firstName: string;
  lastName: string;
  email: string;
  specialties?: string[];
}

interface Technician {
  id: string;
  name: string;
  email: string;
  specialties?: string[];
}

export default function NewMaintenanceRequestPage() {
  const router = useRouter();
  const { data: session } = useSession();
  const [loading, setLoading] = useState(false);
  const [dataLoading, setDataLoading] = useState(true);
  const [tenants, setTenants] = useState<Tenant[]>([]);
  const [properties, setProperties] = useState<Property[]>([]);
  const [technicians, setTechnicians] = useState<Technician[]>([]);

  // Check if user is a tenant
  const isTenant = session?.user?.role === UserRole.TENANT;

  useEffect(() => {
    // Only fetch form data for admin/manager users
    if (!isTenant) {
      fetchFormData();
    }
  }, [isTenant]);

  const fetchFormData = async () => {
    try {
      setDataLoading(true);

      // Fetch properties, tenants, and technicians in parallel
      const [propertiesRes, tenantsRes, techniciansRes] = await Promise.all([
        fetch("/api/properties?limit=100"),
        fetch("/api/tenants?limit=100"),
        fetch("/api/users?role=manager&limit=100"),
      ]);

      if (propertiesRes.ok) {
        const propertiesData = await propertiesRes.json();
        setProperties(
          Array.isArray(propertiesData.data)
            ? propertiesData.data.map((property: PropertyApiResponse) => ({
                id: property._id,
                name: property.name,
                address: `${property.address.street}, ${property.address.city}, ${property.address.state}`,
                isMultiUnit: property.isMultiUnit,
                units: property.units || [],
              }))
            : []
        );
      } else {
        toast.error("Failed to load properties");
      }

      if (tenantsRes.ok) {
        const tenantsData = await tenantsRes.json();

        // Handle the new User model structure for tenants
        setTenants(
          Array.isArray(tenantsData.data)
            ? tenantsData.data
                .filter((tenant: TenantApiResponse) => tenant && tenant._id) // Filter out invalid entries
                .map((tenant: TenantApiResponse) => ({
                  id: tenant._id,
                  name:
                    `${tenant.firstName || ""} ${
                      tenant.lastName || ""
                    }`.trim() || "Unknown Tenant",
                  email: tenant.email || "",
                  propertyName:
                    tenant.currentLeaseId?.propertyId?.name ||
                    "No active lease",
                  phone: tenant.phone || "",
                  tenantStatus: tenant.tenantStatus || "",
                }))
            : []
        );
      } else {
        toast.error("Failed to load tenants");
      }

      if (techniciansRes.ok) {
        const techniciansData = await techniciansRes.json();
        setTechnicians(
          Array.isArray(techniciansData.data)
            ? techniciansData.data.map((tech: TechnicianApiResponse) => ({
                id: tech._id,
                name: `${tech.firstName} ${tech.lastName}`,
                email: tech.email,
                specialties: tech.specialties || [],
              }))
            : []
        );
      } else {
        toast.error("Failed to load technicians");
      }
    } catch (error) {
      toast.error("Failed to load form data");
    } finally {
      setDataLoading(false);
    }
  };

  const handleSubmit = async (data: {
    title: string;
    description: string;
    category: string;
    priority: string;
    propertyId: string;
    unitId?: string;
    tenantId: string;
    assignedTo?: string;
    estimatedCost?: number;
    scheduledDate?: string;
    images?: string[];
  }) => {
    try {
      setLoading(true);

      const response = await fetch("/api/maintenance", {
        method: "POST",
        headers: {
          "Content-Type": "application/json",
        },
        body: JSON.stringify({
          title: data.title,
          description: data.description,
          category: data.category,
          priority: data.priority,
          propertyId: data.propertyId,
          unitId: data.unitId || undefined,
          tenantId: data.tenantId,
          assignedTo: data.assignedTo || undefined,
          estimatedCost: data.estimatedCost || undefined,
          scheduledDate: data.scheduledDate || undefined,
          images: data.images || [],
        }),
      });

      const result = await response.json();

      if (!response.ok) {
        throw new Error(result.error || "Failed to create maintenance request");
      }

      toast.success("Maintenance request created successfully!");
      router.push(`/dashboard/maintenance/${result.data._id}`);
    } catch (error: unknown) {
      const errorMessage =
        error instanceof Error
          ? error.message
          : "Failed to create maintenance request";
      toast.error(errorMessage);
    } finally {
      setLoading(false);
    }
  };


  return (
    <div className="space-y-6">
      {/* Header */}
      <div className="flex items-center justify-between">
        <div>
          <h1 className="text-2xl font-bold tracking-tight bg-gradient-to-r from-blue-600 to-indigo-600 bg-clip-text text-transparent">
            New Maintenance Request
          </h1>
          <p className="text-muted-foreground">
            Create a new maintenance request for property issues
          </p>
        </div>
        <Button
          variant="ghost"
          size="sm"
          className="hover:bg-blue-50 hover:text-blue-600 border transition-colors"
          onClick={() =>
            router.push(
              isTenant
                ? "/dashboard/maintenance/my-requests"
                : "/dashboard/maintenance"
            )
          }
        >
          <ArrowLeft className="h-4 w-4 mr-2" />
          Back
        </Button>
      </div>

      {/* Form Container */}
      {isTenant ? (
        <TenantMaintenanceRequestForm
          onCancel={() => router.push("/dashboard/maintenance/my-requests")}
          isLoading={loading}
        />
      ) : (
        <MaintenanceRequestForm
          onSubmit={handleSubmit}
          isLoading={loading}
          properties={properties}
          tenants={tenants}
          technicians={technicians}
        />
      )}
    </div>
  );
}
