"use client";

import { useState, useEffect } from "react";
import { useRouter } from "next/navigation";
import { useSession } from "next-auth/react";
import { toast } from "sonner";
import {
  Card,
  CardContent,
  CardDescription,
  CardHeader,
  CardTitle,
} from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select";

import { Skeleton } from "@/components/ui/skeleton";
import {
  Table,
  TableBody,
  TableCell,
  TableHead,
  TableHeader,
  TableRow,
} from "@/components/ui/table";
import { Avatar, AvatarFallback } from "@/components/ui/avatar";
import {
  DropdownMenu,
  DropdownMenuContent,
  DropdownMenuItem,
  DropdownMenuLabel,
  DropdownMenuSeparator,
  DropdownMenuTrigger,
} from "@/components/ui/dropdown-menu";
import {
  AlertDialog,
  AlertDialogAction,
  AlertDialogCancel,
  AlertDialogContent,
  AlertDialogDescription,
  AlertDialogFooter,
  AlertDialogHeader,
  AlertDialogTitle,
} from "@/components/ui/alert-dialog";
import { DeleteConfirmationDialog } from "@/components/ui/confirmation-dialog";
import {
  FileText,
  Plus,
  Search,
  Filter,
  MoreHorizontal,
  Eye,
  Edit,
  Trash2,
  Download,
  RefreshCw,
  LayoutGrid,
  List,
  AlertTriangle,
  Home,
} from "lucide-react";
import { LeaseStatus, UserRole } from "@/types";
import {
  leaseService,
  LeaseResponse,
  PaginatedLeasesResponse,
  LeaseQueryParams,
} from "@/lib/services/lease.service";
import { LeaseCard } from "@/components/leases/LeaseCard";
import { LeaseStatusBadge } from "@/components/leases/LeaseStatusBadge";
import { LeaseInvoiceModal } from "@/components/invoices";

export default function LeasesPage() {
  const router = useRouter();
  const { data: session, status } = useSession();
  const [leases, setLeases] = useState<LeaseResponse[]>([]);
  const [loading, setLoading] = useState(true);
  const [pagination, setPagination] = useState({
    page: 1,
    limit: 12,
    total: 0,
    pages: 0,
    hasNext: false,
    hasPrev: false,
  });
  const [filters, setFilters] = useState<LeaseQueryParams>({
    page: 1,
    limit: 12,
    search: "",
    status: undefined,
    sortBy: "createdAt",
    sortOrder: "desc",
  });

  // Redirect tenants to their "My Leases" page
  // UI state/hooks must be declared before any early returns so hooks order is stable
  const [viewMode, setViewMode] = useState<"table" | "cards">("table");
  const [selectedLeases, setSelectedLeases] = useState<string[]>([]);
  const [showBulkDeleteDialog, setShowBulkDeleteDialog] = useState(false);
  const [leaseToDelete, setLeaseToDelete] = useState<LeaseResponse | null>(
    null
  );
  const [isDeleting, setIsDeleting] = useState(false);
  const [stats, setStats] = useState({
    total: 0,
    active: 0,
    draft: 0,
    pending: 0,
    expired: 0,
    terminated: 0,
    expiringThisMonth: 0,
  });

  const fetchLeases = async () => {
    try {
      setLoading(true);
      console.log("🔍 fetchLeases called with filters:", filters);

      // TENANT view: fetch from tenant dashboard, then apply filters/sort/pagination client-side
      if (session?.user?.role === UserRole.TENANT) {
        const response = await fetch("/api/tenant/dashboard");
        const data = await response.json();

        if (data.success) {
          const allLeases: LeaseResponse[] = (data.data?.allLeases ||
            []) as LeaseResponse[];

          // Apply status + search filters
          const search = (filters.search || "").toLowerCase().trim();
          const filtered = allLeases.filter((lease) => {
            const matchesStatus =
              !filters.status ||
              lease.status === filters.status ||
              (filters.status === LeaseStatus.PENDING &&
                lease.status === LeaseStatus.PENDING_SIGNATURE);
            console.log(
              `Lease ${lease._id}: status=${lease.status}, filters.status=${filters.status}, matchesStatus=${matchesStatus}`
            );
            if (!search) return matchesStatus;

            const haystack = [
              lease.propertyId?.name,
              lease.propertyId?.address?.street,
              lease.propertyId?.address?.city,
              lease.tenantId?.firstName,
              lease.tenantId?.lastName,
              lease.tenantId?.email,
              lease.unit?.unitNumber,
              lease.status,
            ]
              .filter(Boolean)
              .map((v) => String(v).toLowerCase());

            const matchesSearch = haystack.some((v) => v.includes(search));
            return matchesStatus && matchesSearch;
          });

          // Sorting
          const sortBy = filters.sortBy || "createdAt";
          const sortOrder = filters.sortOrder || "desc";
          const get = (obj: any, path: string) =>
            path
              .split(".")
              .reduce((acc, key) => (acc == null ? acc : acc[key]), obj);

          const getSortVal = (lease: LeaseResponse) => {
            if (sortBy === "terms.rentAmount") {
              return lease.unit?.rentAmount ?? lease.terms?.rentAmount ?? 0;
            }
            const v = get(lease, sortBy);
            if (!v) return 0;
            const key = sortBy.toLowerCase();
            if (
              key.includes("date") ||
              key.includes("createdat") ||
              key.includes("updatedat")
            ) {
              const t = new Date(v as any).getTime();
              return Number.isNaN(t) ? 0 : t;
            }
            return typeof v === "number" ? v : String(v).toLowerCase();
          };

          const sorted = [...filtered].sort((a, b) => {
            const av = getSortVal(a);
            const bv = getSortVal(b);

            if (typeof av === "number" && typeof bv === "number") {
              return sortOrder === "asc" ? av - bv : bv - av;
            }
            return sortOrder === "asc"
              ? String(av).localeCompare(String(bv))
              : String(bv).localeCompare(String(av));
          });

          // Pagination
          const page = filters.page || 1;
          const limit = filters.limit || 12;
          const total = sorted.length;
          const pages = Math.max(1, Math.ceil(total / limit));
          const start = (page - 1) * limit;
          const paginated = sorted.slice(start, start + limit);

          setLeases(paginated);
          setPagination({
            page,
            limit,
            total,
            pages,
            hasNext: page < pages,
            hasPrev: page > 1,
          });
        } else {
          toast.error("Failed to fetch your leases");
        }
      } else {
        // ADMIN/MANAGER: fetch all leases and apply client-side filtering
        const baseParams = {
          search: filters.search,
          sortBy: filters.sortBy,
          sortOrder: filters.sortOrder,
          page: 1,
          limit: 1000,
        } as LeaseQueryParams;

        console.log("📡 Fetching all leases for admin/manager");
        const response: PaginatedLeasesResponse = await leaseService.getLeases(
          baseParams
        );
        console.log("📥 Received all leases:", response.data.length);

        let allLeases = response.data;

        // Apply status filter client-side
        if (filters.status) {
          console.log("🔍 Filtering by status:", filters.status);
          allLeases = allLeases.filter((lease) => {
            const matchesStatus =
              lease.status === filters.status ||
              (filters.status === LeaseStatus.PENDING &&
                lease.status === LeaseStatus.PENDING_SIGNATURE);
            console.log(
              `Lease ${lease._id}: status=${lease.status}, filters.status=${filters.status}, match=${matchesStatus}`
            );
            return matchesStatus;
          });
          console.log("✅ Filtered leases count:", allLeases.length);
        }

        // Sort
        const sortBy = filters.sortBy || "createdAt";
        const sortOrder = filters.sortOrder || "desc";
        const get = (obj: any, path: string) =>
          path
            .split(".")
            .reduce((acc, key) => (acc == null ? acc : acc[key]), obj);
        const getSortVal = (lease: LeaseResponse) => {
          if (sortBy === "terms.rentAmount") {
            return lease.unit?.rentAmount ?? lease.terms?.rentAmount ?? 0;
          }
          const v = get(lease, sortBy);
          if (!v) return 0;
          const key = sortBy.toLowerCase();
          if (
            key.includes("date") ||
            key.includes("createdat") ||
            key.includes("updatedat")
          ) {
            const t = new Date(v as any).getTime();
            return Number.isNaN(t) ? 0 : t;
          }
          return typeof v === "number" ? v : String(v).toLowerCase();
        };
        const sorted = [...allLeases].sort((a, b) => {
          const av = getSortVal(a);
          const bv = getSortVal(b);
          if (typeof av === "number" && typeof bv === "number") {
            return sortOrder === "asc" ? av - bv : bv - av;
          }
          return sortOrder === "asc"
            ? String(av).localeCompare(String(bv))
            : String(bv).localeCompare(String(av));
        });

        // Paginate client-side
        const page = filters.page || 1;
        const limit = filters.limit || 12;
        const total = sorted.length;
        const pages = Math.max(1, Math.ceil(total / limit));
        const start = (page - 1) * limit;
        const paginated = sorted.slice(start, start + limit);

        setLeases(paginated);
        setPagination({
          page,
          limit,
          total,
          pages,
          hasNext: page < pages,
          hasPrev: page > 1,
        });
      }
    } catch {
      toast.error("Failed to fetch leases");
    } finally {
      setLoading(false);
    }
  };

  const fetchStats = async () => {
    try {
      if (session?.user?.role === UserRole.TENANT) {
        // For tenants, calculate stats from their leases
        const response = await fetch("/api/tenant/dashboard");
        const data = await response.json();

        if (data.success && data.data.allLeases) {
          const tenantLeases = data.data.allLeases;
          const now = new Date();

          const statsData = {
            total: tenantLeases.length,
            active: tenantLeases.filter(
              (lease: any) => lease.status === "active"
            ).length,
            draft: tenantLeases.filter((lease: any) => lease.status === "draft")
              .length,
            pending: tenantLeases.filter(
              (lease: any) => lease.status === "pending"
            ).length,
            expired: tenantLeases.filter(
              (lease: any) => lease.status === "expired"
            ).length,
            terminated: tenantLeases.filter(
              (lease: any) => lease.status === "terminated"
            ).length,
            expiringThisMonth: tenantLeases.filter((lease: any) => {
              const endDate = new Date(lease.endDate);
              const daysUntilExpiration = Math.ceil(
                (endDate.getTime() - now.getTime()) / (1000 * 60 * 60 * 24)
              );
              return daysUntilExpiration <= 30 && daysUntilExpiration > 0;
            }).length,
          };

          setStats(statsData);
        }
      } else {
        // For admin/manager, use the existing lease service
        const statsData = await leaseService.getLeaseStats();
        setStats(statsData);
      }
    } catch {
      // Silently fail for stats - not critical
    }
  };

  const handleSearch = (value: string) => {
    setFilters((prev) => ({ ...prev, search: value, page: 1 }));
  };

  const handleStatusFilter = (status: string) => {
    console.log("🎯 handleStatusFilter called with:", status);
    const newStatus = status === "all" ? undefined : (status as LeaseStatus);
    console.log("🎯 Setting filters.status to:", newStatus);
    setFilters((prev) => ({
      ...prev,
      status: newStatus,
      page: 1,
    }));
  };

  const handleSort = (sortBy: string, sortOrder: "asc" | "desc") => {
    setFilters((prev) => ({ ...prev, sortBy, sortOrder, page: 1 }));
  };

  const handlePageChange = (page: number) => {
    setFilters((prev) => ({ ...prev, page }));
  };

  useEffect(() => {
    if (status === "authenticated" && session?.user?.role === UserRole.TENANT) {
      router.replace("/dashboard/leases/my-leases");
    }
  }, [status, session, router]);

  useEffect(() => {
    fetchLeases();
    fetchStats();
  }, [filters]); // eslint-disable-line react-hooks/exhaustive-deps

  // Show loading while checking authentication
  if (status === "loading") {
    return (
      <div className="flex items-center justify-center min-h-screen">
        <div className="animate-spin rounded-full h-32 w-32 border-b-2 border-primary"></div>
      </div>
    );
  }

  // Don't render anything for tenants (they'll be redirected)
  if (session?.user?.role === UserRole.TENANT) {
    return null;
  }

  const handleLeaseUpdate = () => {
    fetchLeases();
    fetchStats();
  };

  const handleLeaseDelete = async () => {
    if (!leaseToDelete) return;

    try {
      setIsDeleting(true);
      await leaseService.deleteLease(leaseToDelete._id);
      toast.success("Lease deleted successfully");
      setLeases((prev) =>
        prev.filter((lease) => lease._id !== leaseToDelete._id)
      );
      setStats((prev) => ({ ...prev, total: prev.total - 1 }));
      fetchLeases(); // Refresh the list to ensure consistency
    } catch {
      toast.error("Failed to delete lease");
    } finally {
      setIsDeleting(false);
      setLeaseToDelete(null);
    }
  };

  const handleBulkDelete = async () => {
    try {
      await leaseService.bulkDeleteLeases(selectedLeases);
      toast.success(`${selectedLeases.length} leases deleted successfully`);
      setSelectedLeases([]);
      setShowBulkDeleteDialog(false);
      fetchLeases();
      fetchStats();
    } catch {
      toast.error("Failed to delete leases");
    }
  };

  if (loading && leases.length === 0) {
    return (
      <div className="space-y-6">
        <div className="flex items-center justify-between">
          <div>
            <h1 className="text-3xl font-bold tracking-tight">Leases</h1>
            <p className="text-muted-foreground">Manage your property leases</p>
          </div>
          {<Skeleton className="h-10 w-32" />}
        </div>

        <div className="grid grid-cols-1 md:grid-cols-4 gap-4">
          {Array.from({ length: 4 }).map((_, i) => (
            <Skeleton key={i} className="h-24" />
          ))}
        </div>

        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
          {Array.from({ length: 6 }).map((_, i) => (
            <Skeleton key={i} className="h-64" />
          ))}
        </div>
      </div>
    );
  }

  return (
    <div className="space-y-6">
      {/* Header */}
      <div className="flex items-center justify-between">
        <div>
          <h1 className="text-3xl font-bold tracking-tight">Leases</h1>
          <p className="text-muted-foreground">
            Manage your property leases and agreements
          </p>
        </div>
        <Button onClick={() => router.push("/dashboard/leases/new")}>
          <Plus className="mr-2 h-4 w-4" />
          Create Lease
        </Button>
      </div>

      {/* Stats Cards */}
      <div className="grid grid-cols-1 md:grid-cols-4 lg:grid-cols-7 gap-4">
        {loading ? (
          [...Array(7)].map((_, i) => (
            <Card key={i} className="py-0">
              <CardContent className="p-4">
                <div className="flex items-center justify-between">
                  <div className="space-y-2">
                    <Skeleton className="h-4 w-12" />
                    <Skeleton className="h-8 w-16" />
                  </div>
                  <Skeleton className="h-8 w-8 rounded" />
                </div>
              </CardContent>
            </Card>
          ))
        ) : (
          <>
            <Card className="py-0">
              <CardContent className="p-4">
                <div className="flex items-center justify-between">
                  <div>
                    <p className="text-sm font-medium text-muted-foreground">
                      Total
                    </p>
                    <p className="text-2xl font-bold">{stats.total}</p>
                  </div>
                  <FileText className="h-8 w-8 text-muted-foreground" />
                </div>
              </CardContent>
            </Card>

            <Card className="py-0">
              <CardContent className="p-4">
                <div className="flex items-center justify-between">
                  <div>
                    <p className="text-sm font-medium text-muted-foreground">
                      Active
                    </p>
                    <p className="text-2xl font-bold text-green-600">
                      {stats.active}
                    </p>
                  </div>
                  <div className="h-8 w-8 rounded-full bg-green-100 flex items-center justify-center">
                    <div className="h-4 w-4 rounded-full bg-green-600"></div>
                  </div>
                </div>
              </CardContent>
            </Card>

            <Card className="py-0">
              <CardContent className="p-4">
                <div className="flex items-center justify-between">
                  <div>
                    <p className="text-sm font-medium text-muted-foreground">
                      Draft
                    </p>
                    <p className="text-2xl font-bold text-gray-600">
                      {stats.draft}
                    </p>
                  </div>
                  <div className="h-8 w-8 rounded-full bg-gray-100 flex items-center justify-center">
                    <div className="h-4 w-4 rounded-full bg-gray-600"></div>
                  </div>
                </div>
              </CardContent>
            </Card>

            <Card className="py-0">
              <CardContent className="p-4">
                <div className="flex items-center justify-between">
                  <div>
                    <p className="text-sm font-medium text-muted-foreground">
                      Pending
                    </p>
                    <p className="text-2xl font-bold text-yellow-600">
                      {stats.pending}
                    </p>
                  </div>
                  <div className="h-8 w-8 rounded-full bg-yellow-100 flex items-center justify-center">
                    <div className="h-4 w-4 rounded-full bg-yellow-600"></div>
                  </div>
                </div>
              </CardContent>
            </Card>

            <Card className="py-0">
              <CardContent className="p-4">
                <div className="flex items-center justify-between">
                  <div>
                    <p className="text-sm font-medium text-muted-foreground">
                      Expired
                    </p>
                    <p className="text-2xl font-bold text-red-600">
                      {stats.expired}
                    </p>
                  </div>
                  <div className="h-8 w-8 rounded-full bg-red-100 flex items-center justify-center">
                    <div className="h-4 w-4 rounded-full bg-red-600"></div>
                  </div>
                </div>
              </CardContent>
            </Card>

            <Card className="py-0">
              <CardContent className="p-4">
                <div className="flex items-center justify-between">
                  <div>
                    <p className="text-sm font-medium text-muted-foreground">
                      Terminated
                    </p>
                    <p className="text-2xl font-bold text-red-600">
                      {stats.terminated}
                    </p>
                  </div>
                  <div className="h-8 w-8 rounded-full bg-red-100 flex items-center justify-center">
                    <div className="h-4 w-4 rounded-full bg-red-600"></div>
                  </div>
                </div>
              </CardContent>
            </Card>

            <Card className="py-0">
              <CardContent className="p-4">
                <div className="flex items-center justify-between">
                  <div>
                    <p className="text-sm font-medium text-muted-foreground">
                      Expiring
                    </p>
                    <p className="text-2xl font-bold text-orange-600">
                      {stats.expiringThisMonth}
                    </p>
                  </div>
                  <AlertTriangle className="h-8 w-8 text-orange-600" />
                </div>
              </CardContent>
            </Card>
          </>
        )}
      </div>

      {/* Filters and Actions */}
      <Card className="p-0">
        <CardContent className="p-4">
          <div className="flex flex-col md:flex-row gap-4 items-center justify-between">
            <div className="flex flex-1 gap-4 items-center">
              <div className="relative flex-1 max-w-sm">
                <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 text-muted-foreground h-4 w-4" />
                <Input
                  placeholder="Search leases..."
                  value={filters.search}
                  onChange={(e) => handleSearch(e.target.value)}
                  className="pl-10"
                />
              </div>

              <Select
                value={filters.status || "all"}
                onValueChange={handleStatusFilter}
              >
                <SelectTrigger className="w-40">
                  <SelectValue placeholder="Status" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="all">All Status</SelectItem>
                  <SelectItem value="draft">Draft</SelectItem>
                  <SelectItem value="pending">Pending</SelectItem>
                  <SelectItem value="active">Active</SelectItem>
                  <SelectItem value="expired">Expired</SelectItem>
                  <SelectItem value="terminated">Terminated</SelectItem>
                </SelectContent>
              </Select>

              <DropdownMenu>
                <DropdownMenuTrigger asChild>
                  <Button variant="outline">
                    <Filter className="mr-2 h-4 w-4" />
                    Sort
                  </Button>
                </DropdownMenuTrigger>
                <DropdownMenuContent>
                  <DropdownMenuLabel>Sort by</DropdownMenuLabel>
                  <DropdownMenuItem
                    onClick={() => handleSort("createdAt", "desc")}
                  >
                    Newest First
                  </DropdownMenuItem>
                  <DropdownMenuItem
                    onClick={() => handleSort("createdAt", "asc")}
                  >
                    Oldest First
                  </DropdownMenuItem>
                  <DropdownMenuItem
                    onClick={() => handleSort("startDate", "desc")}
                  >
                    Start Date (Latest)
                  </DropdownMenuItem>
                  <DropdownMenuItem
                    onClick={() => handleSort("endDate", "asc")}
                  >
                    End Date (Soonest)
                  </DropdownMenuItem>
                  <DropdownMenuItem
                    onClick={() => handleSort("terms.rentAmount", "desc")}
                  >
                    Rent (High to Low)
                  </DropdownMenuItem>
                  <DropdownMenuItem
                    onClick={() => handleSort("terms.rentAmount", "asc")}
                  >
                    Rent (Low to High)
                  </DropdownMenuItem>
                </DropdownMenuContent>
              </DropdownMenu>
            </div>

            <div className="flex items-center gap-2">
              {selectedLeases.length > 0 && (
                <Button
                  variant="destructive"
                  size="sm"
                  onClick={() => setShowBulkDeleteDialog(true)}
                >
                  <Trash2 className="mr-2 h-4 w-4" />
                  Delete ({selectedLeases.length})
                </Button>
              )}

              <Button variant="outline" size="sm" onClick={fetchLeases}>
                <RefreshCw className="h-4 w-4" />
              </Button>

              <div className="flex items-center gap-2">
                <Button
                  variant={viewMode === "table" ? "default" : "outline"}
                  size="sm"
                  onClick={() => setViewMode("table")}
                >
                  <List className="h-4 w-4 mr-2" />
                  Table
                </Button>
                <Button
                  variant={viewMode === "cards" ? "default" : "outline"}
                  size="sm"
                  onClick={() => setViewMode("cards")}
                >
                  <LayoutGrid className="h-4 w-4 mr-2" />
                  Cards
                </Button>
              </div>
            </div>
          </div>
        </CardContent>
      </Card>

      {/* Leases List */}
      <Card className="gap-2">
        <CardHeader>
          <div className="flex items-center justify-between">
            <div>
              <CardTitle>Leases ({leases.length})</CardTitle>
              <CardDescription>
                A list of all leases and their current status
              </CardDescription>
            </div>
          </div>
        </CardHeader>
        <CardContent>
          {viewMode === "cards" ? (
            <div className="grid gap-4 md:grid-cols-2 lg:grid-cols-3">
              {loading ? (
                [...Array(6)].map((_, i) => (
                  <Card key={i} className="hover:shadow-md transition-shadow">
                    <CardHeader className="pb-3">
                      <div className="flex items-start justify-between">
                        <div className="space-y-1 flex-1">
                          <div className="flex items-center gap-2">
                            <Skeleton className="h-5 w-5 rounded" />
                            <Skeleton className="h-5 w-32" />
                          </div>
                          <div className="flex items-center gap-1">
                            <Skeleton className="h-4 w-24" />
                            <span>•</span>
                            <Skeleton className="h-4 w-20" />
                          </div>
                        </div>
                        <div className="flex items-center gap-2">
                          <Skeleton className="h-6 w-16 rounded-full" />
                          <Skeleton className="h-8 w-8 rounded" />
                        </div>
                      </div>
                    </CardHeader>
                    <CardContent className="space-y-4">
                      <div className="flex items-center gap-3">
                        <Skeleton className="h-10 w-10 rounded-full" />
                        <div className="space-y-1">
                          <Skeleton className="h-4 w-24" />
                          <Skeleton className="h-3 w-32" />
                        </div>
                      </div>
                      <div className="grid grid-cols-2 gap-4">
                        <div className="space-y-1">
                          <Skeleton className="h-3 w-16" />
                          <Skeleton className="h-4 w-20" />
                        </div>
                        <div className="space-y-1">
                          <Skeleton className="h-3 w-12" />
                          <Skeleton className="h-4 w-16" />
                        </div>
                      </div>
                      <div className="grid grid-cols-2 gap-4">
                        <div className="space-y-1">
                          <Skeleton className="h-3 w-14" />
                          <Skeleton className="h-4 w-18" />
                        </div>
                        <div className="space-y-1">
                          <Skeleton className="h-3 w-12" />
                          <Skeleton className="h-4 w-16" />
                        </div>
                      </div>
                      <div className="flex items-center justify-between pt-2">
                        <Skeleton className="h-6 w-20 rounded-full" />
                        <Skeleton className="h-8 w-20 rounded" />
                      </div>
                    </CardContent>
                  </Card>
                ))
              ) : leases.length === 0 ? (
                <div className="col-span-full text-center py-8">
                  <div className="text-muted-foreground">
                    {filters.search || filters.status
                      ? "No leases match your filters"
                      : "No leases found"}
                  </div>
                </div>
              ) : (
                leases.map((lease) => (
                  <LeaseCard
                    key={lease._id}
                    lease={lease}
                    onUpdate={handleLeaseUpdate}
                    onDelete={handleLeaseDelete}
                  />
                ))
              )}
            </div>
          ) : (
            <Table>
              <TableHeader>
                <TableRow className="bg-muted/50">
                  <TableHead>Property & Unit</TableHead>
                  <TableHead>Tenant</TableHead>
                  <TableHead>Status</TableHead>
                  <TableHead>Rent Amount</TableHead>
                  <TableHead>Start Date</TableHead>
                  <TableHead>End Date</TableHead>
                  <TableHead>Days Remaining</TableHead>
                  <TableHead className="text-right">Actions</TableHead>
                </TableRow>
              </TableHeader>
              <TableBody>
                {loading ? (
                  [...Array(8)].map((_, i) => (
                    <TableRow key={i}>
                      <TableCell>
                        <div className="flex items-center space-x-3">
                          <div className="flex-shrink-0">
                            <Skeleton className="h-8 w-8 rounded" />
                          </div>
                          <div className="space-y-1">
                            <Skeleton className="h-4 w-32" />
                            <Skeleton className="h-3 w-40" />
                          </div>
                        </div>
                      </TableCell>
                      <TableCell>
                        <div className="flex items-center space-x-3">
                          <Skeleton className="h-8 w-8 rounded-full" />
                          <div className="space-y-1">
                            <Skeleton className="h-4 w-24" />
                            <Skeleton className="h-3 w-32" />
                          </div>
                        </div>
                      </TableCell>
                      <TableCell>
                        <Skeleton className="h-6 w-16 rounded-full" />
                      </TableCell>
                      <TableCell>
                        <div className="space-y-1">
                          <Skeleton className="h-4 w-20" />
                          <Skeleton className="h-3 w-16" />
                        </div>
                      </TableCell>
                      <TableCell>
                        <Skeleton className="h-4 w-20" />
                      </TableCell>
                      <TableCell>
                        <Skeleton className="h-4 w-20" />
                      </TableCell>
                      <TableCell>
                        <Skeleton className="h-4 w-16" />
                      </TableCell>
                      <TableCell className="text-right">
                        <Skeleton className="h-8 w-8 rounded ml-auto" />
                      </TableCell>
                    </TableRow>
                  ))
                ) : leases.length === 0 ? (
                  <TableRow>
                    <TableCell colSpan={8} className="text-center py-8">
                      <div className="text-muted-foreground">
                        {filters.search || filters.status
                          ? "No leases match your filters"
                          : "No leases found"}
                      </div>
                    </TableCell>
                  </TableRow>
                ) : (
                  leases?.map((lease) => {
                    const getDaysRemaining = () => {
                      const endDate = new Date(lease.endDate);
                      const now = new Date();
                      const diffTime = endDate.getTime() - now.getTime();
                      const diffDays = Math.ceil(
                        diffTime / (1000 * 60 * 60 * 24)
                      );
                      return diffDays;
                    };

                    const daysRemaining = getDaysRemaining();

                    return (
                      <TableRow key={lease._id}>
                        <TableCell>
                          <div className="flex items-center space-x-3">
                            <div className="flex-shrink-0">
                              <Home className="h-8 w-8 text-muted-foreground" />
                            </div>
                            <div>
                              <div className="font-medium">
                                {lease.propertyId?.name ||
                                  "Property Not Available"}
                                {lease.unit && (
                                  <span className="ml-2 text-sm bg-blue-100 text-blue-800 px-2 py-1 rounded">
                                    Unit {lease.unit.unitNumber}
                                  </span>
                                )}
                              </div>
                              <div className="text-sm text-muted-foreground">
                                {lease.propertyId?.address ? (
                                  <>
                                    {lease.propertyId.address.street},{" "}
                                    {lease.propertyId.address.city}
                                  </>
                                ) : (
                                  "Address Not Available"
                                )}
                                {lease.unit && (
                                  <span className="ml-2">
                                    • {lease.unit.bedrooms}BR/
                                    {lease.unit.bathrooms}BA
                                  </span>
                                )}
                              </div>
                            </div>
                          </div>
                        </TableCell>
                        <TableCell>
                          <div className="flex items-center space-x-3">
                            <Avatar className="h-8 w-8">
                              <AvatarFallback>
                                {lease.tenantId?.firstName?.[0] || "T"}
                                {lease.tenantId?.lastName?.[0] || ""}
                              </AvatarFallback>
                            </Avatar>
                            <div>
                              <div className="font-medium">
                                {lease.tenantId?.firstName &&
                                lease.tenantId?.lastName
                                  ? `${lease.tenantId.firstName} ${lease.tenantId.lastName}`
                                  : "Unknown Tenant"}
                              </div>
                              <div className="text-sm text-muted-foreground">
                                {lease.tenantId?.email || "No email"}
                              </div>
                            </div>
                          </div>
                        </TableCell>
                        <TableCell>
                          <LeaseStatusBadge status={lease.status} />
                        </TableCell>
                        <TableCell>
                          <div className="font-medium">
                            $
                            {(
                              lease.unit?.rentAmount || lease.terms.rentAmount
                            ).toLocaleString()}
                          </div>
                          <div className="text-sm text-muted-foreground">
                            per month
                          </div>
                        </TableCell>
                        <TableCell>
                          <div className="text-sm">
                            {new Date(lease.startDate).toLocaleDateString()}
                          </div>
                        </TableCell>
                        <TableCell>
                          <div className="text-sm">
                            {new Date(lease.endDate).toLocaleDateString()}
                          </div>
                        </TableCell>
                        <TableCell>
                          <div
                            className={`text-sm font-medium ${
                              daysRemaining < 0
                                ? "text-red-600"
                                : daysRemaining <= 30
                                ? "text-orange-600"
                                : "text-green-600"
                            }`}
                          >
                            {daysRemaining < 0
                              ? "Expired"
                              : `${daysRemaining} days`}
                          </div>
                        </TableCell>
                        <TableCell className="text-right">
                          <DropdownMenu>
                            <DropdownMenuTrigger asChild>
                              <Button variant="ghost" size="sm">
                                <MoreHorizontal className="h-4 w-4" />
                              </Button>
                            </DropdownMenuTrigger>
                            <DropdownMenuContent align="end">
                              <DropdownMenuItem
                                onClick={() =>
                                  router.push(`/dashboard/leases/${lease._id}`)
                                }
                              >
                                <Eye className="h-4 w-4 mr-2" />
                                View Details
                              </DropdownMenuItem>

                              {/* Only show Edit for Admin/Manager */}
                              {session?.user?.role !== UserRole.TENANT && (
                                <DropdownMenuItem
                                  onClick={() =>
                                    router.push(
                                      `/dashboard/leases/${lease._id}/edit`
                                    )
                                  }
                                >
                                  <Edit className="h-4 w-4 mr-2" />
                                  Edit
                                </DropdownMenuItem>
                              )}

                              <DropdownMenuSeparator />
                              <DropdownMenuItem
                                onClick={() =>
                                  router.push(
                                    `/dashboard/leases/${lease._id}/invoice`
                                  )
                                }
                              >
                                <FileText className="h-4 w-4 mr-2" />
                                View Invoice
                              </DropdownMenuItem>
                              <LeaseInvoiceModal
                                lease={lease}
                                trigger={
                                  <DropdownMenuItem>
                                    <Download className="h-4 w-4 mr-2" />
                                    Download Invoice
                                  </DropdownMenuItem>
                                }
                              />

                              {/* Only show Delete for Admin/Manager */}
                              {session?.user?.role !== UserRole.TENANT && (
                                <>
                                  <DropdownMenuSeparator />
                                  <DeleteConfirmationDialog
                                    itemName={`${
                                      lease.propertyId?.name ||
                                      "Unknown Property"
                                    } - ${
                                      lease.tenantId?.firstName &&
                                      lease.tenantId?.lastName
                                        ? `${lease.tenantId.firstName} ${lease.tenantId.lastName}`
                                        : "Unknown Tenant"
                                    }`}
                                    itemType="lease"
                                    onConfirm={handleLeaseDelete}
                                    loading={isDeleting}
                                  >
                                    <DropdownMenuItem
                                      onSelect={(e) => {
                                        e.preventDefault();
                                        setLeaseToDelete(lease);
                                      }}
                                      className="text-destructive"
                                    >
                                      <Trash2 className="h-4 w-4 mr-2" />
                                      Delete
                                    </DropdownMenuItem>
                                  </DeleteConfirmationDialog>
                                </>
                              )}
                            </DropdownMenuContent>
                          </DropdownMenu>
                        </TableCell>
                      </TableRow>
                    );
                  })
                )}
              </TableBody>
            </Table>
          )}
        </CardContent>
      </Card>

      {/* Pagination */}
      {pagination.pages > 1 && (
        <div className="flex items-center justify-center gap-2">
          <Button
            variant="outline"
            disabled={!pagination.hasPrev}
            onClick={() => handlePageChange(pagination.page - 1)}
          >
            Previous
          </Button>
          <span className="text-sm text-muted-foreground">
            Page {pagination.page} of {pagination.pages}
          </span>
          <Button
            variant="outline"
            disabled={!pagination.hasNext}
            onClick={() => handlePageChange(pagination.page + 1)}
          >
            Next
          </Button>
        </div>
      )}

      {/* Bulk Delete Dialog */}
      <AlertDialog
        open={showBulkDeleteDialog}
        onOpenChange={setShowBulkDeleteDialog}
      >
        <AlertDialogContent>
          <AlertDialogHeader>
            <AlertDialogTitle>Delete Leases</AlertDialogTitle>
            <AlertDialogDescription>
              Are you sure you want to delete {selectedLeases.length} selected
              lease(s)? This action cannot be undone.
            </AlertDialogDescription>
          </AlertDialogHeader>
          <AlertDialogFooter>
            <AlertDialogCancel>Cancel</AlertDialogCancel>
            <AlertDialogAction
              onClick={handleBulkDelete}
              className="bg-red-600 hover:bg-red-700"
            >
              Delete Leases
            </AlertDialogAction>
          </AlertDialogFooter>
        </AlertDialogContent>
      </AlertDialog>
    </div>
  );
}
