"use client";

import { useState, useEffect } from "react";
import { useRouter } from "next/navigation";
import { toast } from "sonner";
import {
  Card,
  CardContent,
  CardDescription,
  CardHeader,
  CardTitle,
} from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select";
import { Badge } from "@/components/ui/badge";
import { Skeleton } from "@/components/ui/skeleton";
import {
  DropdownMenu,
  DropdownMenuContent,
  DropdownMenuItem,
  DropdownMenuLabel,
  DropdownMenuTrigger,
} from "@/components/ui/dropdown-menu";
import {
  AlertTriangle,
  Plus,
  Search,
  Filter,
  RefreshCw,
  Grid3X3,
  List,
  Calendar,
  Clock,
  RotateCcw,
  FileText,
} from "lucide-react";
import { LeaseStatus } from "@/types";
import {
  leaseService,
  LeaseResponse,
  PaginatedLeasesResponse,
  LeaseQueryParams,
} from "@/lib/services/lease.service";
import { LeaseCard } from "@/components/leases/LeaseCard";

export default function ExpiringLeasesPage() {
  const router = useRouter();
  const [leases, setLeases] = useState<LeaseResponse[]>([]);
  const [loading, setLoading] = useState(true);
  const [pagination, setPagination] = useState({
    page: 1,
    limit: 12,
    total: 0,
    pages: 0,
    hasNext: false,
    hasPrev: false,
  });
  const [filters, setFilters] = useState<LeaseQueryParams>({
    page: 1,
    limit: 12,
    search: "",
    status: LeaseStatus.ACTIVE,
    sortBy: "endDate",
    sortOrder: "asc",
  });
  const [viewMode, setViewMode] = useState<"grid" | "list">("grid");
  const [timeFilter, setTimeFilter] = useState<"30" | "60" | "90">("30");
  const [stats, setStats] = useState({
    expiring30Days: 0,
    expiring60Days: 0,
    expiring90Days: 0,
    totalExpiring: 0,
  });

  useEffect(() => {
    fetchExpiringLeases();
    fetchStats();
  }, [filters, timeFilter]);

  const fetchExpiringLeases = async () => {
    try {
      setLoading(true);

      // Calculate date range based on time filter
      const now = new Date();
      const endDate = new Date();
      endDate.setDate(now.getDate() + parseInt(timeFilter));

      const expiringFilters = {
        ...filters,
        endDate: endDate.toISOString().split("T")[0],
      };

      const response: PaginatedLeasesResponse = await leaseService.getLeases(
        expiringFilters
      );

      // Filter leases that are actually expiring within the time frame
      const filteredLeases = response.data.filter((lease) => {
        const leaseEndDate = new Date(lease.endDate);
        const daysUntilExpiry = Math.ceil(
          (leaseEndDate.getTime() - now.getTime()) / (1000 * 60 * 60 * 24)
        );
        return daysUntilExpiry <= parseInt(timeFilter) && daysUntilExpiry >= 0;
      });

      setLeases(filteredLeases);
      setPagination(response.pagination);
    } catch (error) {
      toast.error("Failed to fetch expiring leases");
    } finally {
      setLoading(false);
    }
  };

  const fetchStats = async () => {
    try {
      const statsData = await leaseService.getLeaseStats();
      setStats({
        expiring30Days: statsData.expiringThisMonth,
        expiring60Days: statsData.expiringThisMonth + 5, // Mock data
        expiring90Days: statsData.expiringThisMonth + 8, // Mock data
        totalExpiring: statsData.expiringThisMonth,
      });
    } catch (error) {
      // Failed to fetch lease stats
    }
  };

  const handleSearch = (value: string) => {
    setFilters((prev) => ({ ...prev, search: value, page: 1 }));
  };

  const handleSort = (sortBy: string, sortOrder: "asc" | "desc") => {
    setFilters((prev) => ({ ...prev, sortBy, sortOrder, page: 1 }));
  };

  const handlePageChange = (page: number) => {
    setFilters((prev) => ({ ...prev, page }));
  };

  const handleLeaseUpdate = () => {
    fetchExpiringLeases();
    fetchStats();
  };

  const getDaysUntilExpiry = (endDate: string) => {
    const now = new Date();
    const expiry = new Date(endDate);
    const diffTime = expiry.getTime() - now.getTime();
    return Math.ceil(diffTime / (1000 * 60 * 60 * 24));
  };

  const getUrgencyColor = (days: number) => {
    if (days <= 7) return "text-red-600";
    if (days <= 30) return "text-orange-600";
    return "text-yellow-600";
  };

  if (loading && leases.length === 0) {
    return (
      <div className="space-y-6">
        <div className="flex items-center justify-between">
          <div>
            <h1 className="text-3xl font-bold tracking-tight">
              Expiring Leases
            </h1>
            <p className="text-muted-foreground">
              Manage leases that are expiring soon
            </p>
          </div>
          <Skeleton className="h-10 w-32" />
        </div>

        <div className="grid grid-cols-1 md:grid-cols-4 gap-4">
          {Array.from({ length: 4 }).map((_, i) => (
            <Skeleton key={i} className="h-24" />
          ))}
        </div>

        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
          {Array.from({ length: 6 }).map((_, i) => (
            <Skeleton key={i} className="h-64" />
          ))}
        </div>
      </div>
    );
  }

  return (
    <div className="space-y-6">
      {/* Header */}
      <div className="flex items-center justify-between">
        <div>
          <h1 className="text-3xl font-bold tracking-tight">Expiring Leases</h1>
          <p className="text-muted-foreground">
            Manage leases that are expiring soon and require attention
          </p>
        </div>
        <Button onClick={() => router.push("/dashboard/leases/new")}>
          <Plus className="mr-2 h-4 w-4" />
          Create Lease
        </Button>
      </div>

      {/* Stats Cards */}
      <div className="grid grid-cols-1 md:grid-cols-4 gap-4">
        <Card>
          <CardContent>
            <div className="flex items-center justify-between">
              <div>
                <p className="text-sm font-medium text-muted-foreground">
                  Next 30 Days
                </p>
                <p className="text-2xl font-bold text-red-600">
                  {stats.expiring30Days}
                </p>
              </div>
              <AlertTriangle className="h-8 w-8 text-red-600" />
            </div>
          </CardContent>
        </Card>

        <Card>
          <CardContent>
            <div className="flex items-center justify-between">
              <div>
                <p className="text-sm font-medium text-muted-foreground">
                  Next 60 Days
                </p>
                <p className="text-2xl font-bold text-orange-600">
                  {stats.expiring60Days}
                </p>
              </div>
              <Calendar className="h-8 w-8 text-orange-600" />
            </div>
          </CardContent>
        </Card>

        <Card>
          <CardContent>
            <div className="flex items-center justify-between">
              <div>
                <p className="text-sm font-medium text-muted-foreground">
                  Next 90 Days
                </p>
                <p className="text-2xl font-bold text-yellow-600">
                  {stats.expiring90Days}
                </p>
              </div>
              <Clock className="h-8 w-8 text-yellow-600" />
            </div>
          </CardContent>
        </Card>

        <Card>
          <CardContent>
            <div className="flex items-center justify-between">
              <div>
                <p className="text-sm font-medium text-muted-foreground">
                  Renewal Ready
                </p>
                <p className="text-2xl font-bold text-blue-600">
                  {Math.floor(stats.totalExpiring * 0.7)}
                </p>
              </div>
              <RotateCcw className="h-8 w-8 text-blue-600" />
            </div>
          </CardContent>
        </Card>
      </div>

      {/* Filters and Actions */}
      <Card>
        <CardContent className="p-4">
          <div className="flex flex-col md:flex-row gap-4 items-center justify-between">
            <div className="flex flex-1 gap-4 items-center">
              <div className="relative flex-1 max-w-sm">
                <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 text-muted-foreground h-4 w-4" />
                <Input
                  placeholder="Search expiring leases..."
                  value={filters.search}
                  onChange={(e) => handleSearch(e.target.value)}
                  className="pl-10"
                />
              </div>

              <Select
                value={timeFilter}
                onValueChange={(value: "30" | "60" | "90") =>
                  setTimeFilter(value)
                }
              >
                <SelectTrigger className="w-40">
                  <SelectValue placeholder="Time Range" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="30">Next 30 Days</SelectItem>
                  <SelectItem value="60">Next 60 Days</SelectItem>
                  <SelectItem value="90">Next 90 Days</SelectItem>
                </SelectContent>
              </Select>

              <DropdownMenu>
                <DropdownMenuTrigger asChild>
                  <Button variant="outline">
                    <Filter className="mr-2 h-4 w-4" />
                    Sort
                  </Button>
                </DropdownMenuTrigger>
                <DropdownMenuContent>
                  <DropdownMenuLabel>Sort by</DropdownMenuLabel>
                  <DropdownMenuItem
                    onClick={() => handleSort("endDate", "asc")}
                  >
                    Expiry Date (Soonest)
                  </DropdownMenuItem>
                  <DropdownMenuItem
                    onClick={() => handleSort("endDate", "desc")}
                  >
                    Expiry Date (Latest)
                  </DropdownMenuItem>
                  <DropdownMenuItem
                    onClick={() => handleSort("terms.rentAmount", "desc")}
                  >
                    Rent (High to Low)
                  </DropdownMenuItem>
                  <DropdownMenuItem
                    onClick={() => handleSort("terms.rentAmount", "asc")}
                  >
                    Rent (Low to High)
                  </DropdownMenuItem>
                </DropdownMenuContent>
              </DropdownMenu>
            </div>

            <div className="flex items-center gap-2">
              <Button variant="outline" size="sm" onClick={fetchExpiringLeases}>
                <RefreshCw className="h-4 w-4" />
              </Button>

              <div className="flex border rounded-md">
                <Button
                  variant={viewMode === "grid" ? "default" : "ghost"}
                  size="sm"
                  onClick={() => setViewMode("grid")}
                  className="rounded-r-none"
                >
                  <Grid3X3 className="h-4 w-4" />
                </Button>
                <Button
                  variant={viewMode === "list" ? "default" : "ghost"}
                  size="sm"
                  onClick={() => setViewMode("list")}
                  className="rounded-l-none"
                >
                  <List className="h-4 w-4" />
                </Button>
              </div>
            </div>
          </div>
        </CardContent>
      </Card>

      {/* Expiring Leases Grid/List */}
      {leases.length === 0 ? (
        <Card className="p-0">
          <CardContent className="p-12 text-center">
            <Calendar className="mx-auto h-12 w-12 text-muted-foreground mb-4" />
            <h3 className="text-lg font-semibold mb-2">
              No expiring leases found
            </h3>
            <p className="text-muted-foreground mb-4">
              {filters.search
                ? "No expiring leases match your search criteria."
                : `There are no leases expiring in the next ${timeFilter} days.`}
            </p>
            <Button onClick={() => router.push("/dashboard/leases")}>
              <FileText className="mr-2 h-4 w-4" />
              View All Leases
            </Button>
          </CardContent>
        </Card>
      ) : (
        <div
          className={
            viewMode === "grid"
              ? "grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6"
              : "space-y-4"
          }
        >
          {leases?.map((lease) => {
            const daysUntilExpiry = getDaysUntilExpiry(lease.endDate);
            return (
              <div key={lease._id} className="relative">
                <LeaseCard
                  lease={lease}
                  onUpdate={handleLeaseUpdate}
                  onDelete={() => {}}
                />
                <div className="absolute top-2 right-2">
                  <Badge
                    variant="destructive"
                    className={`${getUrgencyColor(
                      daysUntilExpiry
                    )} bg-white border`}
                  >
                    {daysUntilExpiry} days left
                  </Badge>
                </div>
              </div>
            );
          })}
        </div>
      )}

      {/* Pagination */}
      {pagination.pages > 1 && (
        <div className="flex items-center justify-center gap-2">
          <Button
            variant="outline"
            disabled={!pagination.hasPrev}
            onClick={() => handlePageChange(pagination.page - 1)}
          >
            Previous
          </Button>
          <span className="text-sm text-muted-foreground">
            Page {pagination.page} of {pagination.pages}
          </span>
          <Button
            variant="outline"
            disabled={!pagination.hasNext}
            onClick={() => handlePageChange(pagination.page + 1)}
          >
            Next
          </Button>
        </div>
      )}
    </div>
  );
}
