/**
 * PropertyPro - File Upload API Route
 * Handle file uploads for avatars, logos, and documents
 */

export const dynamic = "force-dynamic";

import { NextRequest, NextResponse } from "next/server";
import { auth } from "@/lib/auth";
import { writeFile, mkdir } from "fs/promises";
import { existsSync } from "fs";
import path from "path";
import { v4 as uuidv4 } from "uuid";
import {
  createSuccessResponse,
  createErrorResponse,
  handleApiError,
} from "@/lib/api-utils";

// File upload configuration
const UPLOAD_CONFIG = {
  maxFileSize: 10 * 1024 * 1024, // 10MB
  allowedImageTypes: [
    "image/jpeg",
    "image/jpg",
    "image/png",
    "image/webp",
    "image/svg+xml",
  ],
  allowedDocumentTypes: [
    "application/pdf",
    "application/msword",
    "application/vnd.openxmlformats-officedocument.wordprocessingml.document",
  ],
  uploadDir: process.env.UPLOAD_DIR || "./public/uploads",
  baseUrl: process.env.NEXT_PUBLIC_APP_URL || "http://localhost:3000",
};

/**
 * Validate file type and size
 */
function validateFile(
  file: File,
  uploadType: string
): { isValid: boolean; error?: string } {
  // Check file size
  if (file.size > UPLOAD_CONFIG.maxFileSize) {
    return {
      isValid: false,
      error: `File size exceeds ${
        UPLOAD_CONFIG.maxFileSize / 1024 / 1024
      }MB limit`,
    };
  }

  // Check file type based on upload type
  const allowedTypes =
    uploadType === "document"
      ? [
          ...UPLOAD_CONFIG.allowedImageTypes,
          ...UPLOAD_CONFIG.allowedDocumentTypes,
        ]
      : UPLOAD_CONFIG.allowedImageTypes;

  if (!allowedTypes.includes(file.type)) {
    return {
      isValid: false,
      error: `File type ${file.type} is not allowed`,
    };
  }

  return { isValid: true };
}

/**
 * Generate unique filename
 */
function generateFilename(originalName: string, uploadType: string): string {
  const ext = path.extname(originalName);
  const uuid = uuidv4();
  return `${uploadType}_${uuid}${ext}`;
}

/**
 * Save file to disk
 */
async function saveFile(
  buffer: Buffer,
  filename: string,
  uploadType: string
): Promise<string> {
  const typeDir = path.join(UPLOAD_CONFIG.uploadDir, uploadType);

  // Ensure directory exists
  if (!existsSync(typeDir)) {
    await mkdir(typeDir, { recursive: true });
  }

  const filePath = path.join(typeDir, filename);
  await writeFile(filePath, buffer);

  // Return public URL
  return `${UPLOAD_CONFIG.baseUrl}/uploads/${uploadType}/${filename}`;
}

/**
 * Delete old file if exists
 */
async function deleteOldFile(oldUrl: string): Promise<void> {
  try {
    if (oldUrl && oldUrl.includes("/uploads/")) {
      const relativePath = oldUrl.split("/uploads/")[1];
      const fullPath = path.join(UPLOAD_CONFIG.uploadDir, relativePath);

      if (existsSync(fullPath)) {
        const fs = await import("fs/promises");
        await fs.unlink(fullPath);
      }
    }
  } catch (error) {
    console.error("Error deleting old file:", error);
    // Don't throw error, just log it
  }
}

// ============================================================================
// POST /api/upload - Upload files
// ============================================================================
export async function POST(request: NextRequest) {
  try {
    const session = await auth();

    if (!session?.user?.id) {
      return createErrorResponse("Unauthorized", 401);
    }

    const formData = await request.formData();
    const file = formData.get("file") as File;
    const uploadType = formData.get("type") as string;
    const oldFileUrl = formData.get("oldFileUrl") as string;

    if (!file) {
      return createErrorResponse("No file provided", 400);
    }

    if (!uploadType) {
      return createErrorResponse("Upload type is required", 400);
    }

    // Validate upload type
    const validTypes = ["avatar", "logo", "favicon", "document"];
    if (!validTypes.includes(uploadType)) {
      return createErrorResponse("Invalid upload type", 400);
    }

    // Validate file
    const validation = validateFile(file, uploadType);
    if (!validation.isValid) {
      return createErrorResponse(validation.error!, 400);
    }

    // Convert file to buffer
    const bytes = await file.arrayBuffer();
    const buffer = Buffer.from(bytes);

    // Generate filename
    const filename = generateFilename(file.name, uploadType);

    // Save file (no image processing)
    const fileUrl = await saveFile(buffer, filename, uploadType);

    // Delete old file if provided
    if (oldFileUrl) {
      await deleteOldFile(oldFileUrl);
    }

    return createSuccessResponse({
      url: fileUrl,
      filename,
      originalName: file.name,
      size: buffer.length,
      type: file.type,
      uploadType,
      message: "File uploaded successfully",
    });
  } catch (error) {
    console.error("Upload error:", error);
    return handleApiError(error);
  }
}

// ============================================================================
// DELETE /api/upload - Delete files
// ============================================================================
export async function DELETE(request: NextRequest) {
  try {
    const session = await auth();

    if (!session?.user?.id) {
      return createErrorResponse("Unauthorized", 401);
    }

    const { searchParams } = new URL(request.url);
    const fileUrl = searchParams.get("url");

    if (!fileUrl) {
      return createErrorResponse("File URL is required", 400);
    }

    // Delete file
    await deleteOldFile(fileUrl);

    return createSuccessResponse({
      message: "File deleted successfully",
    });
  } catch (error) {
    console.error("Delete error:", error);
    return handleApiError(error);
  }
}

// ============================================================================
// GET /api/upload - Get upload configuration
// ============================================================================
export async function GET() {
  try {
    return createSuccessResponse({
      maxFileSize: UPLOAD_CONFIG.maxFileSize,
      allowedImageTypes: UPLOAD_CONFIG.allowedImageTypes,
      allowedDocumentTypes: UPLOAD_CONFIG.allowedDocumentTypes,
    });
  } catch (error) {
    console.error("Get upload config error:", error);
    return handleApiError(error);
  }
}
