/**
 * PropertyPro - Tenant Notifications API
 * Handles fetching and managing payment notifications for tenants
 */

import { NextRequest, NextResponse } from "next/server";
import { auth } from "@/lib/auth";
import connectDB from "@/lib/mongodb";
import PaymentNotification from "@/models/PaymentNotification";
import { UserRole } from "@/types";

// ============================================================================
// GET - Fetch tenant's payment notifications
// ============================================================================

export async function GET(request: NextRequest) {
  try {
    const session = await auth();

    if (!session?.user) {
      return NextResponse.json(
        { error: "Authentication required" },
        { status: 401 }
      );
    }

    if (session.user.role !== UserRole.TENANT) {
      return NextResponse.json(
        { error: "Access denied. Tenant role required." },
        { status: 403 }
      );
    }

    const { searchParams } = new URL(request.url);
    const page = parseInt(searchParams.get("page") || "1");
    const limit = parseInt(searchParams.get("limit") || "10");
    const type = searchParams.get("type");
    const status = searchParams.get("status");

    await connectDB();

    // Build query
    const query: any = { tenantId: session.user.id };

    if (type) {
      query.type = type;
    }

    if (status) {
      query.status = status;
    }

    // Calculate pagination
    const skip = (page - 1) * limit;

    // Fetch notifications
    const notifications = await PaymentNotification.find(query)
      .populate("paymentId", "amount type dueDate")
      .sort({ createdAt: -1 })
      .skip(skip)
      .limit(limit)
      .lean();

    // Get total count
    const total = await PaymentNotification.countDocuments(query);
    const totalPages = Math.ceil(total / limit);

    return NextResponse.json({
      success: true,
      data: {
        notifications,
        pagination: {
          page,
          limit,
          total,
          totalPages,
        },
      },
    });
  } catch (error) {
    return NextResponse.json(
      {
        error:
          error instanceof Error
            ? error.message
            : "Failed to fetch notifications",
      },
      { status: 500 }
    );
  }
}

// ============================================================================
// POST - Create a new notification (typically used by system)
// ============================================================================

export async function POST(request: NextRequest) {
  try {
    const session = await auth();

    if (!session?.user) {
      return NextResponse.json(
        { error: "Authentication required" },
        { status: 401 }
      );
    }

    // Only allow system/admin to create notifications directly
    if (
      session.user.role !== UserRole.ADMIN &&
      session.user.role !== UserRole.MANAGER
    ) {
      return NextResponse.json(
        { error: "Access denied. Admin or Property Manager role required." },
        { status: 403 }
      );
    }

    const body = await request.json();
    const {
      tenantId,
      paymentId,
      type,
      scheduledDate,
      emailAddress,
      subject,
      message,
    } = body;

    // Validation
    if (
      !tenantId ||
      !paymentId ||
      !type ||
      !emailAddress ||
      !subject ||
      !message
    ) {
      return NextResponse.json(
        { error: "Missing required fields" },
        { status: 400 }
      );
    }

    if (!["reminder", "overdue", "confirmation", "receipt"].includes(type)) {
      return NextResponse.json(
        { error: "Invalid notification type" },
        { status: 400 }
      );
    }

    await connectDB();

    // Create notification
    const notification = new PaymentNotification({
      tenantId,
      paymentId,
      type,
      status: "pending",
      scheduledDate: scheduledDate || new Date(),
      emailAddress,
      subject,
      message,
    });

    await notification.save();

    return NextResponse.json({
      success: true,
      data: notification,
      message: "Notification created successfully",
    });
  } catch (error) {
    return NextResponse.json(
      {
        error:
          error instanceof Error
            ? error.message
            : "Failed to create notification",
      },
      { status: 500 }
    );
  }
}

// ============================================================================
// PUT - Mark notifications as read (bulk operation)
// ============================================================================

export async function PUT(request: NextRequest) {
  try {
    const session = await auth();

    if (!session?.user) {
      return NextResponse.json(
        { error: "Authentication required" },
        { status: 401 }
      );
    }

    if (session.user.role !== UserRole.TENANT) {
      return NextResponse.json(
        { error: "Access denied. Tenant role required." },
        { status: 403 }
      );
    }

    const body = await request.json();
    const { notificationIds, markAsRead = true } = body;

    if (!notificationIds || !Array.isArray(notificationIds)) {
      return NextResponse.json(
        { error: "Invalid notification IDs" },
        { status: 400 }
      );
    }

    await connectDB();

    // Update notifications
    const result = await PaymentNotification.updateMany(
      {
        _id: { $in: notificationIds },
        tenantId: session.user.id,
      },
      {
        $set: {
          isRead: markAsRead,
          readAt: markAsRead ? new Date() : null,
        },
      }
    );

    return NextResponse.json({
      success: true,
      data: {
        modifiedCount: result.modifiedCount,
      },
      message: `${result.modifiedCount} notifications updated successfully`,
    });
  } catch (error) {
    return NextResponse.json(
      {
        error:
          error instanceof Error
            ? error.message
            : "Failed to update notifications",
      },
      { status: 500 }
    );
  }
}

// ============================================================================
// DELETE - Delete notifications (bulk operation)
// ============================================================================

export async function DELETE(request: NextRequest) {
  try {
    const session = await auth();

    if (!session?.user) {
      return NextResponse.json(
        { error: "Authentication required" },
        { status: 401 }
      );
    }

    if (session.user.role !== UserRole.TENANT) {
      return NextResponse.json(
        { error: "Access denied. Tenant role required." },
        { status: 403 }
      );
    }

    const { searchParams } = new URL(request.url);
    const notificationIds = searchParams.get("ids")?.split(",");
    const deleteAll = searchParams.get("all") === "true";

    await connectDB();

    let result;

    if (deleteAll) {
      // Delete all notifications for the tenant
      result = await PaymentNotification.deleteMany({
        tenantId: session.user.id,
      });
    } else if (notificationIds && notificationIds.length > 0) {
      // Delete specific notifications
      result = await PaymentNotification.deleteMany({
        _id: { $in: notificationIds },
        tenantId: session.user.id,
      });
    } else {
      return NextResponse.json(
        { error: "No notifications specified for deletion" },
        { status: 400 }
      );
    }

    return NextResponse.json({
      success: true,
      data: {
        deletedCount: result.deletedCount,
      },
      message: `${result.deletedCount} notifications deleted successfully`,
    });
  } catch (error) {
    return NextResponse.json(
      {
        error:
          error instanceof Error
            ? error.message
            : "Failed to delete notifications",
      },
      { status: 500 }
    );
  }
}
