/**
 * PropertyPro - Privacy Settings API Routes
 * CRUD operations for user privacy settings using separate collection
 */

export const dynamic = "force-dynamic";

import { NextRequest } from "next/server";
import { auth } from "@/lib/auth";
import connectDB from "@/lib/mongodb";
import { PrivacySettings } from "@/models";
import {
  createSuccessResponse,
  createErrorResponse,
  handleApiError,
  parseRequestBody,
} from "@/lib/api-utils";
import { privacySettingsSchema } from "@/lib/validations";
import {
  rateLimit,
  rateLimitConfigs,
  createRateLimitResponse,
  addSecurityHeaders,
  sanitizeInput,
  userBasedKeyGenerator,
} from "@/lib/rate-limit";

// ============================================================================
// GET /api/settings/privacy - Get user privacy settings
// ============================================================================
export async function GET(request: NextRequest) {
  try {
    await connectDB();
    const session = await auth();

    if (!session?.user?.id) {
      return createErrorResponse("Unauthorized", 401);
    }

    const userId = session.user.id;

    // Find existing privacy settings
    let privacySettings = await PrivacySettings.findByUserId(userId);

    // If no settings exist, create default settings
    if (!privacySettings) {
      privacySettings = await PrivacySettings.createDefaultPrivacy(userId);
    }

    const response = createSuccessResponse({
      settings: privacySettings,
      message: "Privacy settings retrieved successfully",
    });

    return addSecurityHeaders(response);
  } catch (error) {
    console.error("Get privacy settings error:", error);
    return handleApiError(error);
  }
}

// ============================================================================
// PUT /api/settings/privacy - Update privacy settings
// ============================================================================
export async function PUT(request: NextRequest) {
  try {
    await connectDB();
    const session = await auth();

    if (!session?.user?.id) {
      return createErrorResponse("Unauthorized", 401);
    }

    const userId = session.user.id;

    // Rate limiting
    const limiter = rateLimit({
      ...rateLimitConfigs.settingsUpdate,
      keyGenerator: () => userBasedKeyGenerator(userId),
    });

    const rateLimitResult = limiter.check(request);
    if (!rateLimitResult.success) {
      return createRateLimitResponse(
        "Too many privacy updates. Please try again later.",
        rateLimitResult.resetTime
      );
    }

    // Parse and validate request body
    const { success, data: body, error } = await parseRequestBody(request);
    if (!success) {
      return createErrorResponse(error!, 400);
    }

    // Sanitize input data
    const sanitizedBody = sanitizeInput(body);

    // Validate the request body
    const validation = privacySettingsSchema.safeParse(sanitizedBody);
    if (!validation.success) {
      const errors = validation.error.errors.map((e) => e.message).join(", ");
      return createErrorResponse(`Validation failed: ${errors}`, 400);
    }

    const privacyData = validation.data;

    // Find existing settings
    let privacySettings = await PrivacySettings.findByUserId(userId);

    if (!privacySettings) {
      // Create new settings if they don't exist
      privacySettings = await PrivacySettings.create({
        userId,
        ...privacyData,
        createdBy: userId,
      });
    } else {
      // Update existing settings
      await privacySettings.updatePrivacy({
        ...privacyData,
        updatedBy: userId,
      });
    }

    const response = createSuccessResponse({
      settings: privacySettings,
      message: "Privacy settings updated successfully",
    });

    return addSecurityHeaders(response);
  } catch (error) {
    console.error("Update privacy settings error:", error);
    return handleApiError(error);
  }
}

// ============================================================================
// PATCH /api/settings/privacy - Partial update privacy settings
// ============================================================================
export async function PATCH(request: NextRequest) {
  try {
    await connectDB();
    const session = await auth();

    if (!session?.user?.id) {
      return createErrorResponse("Unauthorized", 401);
    }

    const userId = session.user.id;

    // Rate limiting
    const limiter = rateLimit({
      ...rateLimitConfigs.settingsUpdate,
      keyGenerator: () => userBasedKeyGenerator(userId),
    });

    const rateLimitResult = limiter.check(request);
    if (!rateLimitResult.success) {
      return createRateLimitResponse(
        "Too many privacy updates. Please try again later.",
        rateLimitResult.resetTime
      );
    }

    // Parse and validate request body
    const { success, data: body, error } = await parseRequestBody(request);
    if (!success) {
      return createErrorResponse(error!, 400);
    }

    // Sanitize input data
    const sanitizedBody = sanitizeInput(body);

    // Validate the request body (partial validation)
    const validation = privacySettingsSchema.partial().safeParse(sanitizedBody);
    if (!validation.success) {
      const errors = validation.error.errors.map((e) => e.message).join(", ");
      return createErrorResponse(`Validation failed: ${errors}`, 400);
    }

    const privacyData = validation.data;

    // Find existing settings
    let privacySettings = await PrivacySettings.findByUserId(userId);

    if (!privacySettings) {
      // Create new settings if they don't exist
      privacySettings = await PrivacySettings.createDefaultPrivacy(userId);
      // Apply the partial update
      await privacySettings.updatePrivacy({
        ...privacyData,
        createdBy: userId,
      });
    } else {
      // Update existing settings
      await privacySettings.updatePrivacy({
        ...privacyData,
        updatedBy: userId,
      });
    }

    const response = createSuccessResponse({
      settings: privacySettings,
      message: "Privacy settings updated successfully",
    });

    return addSecurityHeaders(response);
  } catch (error) {
    console.error("Patch privacy settings error:", error);
    return handleApiError(error);
  }
}

// ============================================================================
// DELETE /api/settings/privacy - Reset privacy settings to defaults
// ============================================================================
export async function DELETE(request: NextRequest) {
  try {
    await connectDB();
    const session = await auth();

    if (!session?.user?.id) {
      return createErrorResponse("Unauthorized", 401);
    }

    const userId = session.user.id;

    // Rate limiting
    const limiter = rateLimit({
      ...rateLimitConfigs.settingsUpdate,
      keyGenerator: () => userBasedKeyGenerator(userId),
    });

    const rateLimitResult = limiter.check(request);
    if (!rateLimitResult.success) {
      return createRateLimitResponse(
        "Too many privacy updates. Please try again later.",
        rateLimitResult.resetTime
      );
    }

    // Find and deactivate existing settings
    const existingSettings = await PrivacySettings.findByUserId(userId);
    if (existingSettings) {
      existingSettings.isActive = false;
      await existingSettings.save();
    }

    // Create new default settings
    const defaultSettings = await PrivacySettings.createDefaultPrivacy(userId);
    defaultSettings.createdBy = userId;
    await defaultSettings.save();

    const response = createSuccessResponse({
      settings: defaultSettings,
      message: "Privacy settings reset to defaults successfully",
    });

    return addSecurityHeaders(response);
  } catch (error) {
    console.error("Delete privacy settings error:", error);
    return handleApiError(error);
  }
}
