/**
 * PropertyPro - Active Leases API Route
 * Get all active leases with pagination and filtering
 */

import { NextRequest } from "next/server";
import { Lease } from "@/models";
import { UserRole, LeaseStatus } from "@/types";
import {
  createSuccessResponse,
  createErrorResponse,
  handleApiError,
  withRoleAndDB,
  parsePaginationParams,
  paginateQuery,
} from "@/lib/api-utils";
import { paginationSchema, validateSchema } from "@/lib/validations";

// ============================================================================
// GET /api/leases/active - Get all active leases with pagination and filtering
// ============================================================================

export const GET = withRoleAndDB([
  UserRole.ADMIN,
  UserRole.MANAGER,
  UserRole.TENANT,
])(async (user, request: NextRequest) => {
  try {
    const { searchParams } = new URL(request.url);
    const paginationParams = parsePaginationParams(searchParams);

    // Parse filter parameters
    const filterParams = {
      ...paginationParams,
      propertyId: searchParams.get("propertyId") || undefined,
      tenantId: searchParams.get("tenantId") || undefined,
      search: searchParams.get("search") || undefined,
    };

    // Validate filter parameters
    const validation = validateSchema(paginationSchema, filterParams);
    if (!validation.success) {
      return createErrorResponse(validation.errors.join(", "), 400);
    }

    const filters = validation.data;

    // Build query based on user role and filters
    let query: any = {
      status: LeaseStatus.ACTIVE, // Only active leases
    };

    // Role-based filtering for single company architecture
    if (user.role === UserRole.TENANT) {
      // For tenant users, filter leases by their user ID directly
      query.tenantId = user.id;
    }
    // Admin and Manager can see all company leases - no filtering needed

    // Apply filters
    if (filters.propertyId) query.propertyId = filters.propertyId;
    if (filters.tenantId) query.tenantId = filters.tenantId;

    // Handle search
    if (filters.search) {
      query.$or = [
        { "propertyId.name": { $regex: filters.search, $options: "i" } },
        { "tenantId.firstName": { $regex: filters.search, $options: "i" } },
        { "tenantId.lastName": { $regex: filters.search, $options: "i" } },
        { "tenantId.email": { $regex: filters.search, $options: "i" } },
      ];
    }

    // Execute paginated query
    const result = await paginateQuery(Lease, query, filters);

    // Populate property and tenant information
    const populatedData = await Lease.populate(result.data, [
      {
        path: "propertyId",
        select:
          "name address type bedrooms bathrooms squareFootage units isMultiUnit totalUnits",
        populate: {
          path: "ownerId",
          select: "firstName lastName email",
        },
      },
      {
        path: "tenantId",
        select:
          "firstName lastName email phone avatar dateOfBirth employmentInfo emergencyContacts creditScore backgroundCheckStatus moveInDate moveOutDate applicationDate tenantStatus",
      },
    ]);

    // Filter out leases with null tenants (orphaned leases) and add unit information
    const validLeases = populatedData
      .filter((lease: any) => lease.tenantId !== null)
      .map((lease: any) => {
        // Convert to plain object if it's a Mongoose document
        const leaseObj = lease.toObject ? lease.toObject() : lease;

        // Add unit information if property has units
        if (
          leaseObj.propertyId?.units &&
          Array.isArray(leaseObj.propertyId.units)
        ) {
          const unitInfo = leaseObj.propertyId.units.find(
            (unit: any) => unit._id.toString() === leaseObj.unitId?.toString()
          );
          if (unitInfo) {
            // Add unit information as a virtual property
            (leaseObj as any).unit = unitInfo;
          }
        }

        return leaseObj;
      });

    return createSuccessResponse(
      {
        data: validLeases,
        pagination: result.pagination,
      },
      "Active leases retrieved successfully"
    );
  } catch (error) {
    return handleApiError(error);
  }
});
