import { NextRequest, NextResponse } from "next/server";
import { auth } from "@/lib/auth";
import { UserRole, PropertyStatus } from "@/types";
import Property from "@/models/Property";
import connectDB from "@/lib/mongodb";

interface OccupancyData {
  totalUnits: number;
  occupiedUnits: number;
  vacantUnits: number;
  maintenanceUnits: number;
  unavailableUnits: number;
  occupancyRate: number;
  propertyBreakdown: Array<{
    propertyName: string;
    totalUnits: number;
    occupiedUnits: number;
    vacantUnits: number;
    maintenanceUnits: number;
    unavailableUnits: number;
    occupancyRate: number;
  }>;
}

export async function GET(request: NextRequest) {
  try {
    const session = await auth();

    if (!session?.user) {
      return NextResponse.json({ error: "Unauthorized" }, { status: 401 });
    }

    const userRole = session.user.role as UserRole;

    // Only allow ADMIN and MANAGER roles to access analytics
    if (![UserRole.ADMIN, UserRole.MANAGER, UserRole.ADMIN].includes(userRole)) {
      return NextResponse.json({ error: "Forbidden" }, { status: 403 });
    }

    const { searchParams } = new URL(request.url);
    const property = searchParams.get("property") || "all";

    // Connect to database
    await connectDB();

    // Build property filter
    let propertyFilter: any = { deletedAt: null };
    if (property !== "all") {
      propertyFilter._id = property;
    }

    // Fetch all properties
    const properties = await Property.find(propertyFilter).select(
      "name units totalUnits status"
    ).lean();

    // Calculate overall statistics
    let totalUnits = 0;
    let occupiedUnits = 0;
    let vacantUnits = 0;
    let maintenanceUnits = 0;
    let unavailableUnits = 0;

    // Property breakdown data
    const propertyBreakdown = properties.map((property) => {
      const units = property.units || [];
      const propertyTotalUnits = units.length || property.totalUnits || 1;

      // Count units by status
      const propertyOccupied = units.filter(
        (unit: any) => unit.status === PropertyStatus.OCCUPIED
      ).length;

      const propertyVacant = units.filter(
        (unit: any) => unit.status === PropertyStatus.AVAILABLE
      ).length;

      const propertyMaintenance = units.filter(
        (unit: any) => unit.status === PropertyStatus.MAINTENANCE
      ).length;

      const propertyUnavailable = units.filter(
        (unit: any) => unit.status === PropertyStatus.UNAVAILABLE
      ).length;

      // If no units array, use property status
      let finalOccupied = propertyOccupied;
      let finalVacant = propertyVacant;
      let finalMaintenance = propertyMaintenance;
      let finalUnavailable = propertyUnavailable;

      if (units.length === 0) {
        // Single unit property - use property status
        switch (property.status) {
          case PropertyStatus.OCCUPIED:
            finalOccupied = 1;
            break;
          case PropertyStatus.AVAILABLE:
            finalVacant = 1;
            break;
          case PropertyStatus.MAINTENANCE:
            finalMaintenance = 1;
            break;
          case PropertyStatus.UNAVAILABLE:
            finalUnavailable = 1;
            break;
        }
      }

      // Add to totals
      totalUnits += propertyTotalUnits;
      occupiedUnits += finalOccupied;
      vacantUnits += finalVacant;
      maintenanceUnits += finalMaintenance;
      unavailableUnits += finalUnavailable;

      // Calculate occupancy rate for this property
      const propertyOccupancyRate = propertyTotalUnits > 0
        ? Math.round((finalOccupied / propertyTotalUnits) * 100 * 10) / 10
        : 0;

      return {
        propertyName: property.name,
        totalUnits: propertyTotalUnits,
        occupiedUnits: finalOccupied,
        vacantUnits: finalVacant,
        maintenanceUnits: finalMaintenance,
        unavailableUnits: finalUnavailable,
        occupancyRate: propertyOccupancyRate,
      };
    });

    // Calculate overall occupancy rate
    const occupancyRate = totalUnits > 0
      ? Math.round((occupiedUnits / totalUnits) * 100 * 10) / 10
      : 0;

    const occupancyData: OccupancyData = {
      totalUnits,
      occupiedUnits,
      vacantUnits,
      maintenanceUnits,
      unavailableUnits,
      occupancyRate,
      propertyBreakdown,
    };

    return NextResponse.json({
      success: true,
      analytics: occupancyData,
      filters: {
        property,
      },
    });

  } catch (error) {
    return NextResponse.json(
      { error: "Internal server error" },
      { status: 500 }
    );
  }
}

// Future: Add POST method for updating occupancy data
export async function POST(request: NextRequest) {
  try {
    const session = await auth();

    if (!session?.user) {
      return NextResponse.json({ error: "Unauthorized" }, { status: 401 });
    }

    const userRole = session.user.role as UserRole;

    // Only allow ADMIN and SUPER_ADMIN to modify analytics data
    if (![UserRole.ADMIN, UserRole.ADMIN].includes(userRole)) {
      return NextResponse.json({ error: "Forbidden" }, { status: 403 });
    }

    // This would handle updating occupancy data in the future
    return NextResponse.json({
      success: true,
      message: "Occupancy data update endpoint - coming soon",
    });

  } catch (error) {
    return NextResponse.json(
      { error: "Internal server error" },
      { status: 500 }
    );
  }
}
